// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.ConfirmButtonExtender Wrapper
    /// </summary>
    /// <TestComponent Name="ConfirmButton">
    ///   <ToolkitType>AjaxControlToolkit.ConfirmButtonExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.ConfirmButton.ConfirmButton_Common</CommonTestSuite>
    /// </TestComponent>
    public class ConfirmButtonBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// ConfirmText
        /// </summary>
        public BehaviorProperty<string> ConfirmText
        {
            get { return _confirmText; }
        }
        private BehaviorProperty<string> _confirmText;

        public BehaviorProperty<string> ClickScript
        {
            get { return _clickScript; }
        }
        private BehaviorProperty<string> _clickScript;

        /// <summary>
        /// OnClientCancel
        /// </summary>
        public BehaviorProperty<string> OnClientCancel
        {
            get { return _onClientCancel; }
        }
        private BehaviorProperty<string> _onClientCancel;

        /// <summary>
        /// ConfirmOnFormSubmit
        /// </summary>
        public BehaviorProperty<bool> ConfirmOnFormSubmit
        {
            get { return _confirmOnFormSubmit; }
        }
        private BehaviorProperty<bool> _confirmOnFormSubmit;

        /// <summary>
        /// DisplayModalPopupID
        /// </summary>
        public BehaviorProperty<string> DisplayModalPopupID
        {
            get { return _displayModalPopupID; }
        }
        private BehaviorProperty<string> _displayModalPopupID;

        /// <summary>
        /// PostBackScript
        /// </summary>
        public BehaviorProperty<string> PostBackScript
        {
            get { return _postBackScript; }
        }
        private BehaviorProperty<string> _postBackScript;

        /// <summary>
        /// Last prompt message provided
        /// </summary>
        public BehaviorProperty<string> LastPrompt
        {
            get { return _lastPrompt; }
        }
        private BehaviorProperty<string> _lastPrompt;

        /// <summary>
        /// Hidden event
        /// </summary>
        public BehaviorEvent<bool?> Hidden
        {
            get { return _hidden; }
        }
        private BehaviorEvent<bool?> _hidden;

        /// <summary>
        /// AjaxControlToolkit.ConfirmButtonExtender Wrapper
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Wrapper</param>
        public ConfirmButtonBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _confirmText = BehaviorProperty<string>.CreateProperty(this, "ConfirmText");
            _clickScript = BehaviorProperty<string>.CreateField(this, "_oldScript", null, "({0}._oldScript || '').toString()", null);
            _onClientCancel = BehaviorProperty<string>.CreateProperty(this, "OnClientCancel");
            _confirmOnFormSubmit = BehaviorProperty<bool>.CreateProperty(this, "ConfirmOnFormSubmit");
            _displayModalPopupID = BehaviorProperty<string>.CreateProperty(this, "displayModalPopupID");
            _postBackScript = BehaviorProperty<string>.CreateProperty(this, "postBackScript");
            _lastPrompt = BehaviorProperty<string>.CreateField(this, "__lastPrompt", ReadStrategy.Demand, WriteStrategy.NextSynchronization, "window.confirm");
            _hidden = BehaviorEvent<bool?>.CreateEvent(this, "hidden",
                "function(sender, eventArgs) { return eventArgs.get_confirmed(); }");
        }

        /// <summary>
        /// Make the confirmation dialog always return true
        /// </summary>
        public void ForceConfirmation()
        {
            ReplaceConfirmationDialog(true);
        }

        /// <summary>
        /// Make the confirmation dialog always return false
        /// </summary>
        public void ForceIgnore()
        {
            ReplaceConfirmationDialog(false);
        }

        /// <summary>
        /// Replace the confirmation dialog with a constant function
        /// </summary>
        /// <param name="confirm">Dialog result</param>
        private void ReplaceConfirmationDialog(bool confirm)
        {
            bool success = (bool) Page.ExecuteScript("(function () { " +
                    "window.__originalConfirm = window.confirm; " +
                    "window.confirm = function(prompt) { window.confirm.__lastPrompt = prompt; return " + (confirm ? "true" : "false") + "; }; " +
                    "return true; " +
                "})()");
            Assert.IsTrue(success,
                "Failed to replace confirmation dialog with function that always returns {0}!", confirm);
        }

        /// <summary>
        /// Restore the confirmation dialog
        /// </summary>
        public void RestoreConfirmationDialog()
        {
            Page.ExecuteScript("(function () { " +
                    "if (window.__originalConfirm) { window.confirm = window.__originalConfirm; }; " +
                    "return true; " +
                "})()");
        }

        /// <summary>
        /// Simulate the user clicking yes
        /// </summary>
        public void AssertConfirmed()
        {
            _hidden.Register();
            ForceConfirmation();
            Element.Click();
            Assert.IsTrue(_hidden.Wait(3));
            if (string.IsNullOrEmpty(DisplayModalPopupID.Value))
            {
                Assert.AreEqual(_confirmText.Value, _lastPrompt.Value);
            }
        }

        /// <summary>
        /// Simulate the user clicking no
        /// </summary>
        public void AssertIgnored()
        {
            _hidden.Register();
            ForceIgnore();
            Element.Click();
            Assert.IsTrue(_hidden.Wait(3) != true);
            if (string.IsNullOrEmpty(DisplayModalPopupID.Value))
            {
                Assert.AreEqual(_confirmText.Value, _lastPrompt.Value);
            }
        }

        /// <summary>
        /// Prevent clicking the target element from causing a postback
        /// </summary>
        public void PreventPostback()
        {
            string script = "(function() { " +
                "var e = $get('" + Element.Id + "'); " +
                "if (!e) { throw 'Failed to find element \"" + Element.Id + "\"!'; }; " +
                "$addHandler(e, 'click', Function.createDelegate(this, function(eventObj) { " +
                    "if (eventObj && eventObj.preventDefault) { eventObj.preventDefault(); } " +
                    "else if (event) { event.returnValue = false; } " +
                    "return false; " +
                "})); " +
                "return true; " +
            "})()";
            Page.ExecuteScript(script);
        }
    }
}