// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.AccordionExtender Model
    /// </summary>
    /// <TestComponent Name="Accordion">
    ///   <ToolkitType>AjaxControlToolkit.AccordionExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.Accordion.Accordion_Common</CommonTestSuite>
    /// </TestComponent>
    public class AccordionBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// AutoSize
        /// </summary>
        public BehaviorProperty<AutoSize> AutoSize
        {
            get { return _autoSize; }
        }
        private BehaviorProperty<AutoSize> _autoSize;

        /// <summary>
        /// TransitionDuration
        /// </summary>
        public BehaviorProperty<int> TransitionDuration
        {
            get { return _transitionDuration; }
        }
        private BehaviorProperty<int> _transitionDuration;

        /// <summary>
        /// FadeTransitions
        /// </summary>
        public BehaviorProperty<bool> FadeTransitions
        {
            get { return _fadeTransitions; }
        }
        private BehaviorProperty<bool> _fadeTransitions;

        /// <summary>
        /// FramesPerSecond
        /// </summary>
        public BehaviorProperty<int> FramesPerSecond
        {
            get { return _framesPerSecond; }
        }
        private BehaviorProperty<int> _framesPerSecond;

        /// <summary>
        /// SelectedIndex
        /// </summary>
        public BehaviorProperty<int> SelectedIndex
        {
            get { return _selectedIndex; }
        }
        private BehaviorProperty<int> _selectedIndex;

        /// <summary>
        /// RequireOpenedPane
        /// </summary>
        public BehaviorProperty<bool> RequireOpenedPane
        {
            get { return _requireOpenedPane; }
        }
        private BehaviorProperty<bool> _requireOpenedPane;

        /// <summary>
        /// SuppressHeaderPostbacks
        /// </summary>
        public BehaviorProperty<bool> SuppressHeaderPostbacks
        {
            get { return _suppressHeaderPostbacks; }
        }
        private BehaviorProperty<bool> _suppressHeaderPostbacks;

        /// <summary>
        /// HeaderCssClass
        /// </summary>
        public BehaviorProperty<string> HeaderCssClass
        {
            get { return _headerCssClass; }
        }
        private BehaviorProperty<string> _headerCssClass;

        /// <summary>
        /// HeaderSelectedCssClass
        /// </summary>
        public BehaviorProperty<string> HeaderSelectedCssClass
        {
            get { return _headerSelectedCssClass; }
        }
        private BehaviorProperty<string> _headerSelectedCssClass;

        /// <summary>
        /// ContentCssClass
        /// </summary>
        public BehaviorProperty<string> ContentCssClass
        {
            get { return _contentCssClass; }
        }
        private BehaviorProperty<string> _contentCssClass;

        /// <summary>
        /// Count
        /// </summary>
        public BehaviorProperty<int> Count
        {
            get { return _count; }
        }
        private BehaviorProperty<int> _count;

        /// <summary>
        /// ElementHeight
        /// </summary>
        public BehaviorProperty<int> ElementHeight
        {
            get { return _elementHeight; }
        }
        private BehaviorProperty<int> _elementHeight;

        /// <summary>
        /// SelectedIndexChanging event
        /// </summary>
        public BehaviorEvent<Tuple<int, int>> SelectedIndexChanging
        {
            get { return _selectedIndexChanging; }
        }
        private BehaviorEvent<Tuple<int, int>> _selectedIndexChanging;

        /// <summary>
        /// SelectedIndexChanged event
        /// </summary>
        public BehaviorEvent<Tuple<int, int>> SelectedIndexChanged
        {
            get { return _selectedIndexChanged; }
        }
        private BehaviorEvent<Tuple<int, int>> _selectedIndexChanged;

        /// <summary>
        /// AjaxControlToolkit.AccordionExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public AccordionBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _autoSize = BehaviorProperty<AutoSize>.CreateProperty(this, "AutoSize");
            _transitionDuration = BehaviorProperty<int>.CreateProperty(this, "TransitionDuration");
            _fadeTransitions = BehaviorProperty<bool>.CreateProperty(this, "FadeTransitions");
            _framesPerSecond = BehaviorProperty<int>.CreateProperty(this, "FramesPerSecond");
            _selectedIndex = BehaviorProperty<int>.CreateProperty(this, "SelectedIndex", ReadStrategy.Always, WriteStrategy.Immediate);
            _requireOpenedPane = BehaviorProperty<bool>.CreateProperty(this, "requireOpenedPane");
            _suppressHeaderPostbacks = BehaviorProperty<bool>.CreateProperty(this, "suppressHeaderPostbacks");
            _headerCssClass = BehaviorProperty<string>.CreateProperty(this, "HeaderCssClass");
            _headerSelectedCssClass = BehaviorProperty<string>.CreateProperty(this, "HeaderSelectedCssClass");
            _contentCssClass = BehaviorProperty<string>.CreateProperty(this, "ContentCssClass");
            _count = BehaviorProperty<int>.CreateProperty(this, "Count", null, null, "throw 'Count property is read-only!'");
            _elementHeight = BehaviorProperty<int>.CreateField(this, "offsetHeight", "{0}.get_element()", null, "{0}.style.height = '{1}px'");

            string eventArgsExpression = "function(sender, eventArgs) { " +
                "return { " +
                    "'old': eventArgs.get_oldIndex(), " +
                    "'selected': eventArgs.get_selectedIndex() "+
                "}; }";
            _selectedIndexChanging = BehaviorEvent<Tuple<int, int>>.CreateEvent(this, "selectedIndexChanging",
                eventArgsExpression, ConvertSelectedIndexChangeEventArgs);
            _selectedIndexChanged = BehaviorEvent<Tuple<int, int>>.CreateEvent(this, "selectedIndexChanged",
                eventArgsExpression, ConvertSelectedIndexChangeEventArgs);
        }

        /// <summary>
        /// Convert SelectedIndexChangeEventArgs into a Tuple of ints
        /// </summary>
        /// <param name="value">Object deserialized from json</param>
        /// <returns>Event args</returns>
        private static Tuple<int, int> ConvertSelectedIndexChangeEventArgs(object value)
        {
            if (value == null)
            {
                return null;
            }

            int old = Common.GetJson<int>(value, "old");
            int selected = Common.GetJson<int>(value, "selected");
            return new Tuple<int, int>(old, selected);
        }

        /// <summary>
        /// Change the selected index of the Accordion
        /// </summary>
        /// <param name="selectedIndex">Selected index</param>
        public void ChangeSelectedIndex(int selectedIndex)
        {
            int old = _selectedIndex.Value;
            
            // Register handlers that will inform us when the index
            // has been changed (including replacing the Accordion's
            // _endPaneChange function with one that will set a finished
            // flag for us to wait on)
            _selectedIndexChanged.Register();
            Page.ExecuteScript("(function() { " +
                "var b = " + BehaviorReferenceExpression + "; " +
                "if (b._endPaneChange._original) { " +
                    "b._endPaneChange._finished = false; " +
                "} else { " +
                    "var f = function(pane, opening) { " +
                        "b._endPaneChange._original(pane, opening); " +
                        "b._endPaneChange._finished = true; " +
                    "}; " +
                    "f._original = b._endPaneChange; " +
                    "b._endPaneChange = f;" +
                "} " +
            "})();");

            // Change the selected index
            _selectedIndex.Value = selectedIndex;
            
            // Verify the selected index changed event
            Tuple<int, int> args = _selectedIndexChanged.Wait(1, selectedIndex >= 0);
            if (args != null)
            {
                Assert.AreEqual(old, args.First);
                Assert.AreEqual(selectedIndex, args.Second);
            }

            // Wait for the animation to finish playing
            if (Count.Value > 0)
            {
                Page.WaitForScript(BehaviorReferenceExpression + "._endPaneChange._finished", 3);
            }

            // Verify the panes
            AssertSelectedIndex(selectedIndex);
        }

        /// <summary>
        /// Verify only the selected pane is open and the rest are closed
        /// </summary>
        /// <param name="expectedIndex">Index that should be selected</param>
        /// <param name="expectedCssClass">CssClass of the expected index</param>
        public void AssertSelectedIndex(int expectedIndex)
        {
            string result = Page.ExecuteScript("(function() { " +
                "var b = " + BehaviorReferenceExpression + "; " +
                "var expectedIndex = " + expectedIndex + "; " +
                "var css = b.get_HeaderCssClass(); " +
                "var selectedCss = b.get_HeaderSelectedCssClass() || css; " +
                "var length = b.get_Count(); " +
                "for (var i = 0; i < length; i++) { " +
                    "var pane = b.get_Pane(i); " +
                    "if (!pane) { return 'Pane ' + i + ' should not be null!'; } " +
                    "var height = pane.content.offsetHeight; " +
                    "var headerCss = pane.header.className; " + 
                    "if (i == expectedIndex) { " +
                        "if (height == 0) { " +
                            "return 'Height of selected pane ' + i + ' should not be 0!'; " +
                        "} else if (headerCss != selectedCss) { " +
                            "return 'CssClass of selected pane ' + i + ' header should be ' + selectedCss + ', not ' + headerCss + '!'; " +
                        "} " +
                    "} else { " +
                        "if (height) { " +
                            "return 'Height of pane ' + i + ' is ' + height + ', not 0!'; " +
                        "} else if (headerCss != css) { " +
                            "return 'CssClass of pane ' + i + ' header should be ' + css + ', not ' + headerCss + '!'; " +
                        "} " +
                    "} " +
                "} " +
                "return null; " +
            "})()") as string;
            Assert.StringIsNullOrEmpty(result, result);
        }
    }
}