// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;

namespace AjaxControlToolkit.Testing.Client.Accordion
{
    /// <summary>
    /// Common Accordion tests
    /// </summary>
    /// <TestSuite>
    ///   <ComponentType>AjaxControlToolkit.Testing.Client.AccordionBehavior</ComponentType>
    /// </TestSuite>
    [WebTestTag("Accordion")]
    public abstract class Accordion_Common
    {
        /// <summary>
        /// Accordion behaviors on the page along with the number of panes in
        /// each Accordion and a button used to cause a postback or partial
        /// postback
        /// </summary>
        public abstract IEnumerable<AccordionScenario> AccordionScenarios
        {
            get;
        }

        /// <summary>
        /// Get an index to select for a scenario making sure that the index
        /// exists.
        /// </summary>
        /// <param name="scenario">Current testing scenario</param>
        /// <param name="desiredIndex">Desired index</param>
        /// <returns>Valid index to select</returns>
        protected static int GetIndexToSelect(AccordionScenario scenario, int desiredIndex)
        {
            return Math.Min(desiredIndex, scenario.ExpectedCount - 1);
        }

        /// <summary>
        /// Ensure the selected pane is open and has the appropriate styles
        /// </summary>
        [WebTestMethod]
        public void Common_InitialPane()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                scenario.Accordion.AssertSelectedIndex(scenario.Accordion.SelectedIndex.Value);
            }
        }

        /// <summary>
        /// Ensure the Accordion has the correct number of panes (important for
        /// databinding scenarios)
        /// </summary>
        [WebTestMethod]
        public void Common_NumberOfPanes()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                Assert.AreEqual(scenario.ExpectedCount, scenario.Accordion.Count.Value);
            }
        }

        /// <summary>
        /// Change the selected index 
        /// </summary>
        [WebTestMethod]
        public virtual void Common_ChangeSelectedIndex()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                scenario.Accordion.ChangeSelectedIndex(GetIndexToSelect(scenario, 1));
            }
        }

        /// <summary>
        /// Verify that the opacity is being changed when FadeTransitions is set
        /// </summary>
        [WebTestMethod]
        public void Common_FadeTransitions()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                if (!scenario.Accordion.FadeTransitions.Value)
                {
                    // Ignore any accordions that aren't fading
                    continue;
                }

                int selectedIndex = GetIndexToSelect(scenario, 2);
                scenario.Accordion.ChangeSelectedIndex(selectedIndex);
                double opacity = Common.ConvertJsonValue<double>(scenario.Accordion.Page.ExecuteScript("(function() { " +
                    "var b = " + scenario.Accordion.BehaviorReferenceExpression + "; " +
                    "var c = b.get_Pane(" + selectedIndex + ").content; " +
                    "return $common.getElementOpacity((b.get_AutoSize() === AjaxControlToolkit.AutoSize.Fill) ? c._original : c); " +
                "})()"));
                Assert.AreNotEqual(0.0, opacity);
            }
        }

        /// <summary>
        /// Verify that the selected index is preserved on postback
        /// </summary>
        [WebTestMethod]
        public virtual void Common_IndexPreservedOnPostback()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                if (scenario.PostBack == null)
                {
                    // Ignore any accordions that don't have postbacks
                    continue;
                }
                int selectedIndex = GetIndexToSelect(scenario, 2);
                scenario.Accordion.ChangeSelectedIndex(selectedIndex);
                scenario.PostBack.Click(true);
                scenario.Accordion.AssertSelectedIndex(selectedIndex);
            }
        }

        /// <summary>
        /// Verify that AutoSize None works correctly
        /// </summary>
        [WebTestMethod]
        public virtual void Common_AutoSizeNone()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                int height = Math.Min(100, scenario.Accordion.ElementHeight.Value / 2);
                if (height <= 50)
                {
                    // This test isn't relevant if we don't have any panes
                    continue;
                }

                scenario.Accordion.ElementHeight.Value = height;
                scenario.Accordion.AutoSize.Value = AjaxControlToolkit.AutoSize.None;
                scenario.Accordion.ChangeSelectedIndex(GetIndexToSelect(scenario, 1));
                scenario.Accordion.Invalidate();
                Assert.IsTrue(scenario.Accordion.ElementHeight.Value > height,
                    "Accordion height should be greater than {0}, not {1}!",
                    height, scenario.Accordion.ElementHeight.Value);
            }
        }

        /// <summary>
        /// Verify that AutoSize Limit works correctly
        /// </summary>
        [WebTestMethod]
        public void Common_AutoSizeLimit()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                scenario.Accordion.ElementHeight.Value = 299;
                scenario.Accordion.AutoSize.Value = AjaxControlToolkit.AutoSize.Limit;
                scenario.Accordion.ChangeSelectedIndex(GetIndexToSelect(scenario, 1));
                scenario.Accordion.Invalidate();
                Assert.IsTrue(scenario.Accordion.ElementHeight.Value < 300,
                    "Accordion height should be less than 300, not {0}!", scenario.Accordion.ElementHeight.Value);
            }
        }

        /// <summary>
        /// Verify that AutoSize Fill works correctly
        /// </summary>
        [WebTestMethod]
        public void Common_AutoSizeFill()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                scenario.Accordion.ElementHeight.Value = 500;
                scenario.Accordion.AutoSize.Value = AjaxControlToolkit.AutoSize.Fill;
                scenario.Accordion.ChangeSelectedIndex(GetIndexToSelect(scenario, 1));
                scenario.Accordion.Invalidate();
                Assert.AreEqual(500, scenario.Accordion.ElementHeight.Value);
            }
        }

        /// <summary>
        /// Close all the panes
        /// </summary>
        [WebTestMethod]
        public void Common_CloseOpenPane()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                if (!scenario.Accordion.RequireOpenedPane.Value)
                {
                    scenario.Accordion.ChangeSelectedIndex(-1);
                }
            }
        }

        /// <summary>
        /// Select all the panes one after the other
        /// </summary>
        [WebTestMethod]
        [WebTestTag("Stress")]
        public void Common_SelectAllPanes()
        {
            foreach (AccordionScenario scenario in AccordionScenarios)
            {
                Assert.AreEqual(scenario.ExpectedCount, scenario.Accordion.Count.Value);
                int start = (scenario.Accordion.SelectedIndex.Value == 0) ? 1 : 0;
                for (int i = start; i < scenario.ExpectedCount; i++)
                {
                    scenario.Accordion.ChangeSelectedIndex(i);
                }
            }
        }
    }
}