// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Drawing;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit.Testing.Client;
using AjaxControlToolkit.Testing.Client.ResizableControl;

/// <summary>
/// Typical ResizableControl scenarios
/// </summary>
/// <TestSuite>
///   <Path>ResizableControl\Typical.aspx</Path>
///   <ComponentType>AjaxControlToolkit.Testing.Client.ResizableControlBehavior</ComponentType>
///   <WrapperType>AjaxControlToolkit.Testing.Client.ResizableControl.TypicalPage</WrapperType>
/// </TestSuite>
[WebTestClass]
[WebTestTag("ResizableControl.Typical")]
public class ResizableControl_Typical : AjaxControlToolkit.Testing.Client.ResizableControl.ResizableControl_Common
{
    [WebTestMethod]
    public void InitialState()
    {
        TypicalPage page = new TypicalPage();
        AssertPanelAndBehaviorSize(page, new Size(100, 100));
    }

    [WebTestMethod]
    public void ClientResize()
    {
        TypicalPage page = new TypicalPage();
        Size resize = new Size(90, 95);
        page.ResizableControlExtender.SetSize(resize);
        AssertPanelAndBehaviorSize(page, resize);
    }

    [WebTestMethod]
    public void ClientResizeTooFar()
    {
        TypicalPage page = new TypicalPage();
        page.ResizableControlExtender.SetSize(new Size(10, 15));
        AssertPanelAndBehaviorSize(page, new Size(40, 50));
        page.ResizableControlExtender.SetSize(new Size(400, 500));
        AssertPanelAndBehaviorSize(page, new Size(300, 400));
    }

    [WebTestMethod]
    public void PostbackPreservesAndCanChange()
    {
        TypicalPage page = new TypicalPage();

        Size resize = new Size(90, 95);
        page.ResizableControlExtender.SetSize(resize);

        page.Button.Click(true);
        Assert.AreEqual(string.Format("{0}x{1}", resize.Width, resize.Height), page.Label.GetAttributes().InnerText);
        resize.Width += 10;
        resize.Height += 15;
        AssertPanelAndBehaviorSize(page, resize);
    }

    [WebTestMethod]
    public void HandlersCalled()
    {
        TypicalPage page = new TypicalPage();
        page.ResizableControlExtender.ResizeBegin.Register();
        page.ResizableControlExtender.Resizing.Register();
        page.ResizableControlExtender.Resize.Register();

        page.ResizableControlExtender.SetSize(new Size(80, 90));
        page.ResizableControlExtender.ResizeBegin.Wait(3);
        page.ResizableControlExtender.Resizing.Wait(3);
        page.ResizableControlExtender.Resize.Wait(3);
        Assert.IsTrue(Common.ConvertJsonValue<bool>(page.ExecuteScript("resizeFired && resizingFired && resizeBeginFired")));
    }

    /// <summary>
    /// Asserts the Panels and Behavior report the expected size
    /// </summary>
    /// <param name="page">page object</param>
    /// <param name="expectedOffsetSize">expected size</param>
    protected void AssertPanelAndBehaviorSize(TypicalPage page, Size expectedOffsetSize)
    {
        ToolkitAssert.OffsetSize(page.Panel1, expectedOffsetSize);
        ToolkitAssert.OffsetSize(page.Panel2, expectedOffsetSize);
        Size behaviorSize = page.ResizableControlExtender.Size.Value;
        Assert.AreEqual(expectedOffsetSize.Width, behaviorSize.Width);
        Assert.AreEqual(expectedOffsetSize.Height, behaviorSize.Height);
    }
}
