// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Threading;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client.NoBot
{
    using NoBotPageType = Tuple<
        NoBotBehavior,
        HtmlElement,        // LabelValid label
        HtmlElement,        // LabelState label
        HtmlInputElement,   // Postback button
        HtmlInputElement>;  // EmptyCache button

    /// <summary>
    /// Common NoBot tests
    /// </summary>
    /// <TestSuite>
    ///   <ComponentType>AjaxControlToolkit.Testing.Client.NoBotBehavior</ComponentType>
    /// </TestSuite>
    [WebTestTag("NoBot")]
    public abstract class NoBot_Common
    {
        public const int ActualTimeoutSeconds = 3;
        public const int CutoffMaximumInstances = 2;
        protected const int AdequateTimeoutMilliseconds = 3200;
        protected const int InadequateTimeoutMilliseconds = 200;

        public abstract IEnumerable<NoBotPageType> NoBotPages
        {
            get;
        }

        [WebTestMethod]
        public void InitialState()
        {
            foreach (NoBotPageType noBotPage in NoBotPages)
            {
                AssertState(noBotPage.Second, noBotPage.Third, true, NoBotState.Valid);
            }
        }

        [WebTestMethod]
        public void NormalSubmit()
        {
            foreach (NoBotPageType noBotPage in NoBotPages)
            {
                EmptyCache(noBotPage.Fifth);
                DoPostback(noBotPage.Fourth, AdequateTimeoutMilliseconds, noBotPage.Second, noBotPage.Third, true, NoBotState.Valid);
            }
        }

        [WebTestMethod]
        public void PrematureSubmit()
        {
            foreach (NoBotPageType noBotPage in NoBotPages)
            {
                EmptyCache(noBotPage.Fifth);
                DoPostback(noBotPage.Fourth, InadequateTimeoutMilliseconds, noBotPage.Second, noBotPage.Third, false, NoBotState.InvalidResponseTooSoon);
            }
        }

        [WebTestMethod]
        public void GarbledResponse()
        {
            foreach (NoBotPageType noBotPage in NoBotPages)
            {
                EmptyCache(noBotPage.Fifth);
                noBotPage.First.ClientState.Value = "garbled";
                DoPostback(noBotPage.Fourth, AdequateTimeoutMilliseconds, noBotPage.Second, noBotPage.Third, false, NoBotState.InvalidBadResponse);
            }
        }

        [WebTestMethod]
        public void TooMuchActivity()
        {
            foreach (NoBotPageType noBotPage in NoBotPages)
            {
                EmptyCache(noBotPage.Fifth);
                for (int i = 0; i < CutoffMaximumInstances; i++)
                {
                    DoPostback(noBotPage.Fourth, AdequateTimeoutMilliseconds, noBotPage.Second, noBotPage.Third, true, NoBotState.Valid);
                }
                DoPostback(noBotPage.Fourth, AdequateTimeoutMilliseconds, noBotPage.Second, noBotPage.Third, false, NoBotState.InvalidAddressTooActive);
            }
        }

        [WebTestMethod]
        public void EmptyCache()
        {
            foreach (NoBotPageType noBotPage in NoBotPages)
            {
                EmptyCache(noBotPage.Fifth);
                DoPostback(noBotPage.Fourth, AdequateTimeoutMilliseconds, noBotPage.Second, noBotPage.Third, true, NoBotState.Valid);
            }
        }

        protected static void DoPostback(HtmlInputElement postbackButton, int timeoutMilliseconds, HtmlElement labelValid, HtmlElement labelState, bool valid, NoBotState state)
        {
            Thread.Sleep(timeoutMilliseconds);
            postbackButton.Click(true);
            AssertState(labelValid, labelState, valid, state);
        }

        protected static void EmptyCache(HtmlInputElement emptyCacheButton)
        {
            emptyCacheButton.Click(true);
        }

        protected static void AssertState(HtmlElement labelValid, HtmlElement labelState, bool valid, NoBotState state)
        {
            Assert.AreEqual(state.ToString(), labelState.GetAttributes().InnerText);
            Assert.AreEqual(valid.ToString(), labelValid.GetAttributes().InnerText);
        }
    }
}
