// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Drawing;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.DropShadowExtender Model
    /// </summary>
    /// <TestComponent Name="DropShadow">
    ///   <ToolkitType>AjaxControlToolkit.DropShadowExtender</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.DropShadow.DropShadow_Common</CommonTestSuite>
    /// </TestComponent>
    public class DropShadowBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// Opacity
        /// </summary>
        public BehaviorProperty<float> Opacity
        {
            get { return _opacity; }
        }
        private BehaviorProperty<float> _opacity;

        /// <summary>
        /// Width
        /// </summary>
        public BehaviorProperty<int> Width
        {
            get { return _width; }
        }
        private BehaviorProperty<int> _width;

        /// <summary>
        /// TrackPosition
        /// </summary>
        public BehaviorProperty<bool> TrackPosition
        {
            get { return _trackPosition; }
        }
        private BehaviorProperty<bool> _trackPosition;

        /// <summary>
        /// TrackPositionDelay
        /// </summary>
        public BehaviorProperty<int> TrackPositionDelay
        {
            get { return _trackPositionDelay; }
        }
        private BehaviorProperty<int> _trackPositionDelay;

        /// <summary>
        /// Rounded
        /// </summary>
        public BehaviorProperty<bool> Rounded
        {
            get { return _rounded; }
        }
        private BehaviorProperty<bool> _rounded;

        /// <summary>
        /// Radius
        /// </summary>
        public BehaviorProperty<int> Radius
        {
            get { return _radius; }
        }
        private BehaviorProperty<int> _radius;

        /// <summary>
        /// Radius
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Shadow
        {
            get { return _shadow; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _shadow;

        /// <summary>
        /// Bounds of the element
        /// </summary>
        public BehaviorProperty<Rectangle> ElementBounds
        {
            get { return _elementBounds; }
        }
        private BehaviorProperty<Rectangle> _elementBounds;

        /// <summary>
        /// Bounds of the shadow element
        /// </summary>
        public BehaviorProperty<Rectangle> ShadowBounds
        {
            get { return _shadowBounds; }
        }
        private BehaviorProperty<Rectangle> _shadowBounds;

        /// <summary>
        /// Whether the shadow div is defined
        /// </summary>
        public BehaviorProperty<bool> HasShadowElement
        {
            get { return _hasShadowElement; }
        }
        public BehaviorProperty<bool> _hasShadowElement;

                /// <summary>
        /// AjaxControlToolkit.DropShadowExtender Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public DropShadowBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _opacity = BehaviorProperty<float>.CreateProperty(this, "Opacity");
            _width = BehaviorProperty<int>.CreateProperty(this, "Width");
            _trackPosition = BehaviorProperty<bool>.CreateProperty(this, "TrackPosition");
            _trackPositionDelay = BehaviorProperty<int>.CreateProperty(this, "TrackPositionDelay");
            _rounded = BehaviorProperty<bool>.CreateProperty(this, "Rounded");
            _radius = BehaviorProperty<int>.CreateProperty(this, "Radius");
            _shadow = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._shadowDiv");
            _elementBounds = BehaviorProperty<Rectangle>.CreateCustomProperty(this, "{0}._element",
                "(function() {{ var b = $common.getBounds({0}); return {{ 'x': b.x, 'y': b.y, 'width': b.width, 'height': b.height }}; }})()",
                "throw 'ElementBounds property is read-only!'", ConvertToRectangle);
            _shadowBounds = BehaviorProperty<Rectangle>.CreateCustomProperty(this, ReadStrategy.Demand, WriteStrategy.Immediate, "{0}._shadowDiv",
                "(function() {{ var b = $common.getBounds({0}); return {{ 'x': b.x, 'y': b.y, 'width': b.width, 'height': b.height }}; }})()",
                "throw 'ShadowBounds property is read-only!'", ConvertToRectangle);
            _hasShadowElement = BehaviorProperty<bool>.CreateCustomProperty(this, null, "{0}._shadowDiv != null",
                "throw 'HasShadowElement property is read-only!'");
        }

        /// <summary>
        /// Convert a deserialized object into a rectangle
        /// </summary>
        /// <param name="obj">Object</param>
        /// <returns>Rectangle</returns>
        private static Rectangle ConvertToRectangle(object obj)
        {
            int x = Common.GetJson<int>(obj, "x");
            int y = Common.GetJson<int>(obj, "y");
            int width = Common.GetJson<int>(obj, "width");
            int height = Common.GetJson<int>(obj, "height");
            return new Rectangle(x, y, width, height);
        }

        /// <summary>
        /// Ensure the shadow div is positioned correctly
        /// </summary>
        public void AssertBounds()
        {
            _width.Require();
            _hasShadowElement.Require();
            if (_width.Value <= 0 || !_hasShadowElement.Value)
            {
                return;
            }

            _rounded.Require();
            _shadow.Require();
            _elementBounds.Require();
            _shadowBounds.Require();

            Assert.AreEqual(_elementBounds.Value.X + _width.Value, _shadowBounds.Value.X);
// HACK: Why is the drop shadow 2px off in the frame in IE?
            Assert.AreEqual(_elementBounds.Value.Y + _width.Value, _shadowBounds.Value.Y);
            Assert.AreEqual(_elementBounds.Value.Width, _shadowBounds.Value.Width);
            Assert.AreEqual(_elementBounds.Value.Height, _shadowBounds.Value.Height);

            int elementZIndex = Element.GetAttributes().Style.Get<int>("zIndex");
            int shadowZIndex = _shadow.Reference.GetAttributes().Style.Get<int>("zIndex");
            Assert.IsTrue(elementZIndex > shadowZIndex,
                "The z-index <{0}> of the target element should be greater than the z-index <{1}> of the shadow element!",
                elementZIndex, shadowZIndex);
        }

        /// <summary>
        /// Wait for the shadow to be positioned correctly
        /// </summary>
        /// <param name="timeout">Amount of time to wait</param>
        public void WaitForBounds(int timeout)
        {
            Common.Wait(Page, timeout, delegate() { Invalidate(); AssertBounds(); return true; });
        }
    }
}