// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client.TextboxWatermark
{
    /// <summary>
    /// Common TextboxWatermark tests
    /// </summary>
    /// <TestSuite>
    ///   <ComponentType>AjaxControlToolkit.Testing.Client.TextboxWatermarkBehavior</ComponentType>
    /// </TestSuite>
    [WebTestTag("TextboxWatermark")]
    public abstract class TextboxWatermark_Common
    {
        public abstract IEnumerable<TextboxWatermarkScenario> TextboxWatermarkScenarios
        {
            get;
        }

        private string TestText = "Test Text";
        private string TestTextAlternate = ".12345abcde,;!";

        [WebTestMethod]
        public void LoadedProperly()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                VerifyInitialState(textboxWatermarkScenario, null);
            }
        }

        [WebTestMethod]
        [WebTestTag("Smoke")]
        public void FocusAndBlurEmpty()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
                BlurAndCheck(textboxWatermarkScenario.Behavior, null);
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
            }
        }

        [WebTestMethod]
        public void FocusAndBlurFull()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
                textboxWatermarkScenario.Behavior.SetElementText(TestText);
                BlurAndCheck(textboxWatermarkScenario.Behavior, TestText);
                FocusAndCheck(textboxWatermarkScenario.Behavior, TestText);
            }
        }

        [WebTestMethod]
        public void FocusAndBlurMix()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
                textboxWatermarkScenario.Behavior.SetElementText(TestText);
                BlurAndCheck(textboxWatermarkScenario.Behavior, TestText);
                FocusAndCheck(textboxWatermarkScenario.Behavior, TestText);
                textboxWatermarkScenario.Behavior.SetElementText(TestTextAlternate);
                BlurAndCheck(textboxWatermarkScenario.Behavior, TestTextAlternate);
                FocusAndCheck(textboxWatermarkScenario.Behavior, TestTextAlternate);
                textboxWatermarkScenario.Behavior.SetElementText("");
                BlurAndCheck(textboxWatermarkScenario.Behavior, null);
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
                textboxWatermarkScenario.Behavior.SetElementText(TestText);
                BlurAndCheck(textboxWatermarkScenario.Behavior, TestText);
                FocusAndCheck(textboxWatermarkScenario.Behavior, TestText);
                textboxWatermarkScenario.Behavior.SetElementText("");
                BlurAndCheck(textboxWatermarkScenario.Behavior, null);
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
                textboxWatermarkScenario.Behavior.SetElementText(TestText);
                textboxWatermarkScenario.Behavior.SetElementText(TestTextAlternate);
                textboxWatermarkScenario.Behavior.SetElementText("");
                BlurAndCheck(textboxWatermarkScenario.Behavior, null);
            }
        }

        [WebTestMethod]
        public void SubmitEmpty()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                textboxWatermarkScenario.DoPostback();
                VerifyInitialState(textboxWatermarkScenario, null);
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
                BlurAndCheck(textboxWatermarkScenario.Behavior, null);
                VerifyInitialState(textboxWatermarkScenario, null);
            }
        }

        [WebTestMethod]
        [WebTestTag("Smoke")]
        public void SubmitFull()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                FocusAndCheck(textboxWatermarkScenario.Behavior, null);
                textboxWatermarkScenario.Behavior.SetElementText(TestText);
                textboxWatermarkScenario.DoPostback();
                VerifyInitialState(textboxWatermarkScenario, TestText);
                FocusAndCheck(textboxWatermarkScenario.Behavior, TestText);
                BlurAndCheck(textboxWatermarkScenario.Behavior, TestText);
                textboxWatermarkScenario.DoPostback();
                VerifyInitialState(textboxWatermarkScenario, TestText);
                FocusAndCheck(textboxWatermarkScenario.Behavior, TestText);
                textboxWatermarkScenario.Behavior.SetElementText("");
                BlurAndCheck(textboxWatermarkScenario.Behavior, null);
            }
        }

        [WebTestMethod]
        public void TextProperty()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                textboxWatermarkScenario.Behavior.AssertWatermarked();
                textboxWatermarkScenario.Behavior.SetText(TestText);
                textboxWatermarkScenario.Behavior.AssertFull(TestText);
                textboxWatermarkScenario.Behavior.SetText(TestTextAlternate);
                textboxWatermarkScenario.Behavior.AssertFull(TestTextAlternate);
                textboxWatermarkScenario.Behavior.SetText("");
                textboxWatermarkScenario.Behavior.AssertWatermarked();
            }
        }

        [WebTestMethod]
        public void WatermarkProperties()
        {
            foreach (TextboxWatermarkScenario textboxWatermarkScenario in TextboxWatermarkScenarios)
            {
                string watermarkText = "alternate watermark text";
                textboxWatermarkScenario.Behavior.SetWatermarkText(watermarkText);
                textboxWatermarkScenario.Behavior.AssertWatermarked();
                string watermarkCssClass = "alternatewatermark";
                textboxWatermarkScenario.Behavior.SetWatermarkCssClass(watermarkCssClass);
                textboxWatermarkScenario.Behavior.AssertWatermarked();
            }
        }

        // TODO: Test MaxLength support

        private void VerifyInitialState(TextboxWatermarkScenario textboxWatermarkScenario, string text)
        {
            if (string.IsNullOrEmpty(text))
            {
                textboxWatermarkScenario.Behavior.AssertWatermarked();
            }
            else
            {
                textboxWatermarkScenario.Behavior.AssertFull(text);
            }
            Assert.AreEqual(string.Format(textboxWatermarkScenario.TextFormat, text ?? textboxWatermarkScenario.EmptyText), textboxWatermarkScenario.Label.GetAttributes().InnerText);
        }

        private void FocusAndCheck(TextboxWatermarkBehavior behavior, string text)
        {
            if (string.IsNullOrEmpty(text))
            {
                behavior.AssertWatermarked();
                behavior.Focus();
                behavior.AssertEmpty();
            }
            else
            {
                behavior.AssertFull(text);
                behavior.Focus();
                behavior.AssertFull(text);
            }
        }

        private void BlurAndCheck(TextboxWatermarkBehavior behavior, string text)
        {
            if (string.IsNullOrEmpty(text))
            {
                behavior.AssertEmpty();
                behavior.Blur();
                behavior.AssertWatermarked();
            }
            else
            {
                behavior.AssertFull(text);
                behavior.Blur();
                behavior.AssertFull(text);
            }
        }
    }
}
