// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Web.UI.WebControls;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.TabContainer Model
    /// </summary>
    /// <TestComponent Name="Tabs">
    ///   <ToolkitType>AjaxControlToolkit.TabContainer</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.Tabs.Tabs_Common</CommonTestSuite>
    /// </TestComponent>
    public class TabContainerBehavior : Behavior<HtmlElement>
    {
        /// <summary>
        /// ActiveTabIndex
        /// </summary>
        public BehaviorProperty<int> ActiveTabIndex
        {
            get { return _activeTabIndex; }
        }
        private BehaviorProperty<int> _activeTabIndex;

        /// <summary>
        /// ScrollBars
        /// </summary>
        public BehaviorProperty<ScrollBars> ScrollBars
        {
            get { return _scrollBars; }
        }
        private BehaviorProperty<ScrollBars> _scrollBars;

        /// <summary>
        /// UniqueID
        /// </summary>
        public BehaviorProperty<string> UniqueID
        {
            get { return _uniqueID; }
        }
        private BehaviorProperty<string> _uniqueID;

        /// <summary>
        /// CssClass
        /// </summary>
        public BehaviorProperty<string> CssClass
        {
            get { return _cssClass; }
        }
        private BehaviorProperty<string> _cssClass;

        /// <summary>
        /// Height
        /// </summary>
        public BehaviorProperty<int> Height
        {
            get { return _height; }
        }
        private BehaviorProperty<int> _height;

        /// <summary>
        /// Header
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Header
        {
            get { return _header; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _header;

        /// <summary>
        /// Body
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> Body
        {
            get { return _body; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _body;

        /// <summary>
        /// Tabs
        /// </summary>
        public BehaviorProperty<IList<TabPanelBehavior>> TabPanels
        {
            get { return _tabPanels; }
        }
        private BehaviorProperty<IList<TabPanelBehavior>> _tabPanels;

        /// <summary>
        /// ActiveTabChanged
        /// </summary>
        private BehaviorEvent ActiveTabChanged
        {
            get { return _activeTabChanged; }
        }
        private BehaviorEvent _activeTabChanged;

        /// <summary>
        /// ActiveTab
        /// </summary>
        public TabPanelBehavior ActiveTab
        {
            get
            {
                _tabPanels.Require();
                _activeTabIndex.Require();
                return _tabPanels.Value[_activeTabIndex.Value];
            }
            set
            {
                string tab = (value == null) ? "null" : value.BehaviorReferenceExpression;
                Page.ExecuteScript(BehaviorReferenceExpression + ".set_activeTab(" + tab + "), true");
            }
        }

        /// <summary>
        /// AjaxControlToolkit.TabContainer Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public TabContainerBehavior(HtmlElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _activeTabIndex = BehaviorProperty<int>.CreateProperty(this, "activeTabIndex");
            _scrollBars = BehaviorProperty<ScrollBars>.CreateProperty(this, "scrollBars");
            _uniqueID = BehaviorProperty<string>.CreateProperty(this, "autoPostBackId");
            _cssClass = BehaviorProperty<string>.CreateField(this, "className", "{0}._element");
            _height = BehaviorProperty<int>.CreateField(this, "offsetHeight", "{0}._element", null, "{0}.style.height = \'{1}px\'");
            _header = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._header");
            _body = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._body");
            _tabPanels = BehaviorProperty<IList<TabPanelBehavior>>.CreateCustomProperty(this, null, "(function() {{ " +
                    "var tabs = Array.clone({0}.get_tabs()); " +
                    "if (!tabs || tabs.length == 0) {{ return null; }} " +
                    "for (var i = 0; i < tabs.length; i++) {{ tabs[i] = tabs[i]._element.id; }} " +
                    "return tabs.join(' '); " +
                "}})()",
                "throw 'Tabs property is read-only!'", ConvertTabPanels);
            _activeTabChanged = BehaviorEvent<object>.CreateEvent(this, "activeTabChanged");
        }

        /// <summary>
        /// Convert the value into a list of TabPanelBehaviors
        /// </summary>
        private IList<TabPanelBehavior> ConvertTabPanels(object value)
        {
            List<TabPanelBehavior> panels = new List<TabPanelBehavior>();
            string identifiers = value as string;
            if (!string.IsNullOrEmpty(identifiers))
            {
                foreach (string id in identifiers.Split(' '))
                {
                    HtmlElement element = Page.Elements.Find(id);
                    Assert.IsNotNull(element, "Failed to find TabPanel \"{0}\" in TabContainer \"{1}\"!", id, BehaviorID);
                    panels.Add(new TabPanelBehavior(element, id, Page));
                }
            }
            return panels;
        }
    }
}