// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Text;
using System.Web.Script.Serialization;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// The SynchronizationManager is used to read and write properties
    /// of behaviors on the ToolkitTestPage in an aggregate fashion.
    /// </summary>
    public class SynchronizationManager
    {
        // Format strings for variable names used durinc synchronization
        private const string BehaviorVariableFormat = "b{0}";
        private const string CustomVariableFormat = "c{0}";
        private const string ValueVariableFormat = "v{0}";

        /// <summary>
        /// List of properties to read on the next synchronization
        /// </summary>
        public IList<BehaviorProperty> Pending
        {
            get { return _pending; }
        }
        private List<BehaviorProperty> _pending;

        /// <summary>
        /// List of properties to read every synchronization
        /// </summary>
        public IList<BehaviorProperty> AlwaysRead
        {
            get { return _alwaysRead; }
        }
        private List<BehaviorProperty> _alwaysRead;

        /// <summary>
        /// List of properties (and their values) to update on the next
        /// synchronization
        /// </summary>
        public IDictionary<BehaviorProperty, object> Updates
        {
            get { return _updates; }
        }
        private Dictionary<BehaviorProperty, object> _updates;
        
        /// <summary>
        /// Page containing the behaviors to synchronize
        /// </summary>
        public ToolkitTestPage Page
        {
            get { return _page; }
        }
        private ToolkitTestPage _page;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="page">Page containing the behaviors to synchronize</param>
        public SynchronizationManager(ToolkitTestPage page)
        {
            Assert.IsNotNull(page, "page cannot be null!");
            _page = page;

            _pending = new List<BehaviorProperty>();
            _alwaysRead = new List<BehaviorProperty>();
            _updates = new Dictionary<BehaviorProperty, object>();
        }

        /// <summary>
        /// Synchronize the behaviors on the page with their client-side representations
        /// </summary>
        public void Synchronize()
        {
            // Compute list of things we're going to touch
            List<BehaviorProperty> properties = CombineQueues();
            if (properties.Count == 0)
            {
                return;
            }

            // Dynamically generate the script to sync the properties
            StringBuilder script = new StringBuilder();
            script.Append("(function() { ");
            JavaScriptSerializer serializer = new JavaScriptSerializer();

            // The following dictionaries are used to ensure we only lookup
            // each object once.  The expressionMapping table maps from lookup
            // expressions to variable names, the behaviorMapping table maps
            // from Behavior references to variable names, and the propertyMapping
            // table maps from BehaviorProperty references to variable names.
            Dictionary<string, string> expressionMapping = new Dictionary<string, string>();
            Dictionary<Behavior, string> behaviorMapping = new Dictionary<Behavior, string>();
            Dictionary<BehaviorProperty, string> propertyMapping = new Dictionary<BehaviorProperty, string>();
            
            // Compute the variables required for all of the lookup expressions
            ComputeLookupVariables(properties, expressionMapping, behaviorMapping, propertyMapping);
            
            // Create the lookups using the expression to variable name mapping
            WriteLookups(script, expressionMapping, serializer);

            // Write any updates that have been provided
            WriteUpdates(script, propertyMapping, serializer);

            // Get any values that need to be retrieved
            script.Append("return { ");
            WriteReads(script, properties, propertyMapping);
            script.Append(" }; })()");

            // Execute the script
            Dictionary<string, object> results = _page.ExecuteScript(script.ToString()) as Dictionary<string, object>;

            // Pull the values of the properties out of the result
            SaveResults(properties, results);

            // Wipe the synchronization queues
            CleanQueues();
        }

        /// <summary>
        /// Get the list of (unique) properties to be read
        /// </summary>
        /// <returns>List of unique properties</returns>
        private List<BehaviorProperty> CombineQueues()
        {
            List<BehaviorProperty> properties = new List<BehaviorProperty>();
            foreach (BehaviorProperty property in _alwaysRead)
            {
                if (!properties.Contains(property))
                {
                    properties.Add(property);
                }
            }
            foreach (BehaviorProperty property in _pending)
            {
                if (!properties.Contains(property))
                {
                    properties.Add(property);
                }
            }
            return properties;
        }

        /// <summary>
        /// Compute the variables required for all of the lookup expressions
        /// (merging common expressions to the same variable)
        /// </summary>
        /// <param name="properties">List of properties</param>
        /// <param name="expressionMapping">Map expressions to variable names</param>
        /// <param name="behaviorMapping">Map Behavior references to variable names</param>
        /// <param name="propertyMapping">Map BehaviorProperty references to variable names</param>
        private void ComputeLookupVariables(List<BehaviorProperty> properties, Dictionary<string, string> expressionMapping, Dictionary<Behavior, string> behaviorMapping, Dictionary<BehaviorProperty, string> propertyMapping)
        {
            int variableCount = 0;
            
            // Compute the standard behavior lookup expression variables
            // (first, as they'll be used in the custom variables)
            foreach (BehaviorProperty property in properties)
            {
                if (string.IsNullOrEmpty(property.LookupExpression) || property.LookupExpression.Contains("{0}"))
                {
                    string expression = property.Behavior.BehaviorReferenceExpression;
                    string variable;
                    if (!expressionMapping.TryGetValue(expression, out variable))
                    {
                        variable = string.Format(BehaviorVariableFormat, variableCount++);
                    }
                    expressionMapping[expression] = variable;
                    propertyMapping[property] = variable;
                    behaviorMapping[property.Behavior] = variable;
                }
            }

            // Compute the custom lookup expression variables
            foreach (BehaviorProperty property in properties)
            {
                if (string.IsNullOrEmpty(property.LookupExpression))
                {
                    continue;
                }

                // Get the behavior variable so it can be used as part of the expression
                string behaviorVariable = null;
                behaviorMapping.TryGetValue(property.Behavior, out behaviorVariable);

                // Evaluate the expression
                string expression = string.Format(property.LookupExpression, behaviorVariable, property.Behavior.BehaviorID, property.Behavior.BehaviorReferenceExpression);

                // Merge shared expressions into the same variable
                string variable;
                if (!expressionMapping.TryGetValue(expression, out variable))
                {
                    variable = string.Format(CustomVariableFormat, variableCount++);
                }
                expressionMapping[expression] = variable;
                propertyMapping[property] = variable;
            }
        }

        /// <summary>
        /// Create the lookups using the expression to variable name mapping
        /// </summary>
        /// <param name="script">Script being created</param>
        /// <param name="expressionMapping">Map lookup expressions to variable names</param>
        /// <param name="serializer">Serializer</param>
        private void WriteLookups(StringBuilder script, Dictionary<string, string> expressionMapping, JavaScriptSerializer serializer)
        {
            foreach (KeyValuePair<string, string> pair in expressionMapping)
            {
                string variable = pair.Value;
                string expression = pair.Key;
                string escapedExpression = serializer.Serialize(expression);
                script.AppendFormat("var {0} = {1}; ", variable, expression);
                script.AppendFormat("if (!{0}) {{ throw 'Expression ' + {1} + ' evaluated to null!'; }}; ", variable, escapedExpression);
            }
        }

        /// <summary>
        /// Write any updates that have been provided
        /// </summary>
        /// <param name="script">Script being created</param>
        /// <param name="expressionMapping">Map property refences to variable names</param>
        /// <param name="serializer">Serializer</param>
        private void WriteUpdates(StringBuilder script, Dictionary<BehaviorProperty, string> propertyMapping, JavaScriptSerializer serializer)
        {
            foreach (KeyValuePair<BehaviorProperty, object> pair in _updates)
            {
                string variable = null;
                BehaviorProperty property = pair.Key;
                propertyMapping.TryGetValue(property, out variable);
                Assert.StringIsNotNullOrEmpty(variable,
                    "Failed to find variable corresponding to property {0}.{1}!", property.Behavior.BehaviorID, property.Name);
                string value = serializer.Serialize(pair.Value);

                // Write out the standard set expressions
                if (property.MemberType != ClientMemberType.Custom && string.IsNullOrEmpty(property.SetExpression))
                {
                    switch (property.MemberType)
                    {
                        case ClientMemberType.Property:
                            script.AppendFormat("{0}.set_{1}({2}); ", variable, property.Name, value);
                            break;
                        case ClientMemberType.Field:
                            script.AppendFormat("{0}.{1} = {2}; ", variable, property.Name, value);
                            break;
                    }
                }
                // Write out custom set expressions
                else
                {
                    Assert.StringIsNotNullOrEmpty(property.SetExpression,
                        "Custom property of behavior \"{0}\" does not have a SetExpression defined", property.Behavior.BehaviorID);
                    script.AppendFormat("{0}; ", string.Format(property.SetExpression, variable, value));
                }
            }
        }

        /// <summary>
        /// Read any values that need to be updated
        /// </summary>
        /// <param name="script">Script being created</param>
        /// <param name="properties">Properties that need to be updated</param>
        /// <param name="expressionMapping">Map property refences to variable names</param>
        private void WriteReads(StringBuilder script, List<BehaviorProperty> properties, Dictionary<BehaviorProperty, string> propertyMapping)
        {
            for (int i = 0; i < properties.Count; i++)
            {
                BehaviorProperty property = properties[i];
                if (i != 0)
                {
                    script.Append(", ");
                }

                script.Append("'");
                script.Append(string.Format(ValueVariableFormat, i));
                script.Append("': ");

                string variable = null;
                propertyMapping.TryGetValue(property, out variable);
                Assert.StringIsNotNullOrEmpty(variable,
                    "Failed to find variable corresponding to property {0}.{1}!", property.Behavior.BehaviorID, property.Name);

                // Read properties and fields
                if (property.MemberType != ClientMemberType.Custom && string.IsNullOrEmpty(property.GetExpression))
                {
                    switch (property.MemberType)
                    {
                        case ClientMemberType.Property:
                            script.AppendFormat("{0}.get_{1}()", variable, property.Name);
                            break;
                        case ClientMemberType.Field:
                            script.AppendFormat("{0}.{1}", variable, property.Name);
                            break;
                    }
                }
                // Read custom variables
                else
                {
                    Assert.StringIsNotNullOrEmpty(property.GetExpression,
                        "GetExpression for custom property {0}.{1} cannot be null!", property.Behavior.BehaviorID, property.Name);
                    script.AppendFormat(property.GetExpression, variable);
                }
            }
        }

        /// <summary>
        /// Pull the values of the properties out of the results object
        /// </summary>
        private void SaveResults(List<BehaviorProperty> properties, Dictionary<string, object> results)
        {
            Assert.IsNotNull(results, "Failed to retrieve results during synchronization!");

            for (int i = 0; i < properties.Count; i++)
            {
                string variableName = string.Format(ValueVariableFormat, i);
                object value = null;
                results.TryGetValue(variableName, out value);
                properties[i].UpdateValue(value);
            }
        }

        /// <summary>
        /// Wipe the synchronization queues
        /// </summary>
        private void CleanQueues()
        {
            _pending.Clear();
            _updates.Clear();
        }
    }
}