// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// AjaxControlToolkit.PasswordStrength Model
    /// </summary>
    /// <TestComponent Name="PasswordStrength">
    ///   <ToolkitType>AjaxControlToolkit.PasswordStrength</ToolkitType>
    ///   <CommonTestSuite>AjaxControlToolkit.Testing.Client.PasswordStrength.PasswordStrength_Common</CommonTestSuite>
    /// </TestComponent>
    public class PasswordStrengthBehavior : Behavior<HtmlInputElement>
    {
        /// <summary>
        /// PreferredPasswordLength
        /// </summary>
        public BehaviorProperty<int> PreferredPasswordLength
        {
            get { return _preferredPasswordLength; }
        }
        private BehaviorProperty<int> _preferredPasswordLength;

        /// <summary>
        /// MinimumNumericCharacters
        /// </summary>
        public BehaviorProperty<int> MinimumNumericCharacters
        {
            get { return _minimumNumericCharacters; }
        }
        private BehaviorProperty<int> _minimumNumericCharacters;

        /// <summary>
        /// HelpHandleCssClass
        /// </summary>
        public BehaviorProperty<string> HelpHandleCssClass
        {
            get { return _helpHandleCssClass; }
        }
        private BehaviorProperty<string> _helpHandleCssClass;

        /// <summary>
        /// HelpHandlePosition
        /// </summary>
        public BehaviorProperty<DisplayPosition> HelpHandlePosition
        {
            get { return _helpHandlePosition; }
        }
        private BehaviorProperty<DisplayPosition> _helpHandlePosition;

        /// <summary>
        /// HelpStatusLabelID
        /// </summary>
        public ReferenceBehaviorProperty<HtmlElement> HelpStatusLabel
        {
            get { return _helpStatusLabel; }
        }
        private ReferenceBehaviorProperty<HtmlElement> _helpStatusLabel;

        /// <summary>
        /// MinimumSymbolCharacters
        /// </summary>
        public BehaviorProperty<int> MinimumSymbolCharacters
        {
            get { return _minimumSymbolCharacters; }
        }
        private BehaviorProperty<int> _minimumSymbolCharacters;

        /// <summary>
        /// RequiresUpperAndLowerCaseCharacters
        /// </summary>
        public BehaviorProperty<bool> RequiresUpperAndLowerCaseCharacters
        {
            get { return _requiresUpperAndLowerCaseCharacters; }
        }
        private BehaviorProperty<bool> _requiresUpperAndLowerCaseCharacters;

        /// <summary>
        /// TextCssClass
        /// </summary>
        public BehaviorProperty<string> TextCssClass
        {
            get { return _textCssClass; }
        }
        private BehaviorProperty<string> _textCssClass;

        /// <summary>
        /// BarBorderCssClass
        /// </summary>
        public BehaviorProperty<string> BarBorderCssClass
        {
            get { return _barBorderCssClass; }
        }
        private BehaviorProperty<string> _barBorderCssClass;

        /// <summary>
        /// BarIndicatorCssClass
        /// </summary>
        public BehaviorProperty<string> BarIndicatorCssClass
        {
            get { return _barIndicatorCssClass; }
        }
        private BehaviorProperty<string> _barIndicatorCssClass;

        /// <summary>
        /// PrefixText
        /// </summary>
        public BehaviorProperty<string> PrefixText
        {
            get { return _prefixText; }
        }
        private BehaviorProperty<string> _prefixText;

        /// <summary>
        /// DisplayPosition
        /// </summary>
        public BehaviorProperty<DisplayPosition> DisplayPosition
        {
            get { return _displayPosition; }
        }
        private BehaviorProperty<DisplayPosition> _displayPosition;

        /// <summary>
        /// StrengthIndicatorType
        /// </summary>
        public BehaviorProperty<StrengthIndicatorTypes> StrengthIndicatorType
        {
            get { return _strengthIndicatorType; }
        }
        private BehaviorProperty<StrengthIndicatorTypes> _strengthIndicatorType;

        /// <summary>
        /// CalculationWeightings
        /// </summary>
        public BehaviorProperty<string> CalculationWeightings
        {
            get { return _calculationWeightings; }
        }
        private BehaviorProperty<string> _calculationWeightings;

        /// <summary>
        /// TextStrengthDescriptions
        /// </summary>
        public BehaviorProperty<string> TextStrengthDescriptions
        {
            get { return _textStrengthDescriptions; }
        }
        private BehaviorProperty<string> _textStrengthDescriptions;

        /// <summary>
        /// StrengthStyles
        /// </summary>
        public BehaviorProperty<string> StrengthStyles
        {
            get { return _strengthStyles; }
        }
        private BehaviorProperty<string> _strengthStyles;

        /// <summary>
        /// MinimumLowerCaseCharacters
        /// </summary>
        public BehaviorProperty<int> MinimumLowerCaseCharacters
        {
            get { return _minimumLowerCaseCharacters; }
        }
        private BehaviorProperty<int> _minimumLowerCaseCharacters;

        /// <summary>
        /// MinimumUpperCaseCharacters
        /// </summary>
        public BehaviorProperty<int> MinimumUpperCaseCharacters
        {
            get { return _minimumUpperCaseCharacters; }
        }
        private BehaviorProperty<int> _minimumUpperCaseCharacters;

        /// <summary>
        /// TextStrengthDescriptionStyles
        /// </summary>
        public BehaviorProperty<string> TextStrengthDescriptionStyles
        {
            get { return _textStrengthDescriptionStyles; }
        }
        private BehaviorProperty<string> _textStrengthDescriptionStyles;

        public ReferenceBehaviorProperty<HtmlElement> OuterBarIndicator
        {
            get 
            {
                if ((this._outerBarIndicator == null) && (this.StrengthIndicatorType.Value == StrengthIndicatorTypes.BarIndicator))
                {
                    _outerBarIndicator = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._barOuterDiv");
                }
                return this._outerBarIndicator; 
            }
        }

        private ReferenceBehaviorProperty<HtmlElement> _outerBarIndicator;

        public ReferenceBehaviorProperty<HtmlElement> InnerBarIndicator
        {
            get 
            {
                if ((this._innerBarIndicator == null) && (this.StrengthIndicatorType.Value == StrengthIndicatorTypes.BarIndicator))
                {
                    _innerBarIndicator = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._barInnerDiv");
                }
                return this._innerBarIndicator;
            }
        }
        private ReferenceBehaviorProperty<HtmlElement> _innerBarIndicator;

        public ReferenceBehaviorProperty<HtmlElement> TextIndicator
        {
            get 
            {
                if ((this._textIndicator == null) && (this.StrengthIndicatorType.Value == StrengthIndicatorTypes.Text))
                {
                    _textIndicator = ReferenceBehaviorProperty<HtmlElement>.CreateField(this, "id", "{0}._displayDiv");
                }
                return this._textIndicator; 
            }
        }
        private ReferenceBehaviorProperty<HtmlElement> _textIndicator;

        /// <summary>
        /// AjaxControlToolkit.PasswordStrength Model
        /// </summary>
        /// <param name="element">Target element</param>
        /// <param name="behaviorID">Behavior ID</param>
        /// <param name="page">Page Model</param>
        public PasswordStrengthBehavior(HtmlInputElement element, string behaviorID, ToolkitTestPage page)
            : base(element, behaviorID, page)
        {
            _preferredPasswordLength = BehaviorProperty<int>.CreateProperty(this, "PreferredPasswordLength");
            _minimumNumericCharacters = BehaviorProperty<int>.CreateProperty(this, "MinimumNumericCharacters");
            _helpHandleCssClass = BehaviorProperty<string>.CreateProperty(this, "HelpHandleCssClass");
            _helpHandlePosition = BehaviorProperty<DisplayPosition>.CreateProperty(this, "HelpHandlePosition");
            _helpStatusLabel = ReferenceBehaviorProperty<HtmlElement>.CreateProperty(this, "HelpStatusLabelID");
            _minimumSymbolCharacters = BehaviorProperty<int>.CreateProperty(this, "MinimumSymbolCharacters");
            _requiresUpperAndLowerCaseCharacters = BehaviorProperty<bool>.CreateProperty(this, "RequiresUpperAndLowerCaseCharacters");
            _textCssClass = BehaviorProperty<string>.CreateProperty(this, "TextCssClass");
            _barBorderCssClass = BehaviorProperty<string>.CreateProperty(this, "BarBorderCssClass");
            _barIndicatorCssClass = BehaviorProperty<string>.CreateProperty(this, "BarIndicatorCssClass");
            _prefixText = BehaviorProperty<string>.CreateProperty(this, "PrefixText");
            _displayPosition = BehaviorProperty<DisplayPosition>.CreateProperty(this, "DisplayPosition");
            _strengthIndicatorType = BehaviorProperty<StrengthIndicatorTypes>.CreateProperty(this, "StrengthIndicatorType");
            _calculationWeightings = BehaviorProperty<string>.CreateProperty(this, "CalculationWeightings");
            _textStrengthDescriptions = BehaviorProperty<string>.CreateProperty(this, "TextStrengthDescriptions");
            _strengthStyles = BehaviorProperty<string>.CreateProperty(this, "StrengthStyles");
            _minimumLowerCaseCharacters = BehaviorProperty<int>.CreateProperty(this, "MinimumLowerCaseCharacters");
            _minimumUpperCaseCharacters = BehaviorProperty<int>.CreateProperty(this, "MinimumUpperCaseCharacters");
            _textStrengthDescriptionStyles = BehaviorProperty<string>.CreateProperty(this, "TextStrengthDescriptionStyles");
        }


        /// <summary>
        /// Ensure that the indicator is hidden/shown.
        /// </summary>
        public void AssertIndicatorVisibility(Visibility visibility)
        {
            if (this.StrengthIndicatorType.Value == StrengthIndicatorTypes.BarIndicator)
            {
                Assert.AreEqual(visibility, this.OuterBarIndicator.Reference.GetAttributes().Style.Visibility);
                Assert.AreEqual(visibility, this.InnerBarIndicator.Reference.GetAttributes().Style.Visibility);
            }
            else
            {
                Assert.AreEqual(visibility, this.TextIndicator.Reference.GetAttributes().Style.Visibility);
            }
        }

        /// <summary>
        /// Set the password by firing key events and ensure that the indicator is visible
        /// </summary>
        /// <param name="passwordText"></param>
        public void SetPassword(string passwordText)
        {
            this.Element.DispatchEvent(HtmlEventName.Focus);
            Common.TypeText(this.Element, passwordText, true, false);
        }

        /// <summary>
        /// Wipe out the contents of the textbox
        /// </summary>
        public void ClearPassword()
        {
            this.Element.SetText("");
        }
        /// <summary>
        /// Returns the text message in the indicator that lets the user know the strength of the password.
        /// </summary>
        public string TextIndicatorValue
        {
            get
            {
                return this.TextIndicator.Reference.GetAttributes().InnerText;
            }
        }
    }
}