// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Web.UI;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// The BehaviorProperty models an individual field or property of a
    /// client-side behavior.  It is primarily used to support caching the
    /// value of the property with the SynchronizationManager.
    /// </summary>
    public class BehaviorProperty
    {
        /// <summary>
        /// Name of the client-side field or property being modeled
        /// </summary>
        public string Name
        {
            get { return _name; }
        }
        private string _name;

        /// <summary>
        /// Whether the client-side member is a property of field of
        /// the behavior or a custom object
        /// </summary>
        public ClientMemberType MemberType
        {
            get { return _memberType; }
        }
        private ClientMemberType _memberType;

        /// <summary>
        /// Cached value of the client-side field or property
        /// </summary>
        /// <remarks>
        /// Reading the value of a property that has not been cached, had
        /// its cache invalidated, or has a volatile ReadStrategy will
        /// cause a synchronization with the client.  Setting the value
        /// of the property will cause a synchronization if it has an
        /// immediate WriteStrategy.
        /// </remarks>
        public object Value
        {
            get
            {
                if (!_synchronized || IsVolatile)
                {
                    GetValue();
                }
                return _value;
            }
            set
            {
                SetValue(value);
            }
        }
        private object _value;

        /// <summary>
        /// Whether or not the cached value of the property is currently
        /// synchronized (calling Invalidate on the BehaviorProperty, Behavior,
        /// or ToolkitTestPage will undo the synchronization).
        /// </summary>
        public bool Synchronized
        {
            get { return _synchronized; }
        }
        private bool _synchronized;

        /// <summary>
        /// How synchronization of the property should occur when the
        /// value is requested
        /// </summary>
        public ReadStrategy ReadStrategy
        {
            get { return _readStrategy; }
        }
        private ReadStrategy _readStrategy;

        /// <summary>
        /// Whether or not the property has a volatile read strategy and
        /// getting it will force a synchronization
        /// </summary>
        protected bool IsVolatile
        {
            get { return (_readStrategy & ReadStrategy.Volatile) == ReadStrategy.Volatile; }
        }

        /// <summary>
        /// How synchronization of the property should occur when the
        /// value is updated
        /// </summary>
        public WriteStrategy WriteStrategy
        {
            get { return _writeStrategy; }
        }
        private WriteStrategy _writeStrategy;

        /// <summary>
        /// Custom format string used to look up the object containing
        /// the property being modeled.  If the custom expression is null,
        /// a reference to the behavior will be automatically supplied.
        /// The specifier {0} will be replaced by a reference to the behavior
        /// and {1} will be replaced by the behavior's ID.
        /// </summary>
        public string LookupExpression
        {
            get { return _lookupExpression; }
        }
        private string _lookupExpression;

        /// <summary>
        /// Custom format string used to get the value of the property
        /// from an object.  The specifier {0} will be replaced by a
        /// reference to the behavior.  If the custom expression is null and
        /// the ClientMemberType is Field or Property, an appropriate
        /// expression will be used to read the value from the behavior.
        /// An error will be raised if the ClientMemberType is Custom and
        /// there is no GetExpression defined.
        /// </summary>
        public string GetExpression
        {
            get { return _getExpression; }
        }
        private string _getExpression;

        /// <summary>
        /// Custom format string used to get the value of the property
        /// from an object.  The specifier {0} will be replaced by a
        /// reference to the behavior and the specifier {1} will be
        /// replaced with a literal value being assigned to the property.
        /// If the custom expression is null and the ClientMemberType is
        /// Field or Property, an appropriate expression will be used to
        /// set the value on the behavior.  An error will be raised if
        /// the ClientMemberType is Custom and there is no SetExpression
        /// defined.
        /// </summary>
        public string SetExpression
        {
            get { return _setExpression; }
        }
        private string _setExpression;

        /// <summary>
        /// Behavior that contains this property
        /// </summary>
        public Behavior Behavior
        {
            get { return _behavior; }
        }
        private Behavior _behavior;

        /// <summary>
        /// Reference to the page's SynchronizationManager
        /// </summary>
        protected SynchronizationManager SynchronizationManager
        {
            get { return _behavior.Page.SynchronizationManager; }
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="memberType">Type of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        protected BehaviorProperty(Behavior behavior, string name, ClientMemberType memberType, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression)
            : base()
        {
            Assert.IsNotNull(behavior, "behavior cannot be null!");
            _behavior = behavior;
            _behavior.Properties.Add(this);

            _name = name;
            _memberType = memberType;
            _readStrategy = readStrategy;
            _writeStrategy = writeStrategy;
            _lookupExpression = lookupExpression;
            _getExpression = getExpression;
            _setExpression = setExpression;

            // Initialize the cache
            _value = null;
            _synchronized = false;

            // If the property should be set on the first synchronization, call Require
            // to get it added to the queue
            if ((readStrategy & ReadStrategy.Initialize) == ReadStrategy.Initialize)
            {
                Require();
            }

            // If the property should be read on every synchronization, add it to the
            // AlwaysRead queue
            if ((readStrategy & ReadStrategy.Always) == ReadStrategy.Always)
            {
                SynchronizationManager.AlwaysRead.Add(this);
            }
        }

        /// <summary>
        /// Require that a property be added to the synchronization queue if its
        /// cached value is no longer valid so it will be refreshed during the
        /// next synchronization.
        /// </summary>
        public void Require()
        {
            if (!_synchronized)
            {
                SynchronizationManager.Pending.Add(this);
            }
        }

        /// <summary>
        /// Invalidate the cached value of the property so future requests will
        /// require synchronization
        /// </summary>
        public void Invalidate()
        {
            _synchronized = false;
        }

        /// <summary>
        /// Get the value of a property by adding it to the SynchronizationManager's
        /// pending queue and forcing a synchronization
        /// </summary>
        protected virtual void GetValue()
        {
            SynchronizationManager.Pending.Add(this);
            SynchronizationManager.Synchronize();
        }

        /// <summary>
        /// Set the value of a property by adding it to the SynchronizationManager's
        /// pending queue and forcing a synchronization
        /// </summary>
        /// <param name="value"></param>
        /// <remarks>
        /// The value will be retrieved again after updating (in case the client-side
        /// property setter modifies it, etc.).  It will be written now if it has an
        /// immediate synchronizations strategy or on the next refresh otherwise.
        /// </remarks>
        protected virtual void SetValue(object value)
        {
            SynchronizationManager.Updates[this] = value;
            SynchronizationManager.Pending.Add(this);
            if (_writeStrategy == WriteStrategy.Immediate)
            {
                SynchronizationManager.Synchronize();
            }
        }

        /// <summary>
        /// Update the value of the property
        /// </summary>
        /// <param name="value">New value of the property</param>
        internal virtual void UpdateValue(object value)
        {
            _value = value;
            _synchronized = true;
        }
    }

    /// <summary>
    /// Strongly-typed BehaviorProperty
    /// </summary>
    /// <typeparam name="T">Type of the property</typeparam>
    public class BehaviorProperty<T> : BehaviorProperty
    {
        /// <summary>
        /// Cached value of the client-side field or property
        /// </summary>
        public new T Value
        {
            get { return (T) base.Value; }
            set { base.Value = value; }
        }

        /// <summary>
        /// Optional delegate used to convert values returned by GetExpression
        /// </summary>
        public Converter<object, T> PropertyConverter
        {
            get { return _propertyConverter; }
        }
        private Converter<object, T> _propertyConverter;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="memberType">Type of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        protected BehaviorProperty(Behavior behavior, string name, ClientMemberType memberType, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
            : base(behavior, name, memberType, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression)
        {
            _propertyConverter = propertyConverter;
        }

        /// <summary>
        /// Update the value of the property
        /// </summary>
        /// <param name="value">New value of the property</param>
        internal override void UpdateValue(object value)
        {
            value = (_propertyConverter == null) ?
                Common.ConvertJsonValue<T>(value) :
                _propertyConverter(value);
            base.UpdateValue(value);
        }

        #region Static Create Methods
        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, Converter<object, T> propertyConverter)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, null, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, Converter<object, T> propertyConverter)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, string lookupExpression)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, string lookupExpression, Converter<object, T> propertyConverter)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
        {
            return CreateProperty(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, Converter<object, T> propertyConverter)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateProperty(behavior, name, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side property
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateProperty(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
        {
            return new BehaviorProperty<T>(behavior, name, ClientMemberType.Property, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, Converter<object, T> propertyConverter)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, string lookupExpression)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, string lookupExpression, Converter<object, T> propertyConverter)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
        {
            return CreateField(behavior, name, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, null, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, Converter<object, T> propertyConverter)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, null, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, Converter<object, T> propertyConverter)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, lookupExpression, null, null, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateField(behavior, name, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a client-side field
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="name">Name of the client-side member</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateField(Behavior behavior, string name, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
        {
            return new BehaviorProperty<T>(behavior, name, ClientMemberType.Field, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateCustomProperty(Behavior behavior, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateCustomProperty(behavior, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateCustomProperty(Behavior behavior, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
        {
            return CreateCustomProperty(behavior, ReadStrategy.DemandAndInitialize, WriteStrategy.Immediate, lookupExpression, getExpression, setExpression, propertyConverter);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateCustomProperty(Behavior behavior, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression)
        {
            return CreateCustomProperty(behavior, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, null);
        }

        /// <summary>
        /// Create a new BehaviorProperty that references a custom object on the client-side
        /// </summary>
        /// <param name="behavior">Behavior that contains the property</param>
        /// <param name="readStrategy">Strategy used to synchronize the value when requested</param>
        /// <param name="writeStrategy">Strategy used to synchronize the value when set</param>
        /// <param name="lookupExpression">Custom expression used to find the object containing the property</param>
        /// <param name="getExpression">Custom expression used to get the property value from its parent object</param>
        /// <param name="setExpression">Custom expression used to set the property value on its parent object</param>
        /// <param name="propertyConverter">Optional delegate used to convert values returned by getExpression</param>
        /// <returns>New property</returns>
        public static BehaviorProperty<T> CreateCustomProperty(Behavior behavior, ReadStrategy readStrategy, WriteStrategy writeStrategy, string lookupExpression, string getExpression, string setExpression, Converter<object, T> propertyConverter)
        {
            return new BehaviorProperty<T>(behavior, null, ClientMemberType.Custom, readStrategy, writeStrategy, lookupExpression, getExpression, setExpression, propertyConverter);
        }
        #endregion Static Create Methods
    }
}