// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Reflection;
using System.Web.UI;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit;

namespace AjaxControlToolkit.Testing.Client
{
    /// <summary>
    /// The BehaviorEvent models an individual event of a client-side behavior.
    /// It is primarily used to Wait until the event has fired.
    /// </summary>
    public abstract class BehaviorEvent
    {
        /// <summary>
        /// Name of the client-side event being modeled
        /// </summary>
        public string Name
        {
            get { return _name; }
        }
        private string _name;

        /// <summary>
        /// Custom format string used to look up the object containing
        /// the event being modeled.  If the custom expression is null,
        /// a reference to the behavior will be automatically supplied.
        /// The specifier {0} will be replaced by a reference to the behavior
        /// and {1} will be replaced by the behavior's ID.
        /// </summary>
        public string LookupExpression
        {
            get { return _lookupExpression; }
        }
        private string _lookupExpression;

        /// <summary>
        /// JavaScript expression used to return any necessary event arguments
        /// </summary>
        public string EventArgumentsExpression
        {
            get { return _eventArgumentsExpression; }
        }
        private string _eventArgumentsExpression;

        /// <summary>
        /// Behavior that contains this event
        /// </summary>
        public Behavior Behavior
        {
            get { return _behavior; }
        }
        private Behavior _behavior;

        /// <summary>
        /// Reference expression to the object with the event
        /// </summary>
        private string Reference
        {
            get
            {
                return string.IsNullOrEmpty(_lookupExpression) ?
                    _behavior.BehaviorReferenceExpression :
                    string.Format(_lookupExpression, _behavior.BehaviorReferenceExpression, _behavior.BehaviorID);
            }
        }

        /// <summary>
        /// Private default constructor to prevent instantiaton
        /// </summary>
        private BehaviorEvent()
        {
        }

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="behavior">Behavior containing the event</param>
        /// <param name="name">Name of the client-side event</param>
        /// <param name="eventArgumentsExpression">
        /// JavaScript expression used to return any necessary event arguments
        /// </param>
        /// <param name="lookupExpression">
        /// JavaScript expression used to look up the object containing
        /// the event being modeled
        /// </param>
        protected BehaviorEvent(Behavior behavior, string name, string eventArgumentsExpression, string lookupExpression)
        {
            Assert.IsNotNull(behavior, "behavior cannot be null!");
            _behavior = behavior;
            // _behavior.Events.Add(this);

            _name = name;
            _eventArgumentsExpression = eventArgumentsExpression;
            _lookupExpression = lookupExpression;
        }

        /// <summary>
        /// Register the event handler so we'll be able to Wait for it to fire
        /// </summary>
        public void Register()
        {
            string processEventArgs = string.IsNullOrEmpty(_eventArgumentsExpression) ?
                string.Empty :
                "b.add_" + _name + ".__result = (" + _eventArgumentsExpression + ")(sender, eventArgs); ";

            string script = "(function() { " +
                    "var b = " + Reference + "; " +
                    "if (!b) { return false; }; " +
                    "b.add_" + _name + ".__raised = false; " +
                    "b.add_" + _name + "(function(sender, eventArgs) { " +
                        "b.add_" + _name + ".__raised = true; " +
                        processEventArgs +
                    "}); " +
                    "return true; " +
                "})();";
            bool result = Common.ConvertJsonValue<bool>(_behavior.Page.ExecuteScript(script));
            Assert.IsTrue(result, "Failed to register {0} handler on behavior \"{1}\"!", _name, _behavior.BehaviorID);
        }

        /// <summary>
        /// Wait for the event handler to fire
        /// </summary>
        /// <param name="timeout">Number of seconds to wait</param>
        /// <returns>Result of the EventArgumentsExpression</returns>
        public object Wait(int timeout)
        {
            return Wait(timeout, true);
        }

        /// <summary>
        /// Wait for the event handler to fire
        /// </summary>
        /// <param name="timeout">Number of seconds to wait</param>
        /// <param name="throwOnTimeout">Whether a timeout should raise an error</param>
        /// <returns>Result of the EventArgumentsExpression</returns>
        public object Wait(int timeout, bool throwOnTimeout)
        {
            string waitExpression = string.Format("{0}.add_{1}.__raised == true", Reference, _name);
            try
            {
                _behavior.Page.WaitForScript(waitExpression, timeout);
            }
            catch (WebTestingException ex)
            {
                if (throwOnTimeout)
                {
                    throw ex;
                }
                return null;
            }

            string cleanup = "(function() { var b = " + Reference + "; " +
                "b.add_" + _name + ".__raised = false; " +
                "return " +
                    (string.IsNullOrEmpty(_eventArgumentsExpression) ?
                        "null" :
                        "b.add_" + _name + ".__result") +
                "; })()";
            return _behavior.Page.ExecuteScript(cleanup);
        }
    }

    /// <summary>
    /// The BehaviorEvent models an individual event of a client-side behavior.
    /// It is primarily used to Wait until the event has fired.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class BehaviorEvent<T> : BehaviorEvent
    {
        /// <summary>
        /// Optional delegate used to convert anything returned by the
        /// EventArgumentExpression
        /// </summary>
        public Converter<object, T> EventArgumentsConverter
        {
            get { return _eventArgumentsConverter; }
        }
        public Converter<object, T> _eventArgumentsConverter;

        /// <summary>
        /// Constructor
        /// </summary>
        /// <param name="behavior">Behavior containing the event</param>
        /// <param name="name">Name of the client-side event</param>
        /// <param name="eventArgumentsExpression">
        /// JavaScript expression used to return any necessary event arguments
        /// </param>
        /// <param name="eventArgumentsConverter">
        /// Optional delegate used to convert anything returned by the
        /// EventArgumentExpression
        /// </param>
        /// <param name="lookupExpression">
        /// JavaScript expression used to look up the object containing
        /// the event being modeled
        /// </param>
        protected BehaviorEvent(Behavior behavior, string name, string eventArgumentsExpression, Converter<object, T> eventArgumentsConverter, string lookupExpression)
            : base(behavior, name, eventArgumentsExpression, lookupExpression)
        {
            _eventArgumentsConverter = eventArgumentsConverter;
        }

        /// <summary>
        /// Wait for the event handler to fire
        /// </summary>
        /// <param name="timeout">Number of seconds to wait</param>
        /// <returns>Result of the EventArgumentsExpression</returns>
        public new T Wait(int timeout)
        {
            return Wait(timeout, true);
        }

        /// <summary>
        /// Wait for the event handler to fire
        /// </summary>
        /// <param name="timeout">Number of seconds to wait</param>
        /// <param name="throwOnTimeout">Whether a timeout should raise an error</param>
        /// <returns>Result of the EventArgumentsExpression</returns>
        public new T Wait(int timeout, bool throwOnTimeout)
        {
            object result = base.Wait(timeout, throwOnTimeout);
            return (_eventArgumentsConverter == null) ?
                Common.ConvertJsonValue<T>(result) :
                _eventArgumentsConverter(result);
        }

        /// <summary>
        /// Create the behavior event
        /// </summary>
        /// <param name="behavior">Behavior containing the event</param>
        /// <param name="name">Name of the client-side event</param>
        /// <returns>BehaviorEvent</returns>
        public static BehaviorEvent<T> CreateEvent(Behavior behavior, string name)
        {
            return CreateEvent(behavior, name, null);
        }

        /// <summary>
        /// Create the behavior event
        /// </summary>
        /// <param name="behavior">Behavior containing the event</param>
        /// <param name="name">Name of the client-side event</param>
        /// <param name="eventArgumentsExpression">
        /// JavaScript expression used to return any necessary event arguments
        /// </param>
        /// <returns>BehaviorEvent</returns>
        public static BehaviorEvent<T> CreateEvent(Behavior behavior, string name, string eventArgumentsExpression)
        {
            return CreateEvent(behavior, name, eventArgumentsExpression, null, null);
        }

        /// <summary>
        /// Create the behavior event
        /// </summary>
        /// <param name="behavior">Behavior containing the event</param>
        /// <param name="name">Name of the client-side event</param>
        /// <param name="eventArgumentsExpression">
        /// JavaScript expression used to return any necessary event arguments
        /// </param>
        /// <param name="lookupExpression">
        /// JavaScript expression used to look up the object containing
        /// the event being modeled
        /// </param>
        /// <returns>BehaviorEvent</returns>
        public static BehaviorEvent<T> CreateEvent(Behavior behavior, string name, string eventArgumentsExpression, string lookupExpression)
        {
            return CreateEvent(behavior, name, eventArgumentsExpression, null, lookupExpression);
        }

        /// <summary>
        /// Create the behavior event
        /// </summary>
        /// <param name="behavior">Behavior containing the event</param>
        /// <param name="name">Name of the client-side event</param>
        /// <param name="eventArgumentsExpression">
        /// JavaScript expression used to return any necessary event arguments
        /// </param>
        /// <param name="lookupExpression">
        /// JavaScript expression used to look up the object containing
        /// the event being modeled
        /// </param>
        /// <returns>BehaviorEvent</returns>
        public static BehaviorEvent<T> CreateEvent(Behavior behavior, string name, string eventArgumentsExpression, Converter<object, T> eventArgumentsConverter)
        {
            return CreateEvent(behavior, name, eventArgumentsExpression, eventArgumentsConverter, null);
        }

        /// <summary>
        /// Create the behavior event
        /// </summary>
        /// <param name="behavior">Behavior containing the event</param>
        /// <param name="name">Name of the client-side event</param>
        /// <param name="eventArgumentsExpression">
        /// JavaScript expression used to return any necessary event arguments
        /// </param>
        /// <param name="eventArgumentsConverter">
        /// Optional delegate used to convert anything returned by the
        /// EventArgumentExpression
        /// </param>
        /// <param name="lookupExpression">
        /// JavaScript expression used to look up the object containing
        /// the event being modeled
        /// </param>
        /// <returns>BehaviorEvent</returns>
        public static BehaviorEvent<T> CreateEvent(Behavior behavior, string name, string eventArgumentsExpression, Converter<object, T> eventArgumentsConverter, string lookupExpression)
        {
            return new BehaviorEvent<T>(behavior, name, eventArgumentsExpression, eventArgumentsConverter, lookupExpression);
        }
    }
}