<?php
declare(strict_types=1);

namespace popp\r11\parse;

class Scanner
{
    const WORD         = 1;
    const QUOTE        = 2;
    const APOS         = 3;
    const WHITESPACE   = 6;
    const EOL          = 8;
    const CHAR         = 99;
    const EOF          = 0;

    protected $in;
    protected $line_no = 1;
    protected $char_no = 0;
    protected $token;
    protected $token_type;
    protected $regexps;
    public $resultstack = array();

    public function __construct(string $in)
    {
        $this->in = $in;
        $this->setRegexps();
        $this->nextToken();
        $this->eatWhiteSpace();
    }

    // umieść rezultat na stosie rezultatów
    public function pushResult($mixed)
    {
        array_push($this->resultstack, $mixed);
    }

    // usuń rezultat ze stosu rezultatów
    public function popResult()
    {
        return array_pop($this->resultstack);
    }

    // liczba pozycji na stosie rezultatów
    public function resultCount(): int
    {
        return count($this->resultstack);
    }

    // zwróć ostatnią pozycję ze stosu rezultatów
    // ale nie usuwaj jej
    public function peekResult()
    {
        if (empty($this->resultstack)) {
            throw new Exception("pusty stos rezultatów");
        }
        return $this->resultstack[count($this->resultstack) -1 ];
    }

    // konfigurowanie wyrażeń regularnych
    private function setRegexps()
    {
        $this->regexps = array(
                      self::WHITESPACE => '[ \t]',
                      self::EOL => '\n',
                      self::WORD => '[a-zA-Z0-9_-]+\b',
                      self::QUOTE => '"',
                      self::APOS => "'",
        );

        $this->typestrings = array(
                      self::WHITESPACE => 'WHITESPACE',
                      self::EOL => 'EOL',
                      self::WORD => 'WORD',
                      self::QUOTE => 'QUOTE',
                      self::APOS => "APOS",
                      self::CHAR  => 'CHAR',
                      self::EOF => 'EOF'
        );
    }

    // pomijanie białych znaków
    public function eatWhiteSpace(): int
    {
        $ret = 0;
        if ($this->token_type != self::WHITESPACE &&
             $this->token_type != self::EOL ) {
            return $ret;
        }
        while ($this->nextToken() == self::WHITESPACE ||
                $this->token_type == self::EOL ) {
            $ret++;
        }
        return $ret;
    }

    // po podaniu wartości całkowitej, zwróć 
    // odpowiedni ciąg znaków
    // np. 1 => 'WORD',
    public function getTypeString(int $int = -1): string
    {
        if ($int<0) {
            $int=$this->token_type();
        }
        return $this->typestrings[$int];
    }

    // rodzaj bieżącego elementu
    public function tokenType(): int
    {
        return $this->token_type;
    }

    // skanowany tekst ulega skróceniu
    // bo są z niego pobierane
    // kolejne elementy
    public function input(): string
    {
        return $this->in;
    }

    // bieżący element
    public function token(): string
    {
        return $this->token;
    }

    // numer bieżącej linii (przydaje się przy analizie błędów)
    public function lineNo(): int
    {
        return $this->line_no;
    }

    // bieżący numer znaku (char_no)
    public function charNo(): int
    {
        return $this->char_no;
    }

    // próba pobrania kolejnego elementu z wejścia
    // w przypadku nieznalezienia dopasowania
    // jest to element znakowy
    public function nextToken(): int
    {
        if (! strlen($this->in)) {
            return ( $this->token_type = self::EOF );
        }

        $ret = 0;
        foreach ($this->regexps as $type => $regex) {
            if ($ret = $this->testToken($regex, $type)) {
                if ($ret == self::EOL) {
                    $this->line_no++;
                    $this->char_no = 0;
                } else {
                    $this->char_no += strlen($this->token());
                }
                return $ret;
            }
        }
        $this->token = substr($this->in, 0, 1);
        $this->in    = substr($this->in, 1);
        $this->char_no += 1;
        return ( $this->token_type = self::CHAR );
    }

    // sprawdzanie dopasowania za pomocą wyrażenia regularnego
    private function testToken($regex, $type): int
    {
        $matches = array();
        if (preg_match("/^($regex)(.*)/s", $this->in, $matches)) {
            $this->token = $matches[1];
            $this->in    = $matches[2];
            return ( $this->token_type  = $type );
        }
        return 0;
    }

    // klonowanie w przypadku istnienia drugiego skanera
    public function updateToMatch(Scanner $other)
    {
        $this->in = $other->in;
        $this->token = $other->token;
        $this->token_type = $other->token_type;
        $this->char_no = $other->char_no;
        $this->line_no = $other->line_no;
        $this->resultstack = $other->resultstack;
    }
}
