<?php

declare(strict_types=1);

namespace popp\r13\zestaw07;

/* listing 13.40 */
class IdentityObject
{
    protected ?Field $currentfield = null;
    protected array $fields = [];
    private array $enforce = [];

    // obiekt tożsamości może być początkowo pusty albo z polem
    public function __construct(?string $field = null, ?array $enforce = null)
    {
        if (! is_null($enforce)) {
            $this->enforce = $enforce;
        }

        if (! is_null($field)) {
            $this->field($field);
        }
    }

    // nazwy pól, do których ten dany obiekt będzie ograniczony
    public function getObjectFields(): array
    {
        return $this->enforce;
    }

    // nowe pole;
    // jeśli nie jest kompletne, wyrzucamy wyjątek
    // (np. kiedy dostaniemy age zamiast age > 40)
    // metoda zwraca referencję do bieżącego obiektu,
    // umożliwiając konstruowanie kaskadowych wywołań
    public function field(string $fieldname): self
    {
        if (! $this->isVoid() && $this->currentfield->isIncomplete()) {
            throw new \Exception("Niekompletne pole");
        }

        $this->enforceField($fieldname);

        if (isset($this->fields[$fieldname])) {
            $this->currentfield = $this->fields[$fieldname];
        } else {
            $this->currentfield = new Field($fieldname);
            $this->fields[$fieldname] = $this->currentfield;
        }

        return $this;
    }

    // czy obiekt tożsamości ma już jakieś pola?
    public function isVoid(): bool
    {
        return empty($this->fields);
    }

    // czy przekazana nazwa pola jest poprawna?
    public function enforceField(string $fieldname): void
    {
        if (! in_array($fieldname, $this->enforce) && ! empty($this->enforce)) {
            $forcelist = implode(', ', $this->enforce);
            throw new \Exception("{$fieldname} nie jest poprawną nazwą pola ($forcelist)");
        }
    }

    // dodaje operator równości do bieżącego pola
    // (np. 'age' zamienia na age=40)
    // i zwraca referencję bieżącego obiektu (via operator())
    public function eq($value): self
    {
        return $this->operator("=", $value);
    }

    // mniejsze od
    public function lt($value): self
    {
        return $this->operator("<", $value);
    }

    // większe od
    public function gt($value): self
    {
        return $this->operator(">", $value);
    }

    // wykonawca metod operatorów:
    // bierze bieżące pole i dodaje operator z wartością
    private function operator(string $symbol, $value): self
    {
        if ($this->isVoid()) {
            throw new \Exception("brak pól");
        }

        $this->currentfield->addTest($symbol, $value);

        return $this;
    }

    // zwraca wszystkie porównania zmontowane dotychczas
    // (w postaci tablicy asocjacyjnej)
    public function getComps(): array
    {
        $ret = [];

        foreach ($this->fields as $field) {
            $ret = array_merge($ret, $field->getComps());
        }

        return $ret;
    }
/* /listing 13.40 */

    public function __toString(): string
    {
        $ret = [];

        foreach ($this->getComps() as $compdata) {
            $ret[] = "{$compdata['name']} {$compdata['operator']} {$compdata['value']}";
        }

        return implode(" AND ", $ret);
    }
/* listing 13.40 */
}
/* /listing 13.40 */
