# = Class: zendserver
#
# This is the main zendserver class
#
#
# == Parameters
#
# Standard class parameters
# Define the general class behaviour and customizations
#
# [*my_class*]
#   Name of a custom class to autoload to manage module's customizations
#   If defined, zendserver class will automatically "include $my_class"
#   Can be defined also by the (top scope) variable $zendserver_myclass
#
# [*source*]
#   Sets the content of source parameter for main configuration file
#   If defined, zendserver main config file will have the param: source => $source
#   Can be defined also by the (top scope) variable $zendserver_source
#
# [*source_dir*]
#   If defined, the whole zendserver configuration directory content is retrieved
#   recursively from the specified source
#   (source => $source_dir , recurse => true)
#   Can be defined also by the (top scope) variable $zendserver_source_dir
#
# [*source_dir_purge*]
#   If set to true (default false) the existing configuration directory is
#   mirrored with the content retrieved from source_dir
#   (source => $source_dir , recurse => true , purge => true)
#   Can be defined also by the (top scope) variable $zendserver_source_dir_purge
#
# [*template*]
#   Sets the path to the template to use as content for main configuration file
#   If defined, zendserver main config file has: content => content("$template")
#   Note source and template parameters are mutually exclusive: don't use both
#   Can be defined also by the (top scope) variable $zendserver_template
#
# [*options*]
#   An hash of custom options to be used in templates for arbitrary settings.
#   Can be defined also by the (top scope) variable $zendserver_options
#
# [*service_autorestart*]
#   Automatically restarts the zendserver service when there is a change in
#   configuration files. Default: true, Set to false if you don't want to
#   automatically restart the service.
#
# [*service_requires*]
#   Overwrites the service dependencies, which are by default: Package['zendserver'].
#   Set this parameter to a custom set of requirements, if you want to let the
#   Apache service depend on more than just the package dependency.
#
# [*absent*]
#   Set to 'true' to remove package(s) installed by module
#   Can be defined also by the (top scope) variable $zendserver_absent
#
# [*disable*]
#   Set to 'true' to disable service(s) managed by module
#   Can be defined also by the (top scope) variable $zendserver_disable
#
# [*disableboot*]
#   Set to 'true' to disable service(s) at boot, without checks if it's running
#   Use this when the service is managed by a tool like a cluster software
#   Can be defined also by the (top scope) variable $zendserver_disableboot
#
# [*monitor*]
#   Set to 'true' to enable monitoring of the services provided by the module
#   Can be defined also by the (top scope) variables $zendserver_monitor
#   and $monitor
#
# [*monitor_tool*]
#   Define which monitor tools (ad defined in Example42 monitor module)
#   you want to use for zendserver checks
#   Can be defined also by the (top scope) variables $zendserver_monitor_tool
#   and $monitor_tool
#
# [*monitor_target*]
#   The Ip address or hostname to use as a target for monitoring tools.
#   Default is the fact $ipaddress
#   Can be defined also by the (top scope) variables $zendserver_monitor_target
#   and $monitor_target
#
# [*puppi*]
#   Set to 'true' to enable creation of module data files that are used by puppi
#   Can be defined also by the (top scope) variables $zendserver_puppi and $puppi
#
# [*puppi_helper*]
#   Specify the helper to use for puppi commands. The default for this module
#   is specified in params.pp and is generally a good choice.
#   You can customize the output of puppi commands for this module using another
#   puppi helper. Use the define puppi::helper to create a new custom helper
#   Can be defined also by the (top scope) variables $zendserver_puppi_helper
#   and $puppi_helper
#
# [*firewall*]
#   Set to 'true' to enable firewalling of the services provided by the module
#   Can be defined also by the (top scope) variables $zendserver_firewall
#   and $firewall
#
# [*firewall_tool*]
#   Define which firewall tool(s) (ad defined in Example42 firewall module)
#   you want to use to open firewall for zendserver port(s)
#   Can be defined also by the (top scope) variables $zendserver_firewall_tool
#   and $firewall_tool
#
# [*firewall_src*]
#   Define which source ip/net allow for firewalling zendserver. Default: 0.0.0.0/0
#   Can be defined also by the (top scope) variables $zendserver_firewall_src
#   and $firewall_src
#
# [*firewall_dst*]
#   Define which destination ip to use for firewalling. Default: $ipaddress
#   Can be defined also by the (top scope) variables $zendserver_firewall_dst
#   and $firewall_dst
#
# [*debug*]
#   Set to 'true' to enable modules debugging
#   Can be defined also by the (top scope) variables $zendserver_debug and $debug
#
# [*audit_only*]
#   Set to 'true' if you don't intend to override existing configuration files
#   and want to audit the difference between existing files and the ones
#   managed by Puppet.
#   Can be defined also by the (top scope) variables $zendserver_audit_only
#   and $audit_only
#
# Default class params - As defined in zendserver::params.
# Note that these variables are mostly defined and used in the module itself,
# overriding the default values might not affected all the involved components.
# Set and override them only if you know what you're doing.
# Note also that you can't override/set them via top scope variables.
#
# [*package*]
#   The name of zendserver package
#
# [*service*]
#   The name of zendserver service
#
# [*service_status*]
#   If the zendserver service init script supports status argument
#
# [*process*]
#   The name of zendserver process
#
# [*process_args*]
#   The name of zendserver arguments. Used by puppi and monitor.
#   Used only in case the zendserver process name is generic (java, ruby...)
#
# [*process_user*]
#   The name of the user zendserver runs with. Used by puppi and monitor.
#
# [*config_dir*]
#   Main configuration directory. Used by puppi
#
# [*config_file*]
#   Main configuration file path
#
# [*config_file_mode*]
#   Main configuration file path mode
#
# [*config_file_owner*]
#   Main configuration file path owner
#
# [*config_file_group*]
#   Main configuration file path group
#
# [*config_file_init*]
#   Path of configuration file sourced by init script
#
# [*pid_file*]
#   Path of pid file. Used by monitor
#
# [*data_dir*]
#   Path of application data directory. Used by puppi
#
# [*log_dir*]
#   Base logs directory. Used by puppi
#
# [*log_file*]
#   Log file(s). Used by puppi
#
# [*port*]
#   The listening port, if any, of the service.
#   This is used by monitor, firewall and puppi (optional) components
#   Note: This doesn't necessarily affect the service configuration file
#   Can be defined also by the (top scope) variable $zendserver_port
#
# [*protocol*]
#   The protocol used by the the service.
#   This is used by monitor, firewall and puppi (optional) components
#   Can be defined also by the (top scope) variable $zendserver_protocol
#
#
# == Examples
#
# You can use this class in 2 ways:
# - Set variables (at top scope level on in a ENC) and "include zendserver"
# - Call zendserver as a parametrized class
#
# See README for details.
#
#
# == Author
#   Alessandro Franceschi <al@lab42.it/>
#
class zendserver (
  $use_ce              = params_lookup( 'use_ce' ),
  $install_extra_ext   = params_lookup( 'install_extra_ext' ),
  $php_version         = params_lookup( 'php_version' ),
  $my_class            = params_lookup( 'my_class' ),
  $source              = params_lookup( 'source' ),
  $source_dir          = params_lookup( 'source_dir' ),
  $source_dir_purge    = params_lookup( 'source_dir_purge' ),
  $template            = params_lookup( 'template' ),
  $service_autorestart = params_lookup( 'service_autorestart' , 'global' ),
  $options             = params_lookup( 'options' ),
  $absent              = params_lookup( 'absent' ),
  $disable             = params_lookup( 'disable' ),
  $disableboot         = params_lookup( 'disableboot' ),
  $monitor             = params_lookup( 'monitor' , 'global' ),
  $monitor_tool        = params_lookup( 'monitor_tool' , 'global' ),
  $monitor_target      = params_lookup( 'monitor_target' , 'global' ),
  $puppi               = params_lookup( 'puppi' , 'global' ),
  $puppi_helper        = params_lookup( 'puppi_helper' , 'global' ),
  $firewall            = params_lookup( 'firewall' , 'global' ),
  $firewall_tool       = params_lookup( 'firewall_tool' , 'global' ),
  $firewall_src        = params_lookup( 'firewall_src' , 'global' ),
  $firewall_dst        = params_lookup( 'firewall_dst' , 'global' ),
  $debug               = params_lookup( 'debug' , 'global' ),
  $audit_only          = params_lookup( 'audit_only' , 'global' ),
  $package             = params_lookup( 'package' ),
  $service             = params_lookup( 'service' ),
  $service_status      = params_lookup( 'service_status' ),
  $service_requires    = params_lookup( 'service_requires' ),
  $process             = params_lookup( 'process' ),
  $process_args        = params_lookup( 'process_args' ),
  $process_user        = params_lookup( 'process_user' ),
  $config_dir          = params_lookup( 'config_dir' ),
  $config_file         = params_lookup( 'config_file' ),
  $config_file_mode    = params_lookup( 'config_file_mode' ),
  $config_file_owner   = params_lookup( 'config_file_owner' ),
  $config_file_group   = params_lookup( 'config_file_group' ),
  $config_file_init    = params_lookup( 'config_file_init' ),
  $pid_file            = params_lookup( 'pid_file' ),
  $data_dir            = params_lookup( 'data_dir' ),
  $log_dir             = params_lookup( 'log_dir' ),
  $log_file            = params_lookup( 'log_file' ),
  $port                = params_lookup( 'port' ),
  $protocol            = params_lookup( 'protocol' )
) inherits zendserver::params {

  $bool_use_ce=any2bool($use_ce)
  $bool_source_dir_purge=any2bool($source_dir_purge)
  $bool_service_autorestart=any2bool($service_autorestart)
  $bool_absent=any2bool($absent)
  $bool_disable=any2bool($disable)
  $bool_disableboot=any2bool($disableboot)
  $bool_monitor=any2bool($monitor)
  $bool_puppi=any2bool($puppi)
  $bool_firewall=any2bool($firewall)
  $bool_debug=any2bool($debug)
  $bool_audit_only=any2bool($audit_only)

    ### Calculation of variables that dependes on arguments
    $zend_package = $::operatingsystem ? {
        /(?i:Ubuntu|Debian|Mint)/ => "zend-server-php-${php_version}",
        /(?i:SLES|OpenSuSE)/      => '',
        default                   => '',
    }

    $zend_ce_package = $::operatingsystem ? {
        /(?i:Ubuntu|Debian|Mint)/ => "zend-server-ce-php-${php_version}",
        /(?i:SLES|OpenSuSE)/      => '',
        default                   => '',
    }

    $php_package = $::operatingsystem ? {
        /(?i:Ubuntu|Debian|Mint)/ => "libapache2-mod-php-${php_version}-zend-server",
        /(?i:SLES|OpenSuSE)/      => '',
        default                   => '',
    }

    $ext_package = $::operatingsystem ? {
        /(?i:Ubuntu|Debian|Mint)/ => "php-${php_version}-extra-extensions-zend-server",
        /(?i:SLES|OpenSuSE)/      => '',
        default                   => '',
    }

  ### Definition of some variables used in the module
  $manage_package = $zendserver::bool_absent ? {
    true  => 'absent',
    false => 'present',
  }

  $manage_service_enable = $zendserver::bool_disableboot ? {
    true    => false,
    default => $zendserver::bool_disable ? {
      true    => false,
      default => $zendserver::bool_absent ? {
        true  => false,
        false => true,
      },
    },
  }

  $manage_service_ensure = $zendserver::bool_disable ? {
    true    => 'stopped',
    default =>  $zendserver::bool_absent ? {
      true    => 'stopped',
      default => 'running',
    },
  }

  $manage_service_autorestart = $zendserver::bool_service_autorestart ? {
    true    => 'Service[zendserver]',
    false   => undef,
  }

  $manage_file = $zendserver::bool_absent ? {
    true    => 'absent',
    default => 'present',
  }

  $manage_file_replace = $zendserver::bool_audit_only ? {
    true  => false,
    false => true,
  }

  $manage_file_source = $zendserver::source ? {
    ''        => undef,
    default   => $zendserver::source,
  }

  $manage_file_content = $zendserver::template ? {
    ''        => undef,
    default   => template($zendserver::template),
  }

  ### Prerequisites
  require zendserver::prerequisites

  ### Managed resources
  case $::operatingsystem {
      redhat,centos,scientific,oraclelinux : { }
      ubuntu,debian : {
          package { 'zendserver':
              ensure  => $zendserver::manage_package,
              name    => $zendserver::zend_package,
              require => [ Exec['aptget_update'] , File["apt_repository_zend"] ],
          }

          if ($zendserver::bool_use_ce) {
                package { 'zendserver-ce':
                    ensure  => $zendserver::manage_package,
                    name    => $zendserver::zend_ce_package,
                    require => [ Package['zendserver'] ],
                }
          }
      }
      default: { }
  }

  package { 'zendserver-php':
    ensure  => $zendserver::manage_package,
    name    => $zendserver::php_package,
    require => Package['zendserver']
  }

  if $zendserver::install_extra_ext {
    package { 'zendserver-extra-extensions':
      ensure  => $zendserver::manage_package,
      name    => $zendserver::ext_package,
      require => Package['zendserver']
    }
  }

  file { '/usr/local/bin/php':
    ensure  => 'link',
    target  => "${zendserver::bin_dir}/php",
    require => Package['zendserver'],
  }

  file { '/usr/local/bin/phpize':
    ensure  => 'link',
    target  => "${zendserver::bin_dir}/phpize",
    require => Package['zendserver'],
  }

  file { '/usr/local/bin/php-config':
    ensure  => 'link',
    target  => "${zendserver::bin_dir}/php-config",
    require => Package['zendserver'],
  }

  file { '/usr/local/bin/pecl':
    ensure  => 'link',
    target  => "${zendserver::bin_dir}/pecl",
    require => Package['zendserver'],
  }

  ### Include custom class if $my_class is set
  if $zendserver::my_class {
    include $zendserver::my_class
  }
}
