<?php

declare(strict_types=1);

/**
 * Obsługuje czynności administracyjne
 * 
 * PHP w wersji 7
 *
 * LICENCJA: Plik źródłowy udostępniany na licencji MIT opisanej pod adresem
 * http://www.opensource.org/licenses/mit-license.html
 *
 * @author Jason Lengstorf <jason.lengstorf@ennuidesign.com>
 * @copyright 2010 Ennui Design
 * @license http://www.opensource.org/licenses/mit-license.html
 */
class Admin extends DB_Connect
{

    /**
     * Określa długość ciągu zaburzającego, wykorzystywanego przy szyfrowaniu haseł
     * 
     * @var int długość ciągu zaburzającego
     */
    private $_saltLength = 7;

    /**
     * Zapisuje lub tworzy obiekt bazy danych i ustawia długość ciągu zaburzającego
     * 
     * @param object $db obiekt bazy danych
     * @param int $saltLength długość ciągu zaburzającego
     */
    public function __construct($db=NULL, $saltLength=NULL)
    {
        parent::__construct($db);

        /*
         * Jeżeli przekazano liczbę całkowitą, zapisz długość ciągu zaburzającego
         */
        if ( is_int($saltLength) )
        {
            $this->_saltLength = $saltLength;
        }
    }

    /**
     * Sprawdza dane uwierzytelniające użytkownika
     * 
     * @return mixed TRUE w przypadku powodzenia lub komunikat o błędzie
     */
    public function processLoginForm()
    {
        /*
         * Kończy działanie przy braku odpowiedniej wartości parametru action
         */
        if ( $_POST['action']!='user_login' )
        {
            return "Nieprawidłowa wartość parametru action w metodzie processLoginForm.";
        }

        /*
         * Dla bezpieczeństwa zamienia znaki specjalne na encje
         */
        $uname = htmlentities($_POST['uname'], ENT_QUOTES);
        $pword = htmlentities($_POST['pword'], ENT_QUOTES);

        /*
         * Pobiera odpowiednie informacje z bazy, jeżeli istnieją
         */
        $sql = "SELECT
                    `user_id`, `user_name`, `user_email`, `user_pass`
                FROM `users`
                WHERE
                    `user_name` = :uname
                LIMIT 1";
        try
        {
            $stmt = $this->db->prepare($sql);
            $stmt->bindParam(':uname', $uname, PDO::PARAM_STR);
            $stmt->execute();
            $user = array_shift($stmt->fetchAll());
            $stmt->closeCursor();
        }
        catch ( Exception $e )
        {
            die ( $e->getMessage() );
        }

        /*
         * Kończy działanie, jeżeli nazwie użytkownika nie odpowiada żaden rekord w bazie
         */
        if ( !isset($user) )
        {
            return "Nieprawidłowa nazwa użytkownika lub hasło.";
        }

        /*
         * Zaszyfruj hasło podane przez użytkownika
         */
        $hash = $this->_getSaltedHash($pword, $user['user_pass']);

        /*
         * Sprawdź, czy zaszyfrowane hasło odpowiada zapisanemu w bazie
         */
        if ( $user['user_pass']==$hash )
        {
            /*
             * Zapisz w sesji dane użytkownika w postaci tablicy
             */
            $_SESSION['user'] = array(
                    'id' => $user['user_id'],
                    'name' => $user['user_name'],
                    'email' => $user['user_email']
                );

            return TRUE;
        }

        /*
         * Zakończ działanie, jeśli hasła są inne
         */
        else
        {
            return "Nieprawidłowa nazwa użytkownika lub hasło.";
        }
    }

    /**
     * Wylogowuje użytkownika
     * 
     * @return mixed TRUE w przypadku powodzenia, komunikat w przypadku niepowodzenia
     */
    public function processLogout()
    {
        /*
         * Kończy działanie przy braku odpowiedniej wartości parametru action
         */
        if ( $_POST['action']!='user_logout' )
        {
            return "Nieprawidłowa wartość parametru action w metodzie processLogout.";
        }

        /*
         * Usuwa tablicę użytkownika z bieżącej sesji
         */
        session_destroy();
        return TRUE;
    }

    /**
     * Szyfruje podany łańcuch przy użyciu ciągu zaburzającego
     * 
     * @param string $string łańcuch do zaszyfrowania
     * @param string $salt wartość, z której wyodrębnia się ciąg zaburzający
     * @return string skrót hasła zaszyfrowany z wykorzystaniem ciągu zaburzającego
     */
    private function _getSaltedHash($string, $salt=NULL)
    {
        /*
         * Generuj ciąg zaburzający, jeśli nie przekazano go w argumencie
         */
        if ( $salt==NULL )
        {
            $salt = substr(md5((string)time()), 0, $this->_saltLength);
        }

        /*
         * Wyodrębnij ciąg zaburzający, jeżeli przekazano argument $salt
         */
        else
        {
            $salt = substr($salt, 0, $this->_saltLength);
        }

        /*
         * Połącz ciąg zaburzający z zaszyfrowanym hasłem i zwróć otrzymaną wartość
         */
        return $salt . sha1($salt . $string);
    }

}

?>