-- Ten skrypt SQL*Plus (lob_schema.sql):
--   1. Tworzy uytkownika lob_user
--   2. Tworzy baz danych
--   3. Wypenia baz danych przykadowymi danymi
--   4. Tworzy kod PL/SQL

-- Usunicie uytkownika lob_user jeeli istnieje.
-- To polecenie moe zwrci bd jeli uytkownik nie istnieje; nie naley si tym przejmowa.
DROP USER lob_user CASCADE;

-- Usunicie katalogu z przykadowymi plikami jeeli istnieje.
-- To polecenie moe zwrci bd jeli katalog nie istnieje; nie naley si tym przejmowa.
DROP DIRECTORY SAMPLE_FILES_DIR;

-- utworzenie uytkownika lob_user
CREATE USER lob_user IDENTIFIED BY lob_password;

-- utworzenie katalogu BFILE
CREATE DIRECTORY SAMPLE_FILES_DIR AS 'C:\sample_files';

-- przydzielenie uprawnie do odczytu i zapisu w katalogu BFILE
GRANT read, write ON DIRECTORY SAMPLE_FILES_DIR TO lob_user;

-- nadanie uprawnie uytkownikowi lob_user do czenia i tworzenia obiektw bazy danych
GRANT connect, resource TO lob_user;

-- przydzielenie uytkownikowi miejsca w przestrzeni tabel users
ALTER USER lob_user QUOTA 10M ON users;

-- poczenie jako uytkownik lob_user
CONNECT lob_user/lob_password;

-- tworzenie tabel
CREATE TABLE clob_content (
  id          INTEGER PRIMARY KEY,
  clob_column CLOB NOT NULL
);

CREATE TABLE blob_content (
  id          INTEGER PRIMARY KEY,
  blob_column BLOB NOT NULL
);

CREATE TABLE bfile_content (
  id           INTEGER PRIMARY KEY,
  bfile_column BFILE NOT NULL
);


CREATE TABLE long_content (
  id INTEGER PRIMARY KEY,
  long_column LONG NOT NULL
);

CREATE TABLE long_raw_content (
  id INTEGER PRIMARY KEY,
  long_raw_column LONG RAW NOT NULL
);

-- wstawianie wierszy
INSERT INTO clob_content (
  id, clob_column
) VALUES (
  1, TO_CLOB('Wije si w ciasnym kku')
);

INSERT INTO clob_content (
  id, clob_column
) VALUES (
  2, TO_CLOB(' od dnia do dnia ')
);

INSERT INTO blob_content (
  id, blob_column
) VALUES (
  1, TO_BLOB('100111010101011111')
);

INSERT INTO blob_content (
  id, blob_column
) VALUES (
  2, TO_BLOB('A0FFB71CF90DE')
);

INSERT INTO bfile_content (
  id, bfile_column
) VALUES (
  1, BFILENAME('SAMPLE_FILES_DIR', 'textContent.txt')
);

INSERT INTO bfile_content (
  id, bfile_column
) VALUES (
  2, BFILENAME('SAMPLE_FILES_DIR', 'binaryContent.doc')
);

INSERT INTO long_content (
  id, long_column
) VALUES (
  1, ' Wije si w ciasnym kku'
);

INSERT INTO long_content (
  id, long_column
) VALUES (
  2, ' od dnia do dnia'
);

INSERT INTO long_raw_content (
  id, long_raw_column
) VALUES (
  1, '100111010101011111'
);

INSERT INTO long_raw_content (
  id, long_raw_column
) VALUES (
  2, 'A0FFB71CF90DE'
);

CREATE OR REPLACE PROCEDURE get_clob_locator(
  p_clob IN OUT CLOB,
  p_id IN INTEGER
) AS
BEGIN

-- pobiera lokalizator LOB i zapisuje go w p_clob
  SELECT clob_column
  INTO p_clob
  FROM clob_content
  WHERE id = p_id;
END get_clob_locator;
/


CREATE OR REPLACE PROCEDURE get_blob_locator(
  p_blob IN OUT BLOB,
  p_id IN INTEGER
) AS
BEGIN

-- pobiera lokalizator LOB i zapisuje go w p_blob
  SELECT blob_column
  INTO p_blob
  FROM blob_content
  WHERE id = p_id;
END get_blob_locator;
/


CREATE OR REPLACE PROCEDURE read_clob_example(
  p_id IN INTEGER
) AS
  v_clob CLOB;
  v_offset INTEGER := 1;
  v_amount INTEGER := 50;
  v_char_buffer VARCHAR2(50);
BEGIN
  -- pobiera lokalizator LOB i zapisuje go w v_clob
  get_clob_locator(v_clob, p_id); 

  -- odczytuje v_amount znakw z v_clob do v_char_buffer,
  -- rozpoczynajc w pozycji okrelonej przez v_offset
  DBMS_LOB.READ(v_clob, v_amount, v_offset, v_char_buffer);

  -- wywietla zawarto v_char_buffer
  DBMS_OUTPUT.PUT_LINE('v_char_buffer = ' || v_char_buffer);
  DBMS_OUTPUT.PUT_LINE('v_amount = ' || v_amount);
END read_clob_example;
/


CREATE OR REPLACE PROCEDURE read_blob_example(
  p_id IN INTEGER
) AS
  v_blob BLOB;
  v_offset INTEGER := 1;
  v_amount INTEGER := 25;
  v_binary_buffer RAW(25);
BEGIN
  -- pobiera lokalizator obiektu LOB i zapisuje go w v_blob
  get_blob_locator(v_blob, p_id);

  -- odczytuje v_amount bajtw z v_vlob do v_binary_buffer,
  -- rozpoczynajc w pozycji okrelonej przez v_offset
  DBMS_LOB.READ(v_blob, v_amount, v_offset, v_binary_buffer);

  -- wywietla zawarto v_binary_buffer
  DBMS_OUTPUT.PUT_LINE('v_binary_buffer = ' || v_binary_buffer);
  DBMS_OUTPUT.PUT_LINE('v_amount = ' || v_amount);
END read_blob_example;
/

CREATE OR REPLACE PROCEDURE append_example AS
  v_src_clob CLOB;
  v_dest_clob CLOB;
BEGIN
  -- pobiera do v_src_clob lokalizator LOB obiektu
  -- CLOB znajdujcego si w wierszu 2 tabeli clob_content
  get_clob_locator(v_src_clob, 2);

  -- pobiera w celu modyfikacji do v_dest_clob lokalizator LOB obiektu CLOB
  -- znajdujcego si w wierszu 1 tabeli clob_content
  -- (w celu modyfikacji, poniewa pniej za pomoc metody APPEND() zostanie
  -- doczony obiekt CLOB
  SELECT clob_column
  INTO v_dest_clob
  FROM clob_content
  WHERE id = 1
  FOR UPDATE;

  -- czyta i wywietla zawarto CLOB nr 1
  read_clob_example(1);

  -- uywa APPEND() do skopiowania zawarto v_src_clob do v_dest_clob
  DBMS_LOB.APPEND(v_dest_clob, v_src_clob);

  -- czyta i wywietla zawarto CLOB nr 1
  -- a nastpnie wycofuje zmian
  read_clob_example(1);
  ROLLBACK;
END append_example;
/

CREATE OR REPLACE PROCEDURE compare_example AS
  v_clob1 CLOB;
  v_clob2 CLOB;
  v_return INTEGER;
BEGIN
  -- pobiera lokalizatory LOB
  get_clob_locator(v_clob1, 1);
  get_clob_locator(v_clob2, 2);

  -- porwnuje v_clob1 z v_clob2 (COMPARE() zwraca 1
  -- poniewa zawarto v_clob1 z v_clob2 jest rna)
  DBMS_OUTPUT.PUT_LINE('Porwnywanie v_clob1 i v_clob2');
  v_return := DBMS_LOB.COMPARE(v_clob1, v_clob2);
  DBMS_OUTPUT.PUT_LINE('v_return = ' || v_return);

  -- porwnuje v_clob1 z v_clob1 (COMPARE() zwraca 0
  -- poniewa zawarto obiektw jest identyczna)
  DBMS_OUTPUT.PUT_LINE('Porwnywanie v_clob1 z v_clob1');
  v_return := DBMS_LOB.COMPARE(v_clob1, v_clob1);
  DBMS_OUTPUT.PUT_LINE('v_return = ' || v_return);
END compare_example;
/
CREATE OR REPLACE PROCEDURE copy_example AS
  v_src_clob CLOB;
  v_dest_clob CLOB;
  v_src_offset INTEGER := 1;
  v_dest_offset INTEGER := 7;
  v_amount INTEGER := 5;
BEGIN
  -- pobiera do v_src_clob lokalizator LOB obiektu CLOB znajdujcego si
  -- w wierszu 2 tabeli clob_content table
  get_clob_locator(v_src_clob, 2);

  -- pobiera do v_dest_clob lokalizator LOB obiektu CLOB znajdujcego si
  -- w wierszu 1 tabeli clob_content table (w celu aktualizacji,
  -- poniewa pniej zostan dodane dane za pomoc metody COPY())
  SELECT clob_column
  INTO v_dest_clob
  FROM clob_content
  WHERE id = 1
  FOR UPDATE;

  -- czyta i wywietla zawarto CLOB nr 1
  read_clob_example(1);

  -- uywajc COPY() kopiuje v_amount znakw z v_src_clob do v_dest_clob,
  -- rozpoczynajc w przesuniciach okrelonych przez v_dest_offset i
  -- v_src_offset
  DBMS_LOB.COPY(
    v_dest_clob, v_src_clob,
    v_amount, v_dest_offset, v_src_offset
  );

  -- czyta i wywietla zawarto CLOB nr 1,
  -- a nastpnie wycofuje zmian
  read_clob_example(1);
ROLLBACK;
END copy_example;
/
CREATE OR REPLACE PROCEDURE temporary_lob_example AS
  v_clob CLOB;
  v_amount INTEGER;
  v_offset INTEGER := 1;
  v_char_buffer VARCHAR2(18) := 'Julia jest Socem';
BEGIN
  -- za pomoc CREATETEMPORARY() tworzy tymczasowy obiekt CLOB o nazwie v_clob
  DBMS_LOB.CREATETEMPORARY(v_clob, TRUE);
  -- za pomoc WRITE() zapisuje zawarto v_char_buffer do v_clob
  v_amount := LENGTH(v_char_buffer);
  DBMS_LOB.WRITE(v_clob, v_amount, v_offset, v_char_buffer);

  -- za pomoc ISTEMPORARY() sprawdza, czy v_clob  jest obiektem tymczasowym
  IF (DBMS_LOB.ISTEMPORARY(v_clob) = 1) THEN
    DBMS_OUTPUT.PUT_LINE('v_clob jest obiektem tymczasowym');
  END IF;

  -- za pomoc READ() czyta zawarto v_clob do v_char_buffer
  DBMS_LOB.READ(
    v_clob, v_amount, v_offset, v_char_buffer
  );
  DBMS_OUTPUT.PUT_LINE('v_char_buffer = ' || v_char_buffer);

  -- za pomoc FREETEMPORARY() zwalnia v_clob
  DBMS_LOB.FREETEMPORARY(v_clob);
END temporary_lob_example;
/
CREATE OR REPLACE PROCEDURE erase_example IS
  v_clob CLOB;
  v_offset INTEGER := 2;
  v_amount INTEGER := 5;
BEGIN
  -- pobiera do v_dest_clob lokalizator LOB obiektu CLOB
  -- znajdujcego si w wierszu nr 1 tabeli clob_content table
  -- (w celu modyfikacji, poniewa CLOB zostanie pniej usunity
  -- za pomoc metody ERASE())
  SELECT clob_column
  INTO v_clob
  FROM clob_content
  WHERE id = 1
  FOR UPDATE;

  -- czyta i wywietla zawarto CLOB nr 1
  read_clob_example(1);

  -- za pomoc ERASE() usuwa v_amount znakw
  -- z v_clob, rozpoczynajc w v_offset
  DBMS_LOB.ERASE(v_clob, v_amount, v_offset);

  -- czyta i wywietla zawarto  CLOB nr 1,
  -- a nastpnie wycofuje zmian
  read_clob_example(1);
  ROLLBACK;
END erase_example;
/
CREATE OR REPLACE PROCEDURE instr_example AS
  v_clob CLOB;
  v_char_buffer VARCHAR2(50) := ' Ono jest wschodem, a Julia jest socem';
  v_pattern NVARCHAR2(7);
  v_offset INTEGER := 1;
  v_amount INTEGER;
  v_occurrence INTEGER;
  v_return INTEGER;
BEGIN
  -- za pomoc CREATETEMPORARY() tworzy tymczasowy obiekt CLOB o nazwie v_clob
  DBMS_LOB.CREATETEMPORARY(v_clob, TRUE);

  -- za pomoc WRITE() zapisuje zawarto v_char_buffer do v_clob
  v_amount := LENGTH(v_char_buffer);
  DBMS_LOB.WRITE(v_clob, v_amount, v_offset, v_char_buffer);

  -- za pomoc READ() odczytuje zawarto v_clob do v_char_buffer
  DBMS_LOB.READ(v_clob, v_amount, v_offset, v_char_buffer);
  DBMS_OUTPUT.PUT_LINE('v_char_buffer = ' || v_char_buffer);

  -- za pomoc INSTR() wyszukuje w v_clob drugiego wystpienia sowa "jest",
  -- INSTR() zwraca 29
  DBMS_OUTPUT.PUT_LINE('Wyszukiwanie drugiego ''jest''');
    v_pattern := 'jest';
    v_occurrence := 2;
    v_return := DBMS_LOB.INSTR(v_clob, v_pattern, v_offset, v_occurrence);
  DBMS_OUTPUT.PUT_LINE('v_return = ' || v_return);

  -- za pomoc INSTR() wyszukuje w v_clob pierwszego wystpienia sowa "Ksiyc"
  -- INSTR() zwraca 0, poniewa "Ksiyc" nie wystpuje w v_clob
  DBMS_OUTPUT.PUT_LINE('Wyszukiwanie sowa ''Ksiyc''');
    v_pattern := 'Ksiyc';
    v_occurrence := 1;
    v_return := DBMS_LOB.INSTR(v_clob, v_pattern, v_offset, v_occurrence);
  DBMS_OUTPUT.PUT_LINE('v_return = ' || v_return);

  -- zwalnia v_clob za pomoc FREETEMPORARY()
  DBMS_LOB.FREETEMPORARY(v_clob);
END instr_example;
/

CREATE OR REPLACE PROCEDURE copy_file_data_to_clob(
  p_clob_id INTEGER,
  p_directory VARCHAR2,
  p_file_name VARCHAR2
) AS
  v_file UTL_FILE.FILE_TYPE;
  v_chars_read INTEGER;
  v_dest_clob CLOB;
  v_amount INTEGER := 32767;
  v_char_buffer NVARCHAR2(32767);
BEGIN
  -- wstawia pusty obiekt CLOB
  INSERT INTO clob_content(
  id, clob_column
  ) VALUES (
    p_clob_id, EMPTY_CLOB()
  );

  -- pobiera lokalizator LOB obiektu CLOB
  SELECT clob_column
  INTO v_dest_clob
  FROM clob_content
  WHERE id = p_clob_id
  FOR UPDATE;
  
  -- otwiera plik do odczytu tekstu (do v_amount znakw w wierszu)
  v_file := UTL_FILE.FOPEN(p_directory, p_file_name, 'r', v_amount);
  
  -- kopiuje dane z pliku do v_dest_clob po jednej linii
  LOOP
    BEGIN
    -- odczytuje wiersz z pliku do v_char_buffer;
    -- GET_LINE() nie kopiuje znaku nowego wiersza 
    -- do v_char_buffer
    UTL_FILE.GET_LINE(v_file, v_char_buffer);
    v_chars_read := LENGTH(v_char_buffer);

    -- docza wiersz do v_dest_clob
    DBMS_LOB.WRITEAPPEND(v_dest_clob, v_chars_read, v_char_buffer);
    -- docza znak nowego wiersza do v_dest_clob, poniewa v_char_buffer go nie zawiera;
    -- kod ASCII znaku nowego wiersza to 10, wic CHR(10) zwraca ten znak
    DBMS_LOB.WRITEAPPEND(v_dest_clob, 1, CHR(10));
  EXCEPTION
    -- jeeli w pliku nie ma wicej danych, zakocz dziaanie
      WHEN NO_DATA_FOUND THEN
      EXIT;
    END;
  END LOOP;

  -- zamyka plik
  UTL_FILE.FCLOSE(v_file);

  DBMS_OUTPUT.PUT_LINE('Kopiowanie zakoczyo si powodzeniem.');
END copy_file_data_to_clob;
/

CREATE OR REPLACE PROCEDURE copy_file_data_to_blob(
  p_blob_id INTEGER,
  p_directory VARCHAR2,
  p_file_name VARCHAR2
) AS
 v_file UTL_FILE.FILE_TYPE;
  v_bytes_read INTEGER;
  v_dest_blob BLOB;
  v_amount INTEGER := 32767;
 v_binary_buffer RAW(32767);
BEGIN
  -- wstawia pusty obiekt BLOB
  INSERT INTO blob_content(
    id, blob_column
  ) VALUES (
    p_blob_id, EMPTY_BLOB()
  );

  -- pobiera lokalizator LOB obiektu BLOB
  SELECT blob_column
  INTO v_dest_blob
  FROM blob_content
  WHERE id = p_blob_id
  FOR UPDATE;

  -- otwiera plik w trybie odczytu bajtw (do v_amount bajtw w jednej operacji)
  v_file := UTL_FILE.FOPEN(p_directory, p_file_name, 'rb', v_amount);

  -- kopiuje dane z pliku do v_dest_blob
  LOOP
  BEGIN
    -- czyta dane binarne z pliku do v_binary_buffer
    UTL_FILE.GET_RAW(v_file, v_binary_buffer, v_amount);
    v_bytes_read := LENGTH(v_binary_buffer);
    -- docza v_binary_buffer do v_dest_blob
    DBMS_LOB.WRITEAPPEND(v_dest_blob, v_bytes_read/2,
    v_binary_buffer);
    EXCEPTION
    -- zakocz, jeeli nie ma wicej danych w pliku
      WHEN NO_DATA_FOUND THEN
        EXIT;
    END;
  END LOOP;

  -- zamyka plik
  UTL_FILE.FCLOSE(v_file);

  DBMS_OUTPUT.PUT_LINE('Kopiowanie zakoczone powodzeniem.');
END copy_file_data_to_blob;
/

CREATE OR REPLACE PROCEDURE copy_clob_data_to_file(
  p_clob_id INTEGER,
  p_directory VARCHAR2,
  p_file_name VARCHAR2
  ) AS
  v_src_clob CLOB;
  v_file UTL_FILE.FILE_TYPE;
  v_offset INTEGER := 1;
  v_amount INTEGER := 32767;
  v_char_buffer NVARCHAR2(32767);
BEGIN
  -- pobiera lokalizator LOB obiektu CLOB
  SELECT clob_column
  INTO v_src_clob
  FROM clob_content
  WHERE id = p_clob_id;

  -- otwiera plik w trybie zapisu tekstu (do v_amount znakw w operacji)
  v_file := UTL_FILE.FOPEN(p_directory, p_file_name, 'w', v_amount);

  -- kopiuje dane z v_src_clob do pliku
  LOOP
    BEGIN
    -- czyta znaki z v_src_clob do v_char_buffer
    DBMS_LOB.READ(v_src_clob, v_amount, v_offset, v_char_buffer);

    -- kopiuje znaki z v_char_buffer do pliku
    UTL_FILE.PUT(v_file, v_char_buffer);

    -- dodaje v_amount do v_offset
    v_offset := v_offset + v_amount;
    EXCEPTION
      -- zakocz, jeeli nie ma wicej danych w pliku
      WHEN NO_DATA_FOUND THEN
        EXIT;
    END;
    END LOOP;
  -- zapisz pozostae dane do pliku 
  UTL_FILE.FFLUSH(v_file);

  -- zamknij plik
  UTL_FILE.FCLOSE(v_file);

  DBMS_OUTPUT.PUT_LINE('Kopiowanie zakoczyo si powodzeniem.');
END copy_clob_data_to_file;
/


CREATE PROCEDURE copy_blob_data_to_file(
  p_blob_id INTEGER,
  p_directory VARCHAR2,
  p_file_name VARCHAR2
) AS
  v_src_blob BLOB;
  v_file UTL_FILE.FILE_TYPE;
  v_offset INTEGER := 1;
  v_amount INTEGER := 32767;
  v_binary_buffer RAW(32767);
BEGIN
  -- pobiera lokalizator LOB obiektu BLOB
  SELECT blob_column
  INTO v_src_blob
  FROM blob_content
  WHERE id = p_blob_id;

  -- otwiera plik do zapisu bajtw (do v_amount bajtw w operacji)
  v_file := UTL_FILE.FOPEN(p_directory, p_file_name, 'wb', v_amount);
  
  -- kopiuje dane z v_src_blob do pliku
  LOOP
    BEGIN
    -- czyta dane binarne z v_src_blob do v_binary_buffer
    DBMS_LOB.READ(v_src_blob, v_amount, v_offset, v_binary_buffer);

    -- kopiuje dane binarne z v_binary_buffer do pliku
    UTL_FILE.PUT_RAW(v_file, v_binary_buffer);

    -- dodaje v_amount do v_offset
    v_offset := v_offset + v_amount;
    EXCEPTION
      -- jeeli nie ma wicej danych, wychodzi z ptli
      WHEN NO_DATA_FOUND THEN
        EXIT;
    END;
  END LOOP;

  -- zapisuje pozostae dane do pliku
  UTL_FILE.FFLUSH(v_file);

  -- zamyka plik
  UTL_FILE.FCLOSE(v_file);

  DBMS_OUTPUT.PUT_LINE('Kopiowanie zakoczone powodzeniem.');
END copy_blob_data_to_file;
/

CREATE PROCEDURE copy_bfile_data_to_clob(
  p_bfile_id INTEGER,
  p_clob_id INTEGER
) AS
  v_src_bfile BFILE;
  v_directory VARCHAR2(200);
  v_filename VARCHAR2(200);
  v_length INTEGER;
  v_dest_clob CLOB;
  v_amount INTEGER := DBMS_LOB.LOBMAXSIZE;
  v_dest_offset INTEGER := 1;
  v_src_offset INTEGER := 1;
  v_src_csid INTEGER := DBMS_LOB.DEFAULT_CSID;
  v_lang_context INTEGER := DBMS_LOB.DEFAULT_LANG_CTX;
  v_warning INTEGER;
BEGIN
  -- pobiera lokalizator obiektu BFILE
  SELECT bfile_column
  INTO v_src_bfile
  FROM bfile_content
  WHERE id = p_bfile_id;

  -- za pomoc FILEEXISTS() sprawdza, czy plik istnieje
  -- (FILEEXISTS() zwraca 1, jeeli plik istnieje)
  IF (DBMS_LOB.FILEEXISTS(v_src_bfile) = 1) THEN
    -- otwiera plik za pomoc OPEN() 
    DBMS_LOB.OPEN(v_src_bfile);

    -- pobiera nazw pliku i  katalogu za pomoc FILEGETNAME()
    DBMS_LOB.FILEGETNAME(v_src_bfile, v_directory, v_filename);
    DBMS_OUTPUT.PUT_LINE('Directory = ' || v_directory);
    DBMS_OUTPUT.PUT_LINE('Filename = ' || v_filename);

    -- wstawia pusty obiekt CLOB
    INSERT INTO clob_content(
      id, clob_column
    ) VALUES (
      p_clob_id, EMPTY_CLOB()

    );
    -- pobiera lokalizator LOB obiektu CLOB (w celu modyfikacji obiektu)
    SELECT clob_column
    INTO v_dest_clob
    FROM clob_content
    WHERE id = p_clob_id
    FOR UPDATE;

    -- za pomoc LOADCLOBFROMFILE() pobiera z v_src_bfile maksymalnie v_amount znakw 
    -- i zapisuje je w v_dest_clob (rozpoczyna w przesuniciach 1 w v_src_bfile i v_dest_clob
    DBMS_LOB.LOADCLOBFROMFILE(
      v_dest_clob, v_src_bfile,
      v_amount, v_dest_offset, v_src_offset,
      v_src_csid, v_lang_context, v_warning
    );

    -- sprawdza w v_warning, czy wystpi niekonwertowalny znak 
    IF (v_warning = DBMS_LOB.WARN_INCONVERTIBLE_CHAR) THEN
      DBMS_OUTPUT.PUT_LINE('Uwaga! Konwersja znaku nie powioda si.');
    END IF;

    -- zamyka v_src_bfile za pomoc CLOSE()
    DBMS_LOB.CLOSE(v_src_bfile);
    DBMS_OUTPUT.PUT_LINE('Kopiowanie zakoczone powodzeniem.');
  ELSE
    DBMS_OUTPUT.PUT_LINE('Plik nie istnieje.');
  END IF;
END copy_bfile_data_to_clob;
/

CREATE PROCEDURE copy_bfile_data_to_blob(
  p_bfile_id INTEGER,
  p_blob_id INTEGER
) AS
  v_src_bfile BFILE;
  v_directory VARCHAR2(200);
  v_filename VARCHAR2(200);
  v_length INTEGER;
  v_dest_blob BLOB;
  v_amount INTEGER := DBMS_LOB.LOBMAXSIZE;
  v_dest_offset INTEGER := 1;
  v_src_offset INTEGER := 1;
BEGIN
  -- pobiera lokalizator obiektu BFILE
  SELECT bfile_column
  INTO v_src_bfile
  FROM bfile_content
  WHERE id = p_bfile_id;

  -- za pomoc FILEEXISTS() sprawdza, czy plik istnieje
  -- (FILEEXISTS() zwraca 1, jeeli plik istnieje)
  IF (DBMS_LOB.FILEEXISTS(v_src_bfile) = 1) THEN
    -- otwiera plik za pomoc OPEN() 
    DBMS_LOB.OPEN(v_src_bfile);

    -- pobiera nazw pliku i  katalogu za pomoc FILEGETNAME()
    DBMS_LOB.FILEGETNAME(v_src_bfile, v_directory, v_filename);
    DBMS_OUTPUT.PUT_LINE('Katalog = ' || v_directory);
    DBMS_OUTPUT.PUT_LINE('Nazwa pliku = ' || v_filename);

    -- wstawia pusty obiekt BLOB
    INSERT INTO blob_content(
      id, blob_column
    ) VALUES (
      p_blob_id, EMPTY_BLOB()
    );

    -- pobiera lokalizator LOB nowego obiektu BLOB (w celu modyfikacji obiektu)
    SELECT blob_column
    INTO v_dest_blob
    FROM blob_content
    WHERE id = p_blob_id
    FOR UPDATE;

    -- za pomoc LOADBLOBFROMFILE() pobiera z v_src_bfile maksymalnie v_amount bajtw 
    -- i zapisuje je w v_dest_blob (rozpoczyna w przesuniciach 1 w v_src_bfile i v_dest_blob)
    DBMS_LOB.LOADBLOBFROMFILE(
      v_dest_blob, v_src_bfile,
      v_amount, v_dest_offset, v_src_offset
    );

    -- zamyka v_src_bfile za pomoc CLOSE()
    DBMS_LOB.CLOSE(v_src_bfile);
    DBMS_OUTPUT.PUT_LINE('Kopiowanie zakoczone powodzeniem.');
  ELSE
    DBMS_OUTPUT.PUT_LINE('Plik nie istnieje.');
  END IF;
END copy_bfile_data_to_blob;
/


-- zatwierdza transakcj
COMMIT;
