/* ================================================================
||   Program: stack_trace_management.sql
||   Data:       2013-07-25
||   Książka:    Oracle Database 12c. Programowanie w języku PL/SQL
||   Rozdział:   7
||   Autor:  Michael McLaughlin
|| ----------------------------------------------------------------
||   Zawartość:
||   ---------
||   Ten skrypt zawiera programy związane z zarządzaniem 
||   śladem stosu.
|| ================================================================*/


CREATE OR REPLACE PROCEDURE handle_errors
( pv_object_name         IN  VARCHAR2
, pv_module_name         IN  VARCHAR2 := NULL
, pv_table_name          IN  VARCHAR2 := NULL
, pv_error_code          IN  NUMBER   := NULL
, pv_error_message       IN  VARCHAR2 := NULL
, pv_user_error_message  IN  VARCHAR2 := NULL ) IS

  /* Deklaracja lokalnego wyjątku. */
  lv_error  EXCEPTION;

  -- Definicja typu kolekcji i inicjowanie zmiennej tego typu.
  TYPE error_stack IS TABLE OF VARCHAR2(100);
  lv_errors        ERROR_STACK := error_stack();

  /* Definicja lokalnej funkcji get_object_type. */
  FUNCTION get_object_type
  ( pv_object_name  IN  VARCHAR2 ) RETURN VARCHAR2 IS
    /* Lokalna zwracana zmienna. */
    lv_return_type  VARCHAR2(12) := 'Niezidentyfikowany';
  BEGIN
    FOR i IN ( SELECT object_type FROM user_objects
               WHERE  object_name = pv_object_name ) LOOP
      lv_return_type := i.object_type;
    END LOOP;
    RETURN lv_return_type;
  END get_object_type;
BEGIN
  -- Przydzielanie pamięci i przypisywanie wartości do kolekcji.
  lv_errors.EXTEND;
  lv_errors(lv_errors.COUNT) :=
    get_object_type(pv_object_name)||' ['||pv_object_name||']';

  -- Podstawianie argumentów za wartości domyślne.
  IF pv_module_name IS NOT NULL THEN
    lv_errors.EXTEND;
    lv_errors(lv_errors.COUNT) := 'Nazwa modułu: ['||pv_module_name||']';
  END IF;
  IF pv_table_name IS NOT NULL THEN
    lv_errors.EXTEND;
    lv_errors(lv_errors.COUNT) := 'Nazwa tabeli: ['||pv_table_name||']';
  END IF;
  IF pv_error_code IS NOT NULL THEN
    lv_errors.EXTEND;
    lv_errors(lv_errors.COUNT) := 'Wartość SQLCODE: ['||pv_error_code||']';
  END IF;
  IF pv_error_message IS NOT NULL THEN
    lv_errors.EXTEND;
    lv_errors(lv_errors.COUNT) := 'Wartość SQLERRM: ['||pv_error_message||']';
  END IF;
  IF pv_user_error_message IS NOT NULL THEN
    lv_errors.EXTEND;
    lv_errors(lv_errors.COUNT) := pv_user_error_message;
  END IF;

  lv_errors.EXTEND;
  lv_errors(lv_errors.COUNT) := '----------------------------------------';
  RAISE lv_error;
EXCEPTION
  WHEN lv_error THEN
    FOR i IN 1..lv_errors.COUNT LOOP
      dbms_output.put_line(lv_errors(i));
    END LOOP;
    RETURN;
END;
/

CREATE OR REPLACE PROCEDURE pear IS
  /* Deklaracje dwóch zmiennych. */
  lv_one_character  VARCHAR2(1);
  lv_two_character  VARCHAR2(2) := 'AB';
BEGIN
  lv_one_character := lv_two_character; 
END pear;
/

SHOW ERRORS

CREATE OR REPLACE PROCEDURE orange IS
BEGIN
  pear();
END orange;
/

SHOW ERRORS

CREATE OR REPLACE PROCEDURE apple IS
BEGIN
  orange();
END apple;
/

SHOW ERRORS

BEGIN
  apple;
EXCEPTION
  WHEN others THEN
    FOR i IN REVERSE 1..utl_call_stack.backtrace_depth LOOP
      /* Wykrywanie bloku anonimowego. */
      IF utl_call_stack.backtrace_unit(i) IS NULL THEN
        /* utl_call_stack nie wyświetla błędu; należy ręcznie ustawić komunikat. */
        dbms_output.put_line(
       	  'ORA-06512: w Bloku Anonimowym, wiersz '||
           utl_call_stack.backtrace_line(i));
      ELSE
         /* utl_call_stack nie wyświetla błędu; należy ręcznie ustawić komunikat. */
        dbms_output.put_line(
          'ORA-06512: w '||utl_call_stack.backtrace_unit(i)||
          ', line '||utl_call_stack.backtrace_line(i));
      END IF;

      /* Poziomy ścieżki wstecznej i stosu błędów nie są od siebie zależne. 
	     Poziom wywołań może być (i zwykle jest) wyższy od poziomu stosu błędów. */
      IF i = utl_call_stack.error_depth THEN
        dbms_output.put_line(
        	'ORA-'||LPAD(utl_call_stack.error_number(i),5,0)
             ||' '||utl_call_stack.error_msg(i));
      END IF;
    END LOOP;
END;
/

DECLARE
  lv_length   NUMBER;
  lv_counter  NUMBER := 0;
  lv_begin    NUMBER := 1;
  lv_end      NUMBER;
  lv_index    NUMBER := 0;
  lv_trace    VARCHAR2(2000);
BEGIN
  apple;
EXCEPTION
  WHEN others THEN
    FOR i IN REVERSE 1..utl_call_stack.backtrace_depth LOOP
      /* Poziomy ścieżki wstecznej i stosu błędów nie są od siebie zależne. 
	     Poziom wywołań może być (i zwykle jest) wyższy od poziomu stosu błędów.    */
      IF i = utl_call_stack.error_depth THEN
        /* Pobieranie śladu stosu. */
        lv_trace := dbms_utility.format_error_backtrace;

        /* Zliczanie znaków nowego wiersza (kod ASCII 10). */
        lv_length := REGEXP_COUNT(lv_trace,CHR(10),1);

        /* Wczytywanie stosu w celu usunięcia znaków nowego wiersza. */
        WHILE (lv_counter < lv_length) LOOP
          /* Zwiększanie wartości licznika na początku. */
          lv_counter := lv_counter + 1;

          /* Pobieranie następnego znaku nowego wiersza. */
          lv_end := REGEXP_INSTR(lv_trace,CHR(10),lv_begin,1);

          /* Pobieranie pierwszego podłańcucha ze śladu stosu. */
          dbms_output.put_line(SUBSTR(lv_trace,lv_begin,lv_end - lv_begin));

          /* Ustawianie początku podłańcucha na pozostałą część łańcucha. */
          lv_begin := lv_end + 1;

        END LOOP;

        /* Wyświetlanie pierwotnego komunikatu o błędzie. */
        dbms_output.put_line(
            'ORA-'||LPAD(utl_call_stack.error_number(i),5,0)
             ||' '||utl_call_stack.error_msg(i));
      END IF;
    END LOOP;
END;
/

DECLARE
  lv_length   NUMBER;
  lv_counter  NUMBER := 0;
  lv_begin    NUMBER := 1;
  lv_end      NUMBER;
  lv_index    NUMBER := 0;
  lv_trace    VARCHAR2(2000);
  lv_break    VARCHAR2(6) := '<br />';
BEGIN
  apple;
EXCEPTION
  WHEN others THEN
    FOR i IN REVERSE 1..utl_call_stack.backtrace_depth LOOP
      /* Poziomy ścieżki wstecznej i stosu błędów nie są od siebie zależne. 
	     Poziom wywołań może być (i zwykle jest) wyższy od poziomu stosu błędów.    */
      IF i = utl_call_stack.error_depth THEN
        /* Pobieranie śladu stosu. */
        lv_trace := dbms_utility.format_error_backtrace;

        /* Zliczanie znaków nowego wiersza (kod ASCII 10). */
        lv_length := REGEXP_COUNT(lv_trace,CHR(10),1);

        /* Przechodzenie po stosie w celu usunięcia znaków nowego wiersza. */
        WHILE (lv_counter < lv_length) LOOP
          /* Zwiększanie licznika na początku. */
          lv_counter := lv_counter + 1;

          /* Znajdowanie następnego znaku nowego wiersza. */
          lv_end := REGEXP_INSTR(lv_trace,CHR(10),lv_begin,1);

          /* Pobieranie pierwszego podłańcucha ze śladu stosu. */
          lv_trace := REGEXP_REPLACE(lv_trace,CHR(10),lv_break,lv_end,1);
          lv_end := lv_end + LENGTH(lv_break);
          dbms_output.put_line(
            SUBSTR(lv_trace,lv_begin,lv_end - lv_begin));

          /* Ustawianie początku podłańcucha na pozostałą część łańcucha. */
          lv_begin := lv_end;
        END LOOP;
      END IF;
    END LOOP;
END;
/