/* ================================================================
||   Program: load_blob_from_file.sql
||   Data:       2013-07-25
||   Książka:    Oracle Database 12c. Programowanie w języku PL/SQL
||   Rozdział:   10
||   Autor:  Michael McLaughlin
|| ----------------------------------------------------------------
||   Zawartość:
||   ---------
||   UWAGI:
||
||   Plik HarryPotter1.png należy umieścić w katalogu /tmp 
||   (Linux lub Unix) albo C:\WINDOWS\TEMP (Windows).
||   Inna możliwość to zdefiniowanie katalogu wirtualnego GENERIC powiązanego
||   z wybranym katalogiem. Upewnij się jednak, że baza Oracle
||   ma uprawnienia do odczytu danych z tego katalogu.
||
||   Ten skrypt przedstawia procedurę wczytującą duży plik tekstowy
||   do obiektu typu BLOB.
||
||   TRYB DEBUGOWANIA:
||   ==========
||   Zmień ustawienia sesji za pomocą poniższej instrukcji.
||
||   ALTER SESSION SET PLSQL_CCFLAGS = 'debug:1';
|| ================================================================*/

-- Po zakończeniu debugowania umieść w komentarzu.
SET ECHO ON
SET FEEDBACK ON
SET PAGESIZE 49999
SET SERVEROUTPUT ON SIZE 1000000

-- Procedura składowana wczytująca dane do obiektu typu BLOB.
CREATE OR REPLACE PROCEDURE load_blob_from_file
( src_file_name     IN VARCHAR2
, table_name        IN VARCHAR2
, column_name       IN VARCHAR2
, primary_key_name  IN VARCHAR2
, primary_key_value IN VARCHAR2 ) IS

  -- Definicje zmiennych lokalnych na potrzeby procedury DBMS_LOB.LOADCLOBFROMFILE.
  des_blob      BLOB;
  src_blob      BFILE := BFILENAME('GENERIC',src_file_name);
  des_offset    NUMBER := 1;
  src_offset    NUMBER := 1;
  
  -- Zmienna z wielkością obiektu BLOB.
  src_blob_size NUMBER;
  
  -- Definicja zmiennej lokalnej na potrzeby instrukcji NDS.
  stmt VARCHAR2(2000);

BEGIN

  -- Trzeba otworzyć plik źródłowy.
  IF dbms_lob.fileexists(src_blob) = 1 AND NOT dbms_lob.isopen(src_blob) = 1 THEN
    src_blob_size := dbms_lob.getlength(src_blob);
    dbms_lob.open(src_blob,DBMS_LOB.LOB_READONLY);
  END IF;
  
  -- Przypisywanie dynamicznego łańcucha znaków do instrukcji.
  stmt := 'UPDATE '||table_name||' '
       || 'SET    '||column_name||' = empty_blob() '
       || 'WHERE  '||primary_key_name||' = '||''''||primary_key_value||''' '
       || 'RETURNING '||column_name||' INTO :locator';

  -- Wykonywanie dynamicznej instrukcji.       
  EXECUTE IMMEDIATE stmt USING OUT des_blob;
 
  -- Odczyt pliku i zapis danych do obiektu BLOB.
  dbms_lob.loadblobfromfile( dest_lob     => des_blob
                           , src_bfile    => src_blob
                           , amount       => dbms_lob.getlength(src_blob)
                           , dest_offset  => des_offset
                           , src_offset   => src_offset );

  -- Zamykanie otwartego pliku źródłowego.
  dbms_lob.close(src_blob);

  -- Zatwierdzanie zapisu i warunkowe potwierdzanie powodzenia.
  IF src_blob_size = dbms_lob.getlength(des_blob) THEN
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Powodzenie!');
    $END
    COMMIT;
  ELSE
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Niepowodzenie.');
    $END
    RAISE dbms_lob.operation_failed;
  END IF;
  
END load_blob_from_file;
/

SHOW ERRORS

-- Formatowanie kolumn w celu wyświetlenia danych.
COL item_id    FORMAT 9999
COL item_title FORMAT A50
COL size       FORMAT 9,999,990

-- Sprawdzanie danych przed wczytaniem.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_blob) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_blob) > 0;

-- Wstawianie opisu we wszystkich pasujących wierszach.
BEGIN
  FOR i IN (SELECT item_id
            FROM   item
            WHERE  item_title = 'Harry Potter and the Sorcerer''s Stone'
            AND    item_type IN
             (SELECT common_lookup_id
              FROM   common_lookup
              WHERE  common_lookup_table = 'ITEM'
              AND    common_lookup_column = 'ITEM_TYPE'
              AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'))) LOOP
    -- Wywołanie procedury dla pasujących wierszy.
    load_blob_from_file( src_file_name     => 'HarryPotter1.png'
                       , table_name        => 'ITEM'
                       , column_name       => 'ITEM_BLOB'
                       , primary_key_name  => 'ITEM_ID'
                       , primary_key_value => TO_CHAR(i.item_id) );
  END LOOP;
END;
/

-- Sprawdzanie danych po wczytaniu.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_blob) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_blob) > 0;

-- Wstawianie rekordu tylko wtedy, gdy go nie znaleziono (skrypt można więc uruchamiać wielokrotnie).
DECLARE
  -- Deklaracja numeru wiersza.
  row NUMBER :=0;

  -- Deklaracja kursora wyszukującego obiekty BLOB.
  CURSOR c IS
    SELECT item_id
    FROM   item
    WHERE  item_title = 'The Blob - Criterion Collection'
    AND    item_type IN
            (SELECT common_lookup_id
             FROM   common_lookup
             WHERE  common_lookup_table = 'ITEM'
             AND    common_lookup_column = 'ITEM_TYPE'
             AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'));
BEGIN
  OPEN c;
  LOOP
    FETCH c INTO row;
    IF c%NOTFOUND AND c%ROWCOUNT = 1 THEN
      INSERT INTO item VALUES
      ( item_s1.nextval
      ,'ASIN: B00004W3HE'
      ,(SELECT   common_lookup_id
        FROM     common_lookup
        WHERE    common_lookup_type = 'XBOX')
      ,'The Blob - Criterion Collection'
      ,''
      , empty_clob()
      , empty_blob()
      , NULL
      ,'NR'
      ,'MPAA'
      ,'14-NOV-2000'
      , 3, SYSDATE, 3, SYSDATE);
      EXIT;
    ELSE
      EXIT;
    END IF;
  END LOOP;
END;
/

-- Wstawianie opisu we wszystkich pasujących wierszach.
BEGIN
  FOR i IN (SELECT item_id
            FROM   item
            WHERE  item_title = 'The Blob - Criterion Collection'
            AND    item_type IN
             (SELECT common_lookup_id
              FROM   common_lookup
              WHERE  common_lookup_table = 'ITEM'
              AND    common_lookup_column = 'ITEM_TYPE'
              AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'))) LOOP
    -- Wywołanie procedury dla pasujących wierszy.
    load_blob_from_file( src_file_name     => 'TheBlob.pdf'
                       , table_name        => 'ITEM'
                       , column_name       => 'ITEM_BLOB'
                       , primary_key_name  => 'ITEM_ID'
                       , primary_key_value => TO_CHAR(i.item_id) );
  END LOOP;
END;
/

