/* ================================================================
||   Program Name: avatar.sql
||   Data:       2013-07-25
||   Książka:    Oracle Database 12c. Programowanie w języku PL/SQL
||   Rozdział:   8
||   Autor:  Michael McLaughlin
|| ----------------------------------------------------------------
||   Zawartość:
||   ---------
||   Ten skrypt zawiera różne przykładowe procedury z tego rozdziału.
|| ================================================================*/

SET SERVEROUTPUT ON SIZE UNLIMITED

-- Usuwanie tabeli avatar.
DROP TABLE avatar CASCADE CONSTRAINTS;

-- Tworzenie tabeli avatar.
CREATE TABLE avatar
( avatar_id    NUMBER GENERATED AS IDENTITY
, avatar_name  VARCHAR2(30));

-- Tworzenie funkcji add_avatar.
CREATE OR REPLACE FUNCTION add_avatar
( pv_avatar_name  VARCHAR2 ) RETURN BOOLEAN IS
  /* Ustawianie funkcji tak, aby działała we własnym zasięgu transakcji. */
  PRAGMA AUTONOMOUS_TRANSACTION;
  /* Ustawianie domyślnej zwracanej wartości. */
  lv_retval  BOOLEAN := FALSE;
BEGIN
  /* Wstawianie wiersza do tabeli avatar. */
  INSERT INTO avatar (avatar_name)
  VALUES (pv_avatar_name);
  /* Zapisywanie zmian we własnym zasięgu transakcji. */
  COMMIT;
  /* Ustawianie zwracanej wartości na true. */
  lv_retval := TRUE;
  RETURN lv_retval;
END;
/

SHOW ERRORS

-- Tworzenie bloku anonimowego.
DECLARE
  /* Deklaracja zmiennej lokalnej. */
  lv_avatar  VARCHAR2(30);
  /* Deklaracja lokalnego kursora. */
  CURSOR capture_result
  (cv_avatar_name  VARCHAR2) IS
    SELECT   avatar_name
    FROM     avatar
    WHERE    avatar_name = cv_avatar_name;
BEGIN
  IF add_avatar('Earthbender') THEN
    dbms_output.put_line('Rekord wstawiono');
    ROLLBACK;
  ELSE
    dbms_output.put_line('Rekordu nie wstawiono');
  END IF;
  OPEN capture_result('Earthbender');
  FETCH capture_result INTO lv_avatar;
  CLOSE capture_result;
  dbms_output.put_line('Wartość ['||lv_avatar||']');
END;
/

-- Tworzenie funkcji avatar.
CREATE OR REPLACE FUNCTION add_avatar
( pv_avatar_name  VARCHAR2 ) RETURN NUMBER IS
  /* Ustawianie funkcji tak, aby działała we własnym zasięgu transakcji. */
  PRAGMA AUTONOMOUS_TRANSACTION;
  /* Ustawianie domyślnej zwracanej wartości. */
  lv_retval  NUMBER := 0;
BEGIN
  /* Wstawianie wiersza do tabeli avatar. */
  INSERT INTO avatar (avatar_name)
  VALUES (pv_avatar_name);
  /* Zapisywanie zmian we własnym zasięgu transakcji. */
  COMMIT;
  /* Ustawianie zwracanej wartości na true. */
  lv_retval := 1;
  RETURN lv_retval;
END;
/

SHOW ERRORS

-- Pobieranie wyników z funkcji avatar.
SELECT   CASE
           WHEN add_avatar('Firebender') <> 0 THEN 'Powodzenie' ELSE 'Niepowodzenie'
         END AS Autonomous
FROM     dual;

SELECT   *
FROM     avatar;

-- Usuwanie tabeli avatar i sekwencji avatar_s.
DROP TABLE avatar CASCADE CONSTRAINTS;
DROP SEQUENCE avatar_s;

-- Tworzenie tabeli avatar.
CREATE TABLE avatar
( avatar_id    NUMBER CONSTRAINT avatar_pk PRIMARY KEY
, avatar_name  VARCHAR2(30));

-- Usuwanie tabeli episode.
DROP TABLE episode CASCADE CONSTRAINTS;
DROP SEQUENCE episode_s;

-- Tworzenie tabeli episode.
CREATE TABLE episode
( episode_id   NUMBER CONSTRAINT episode_pk PRIMARY KEY
, avatar_id    NUMBER
, episode_name  VARCHAR2(30)
, CONSTRAINT episode_fk1 FOREIGN KEY(avatar_id)
  REFERENCES avatar(avatar_id));

-- Tworzenie procedury adding_avatar.
CREATE OR REPLACE PROCEDURE adding_avatar
( pv_avatar_name   VARCHAR2
, pv_episode_name  VARCHAR2 ) IS

  /* Deklaracja zmiennej lokalnej do zarządzania kluczami sztucznymi. */
  lv_avatar_id  NUMBER;
  lv_episode_id NUMBER;
BEGIN
  /* Ustawianie punktu zapisu. */
  SAVEPOINT all_or_none;

  /* Zapisywanie sekwencji w zmiennej lokalnej. */
  SELECT   avatar_s.NEXTVAL
  INTO     lv_avatar_id
  FROM     dual;

  /* Wstawianie wiersza do tabeli avatar. */
  INSERT INTO avatar (avatar_id, avatar_name)
  VALUES (lv_avatar_id, pv_avatar_name);

  /* Pobieranie wartości sekwencji do zmiennej lokalnej. */
  SELECT   episode_s.NEXTVAL
  INTO     lv_episode_id
  FROM     dual;

  /* Wstawianie wiersza do tabeli avatar. */
  INSERT INTO episode (episode_id, avatar_id, episode_name)
  VALUES (lv_avatar_id, pv_episode_name);

  /* Zapisywanie zmian we własnym zasięgu transakcji. */
  COMMIT;
EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK TO all_or_none;
END;
/

-- Tworzenie bloku anonimowego.
BEGIN
  adding_avatar('Airbender','Episode 1');
END;
/

-- Usuwanie tabeli avatar.
DROP TABLE avatar CASCADE CONSTRAINTS;

-- Tworzenie tabeli avatar.
CREATE TABLE avatar
( avatar_id    NUMBER GENERATED AS IDENTITY
               CONSTRAINT avatar_pk PRIMARY KEY
, avatar_name  VARCHAR2(30));

-- Usuwanie tabeli episode.
DROP TABLE episode;

-- Tworzenie zależnej tabeli episode.
CREATE TABLE episode
( episode_id   NUMBER GENERATED AS IDENTITY
               CONSTRAINT episode_pk PRIMARY KEY
, avatar_id    NUMBER CONSTRAINT episode_nn1 NOT NULL
, episode_name  VARCHAR2(30)
, CONSTRAINT episode_fk1 FOREIGN KEY(avatar_id)
  REFERENCES avatar(avatar_id));

-- Tworzenie procedury adding_avatar.
CREATE OR REPLACE PROCEDURE adding_avatar
( pv_avatar_name   VARCHAR2
, pv_episode_name  VARCHAR2 ) IS

  /* Deklaracja zmiennej lokalnej do zarządzania kolumną
     identyfikacyjną z kluczem sztucznym. */
  lv_avatar_id  NUMBER;
BEGIN
  /* Ustawianie punktu zapisu. */
  SAVEPOINT all_or_none;

  /* Wstawianie wiersza do tabeli avatar. */
  INSERT INTO avatar (avatar_name)
  VALUES (pv_avatar_name)
  RETURNING avatar_id INTO lv_avatar_id;

  /* Wstawianie wiersza do tabeli avatar. */
  INSERT INTO episode (avatar_id, episode_name)
  VALUES (lv_avatar_id, pv_episode_name);

  /* Zapisywanie zmian we własnym zasięgu transakcji. */
  COMMIT;
EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK TO all_or_none;
END;
/

-- Tworzenie bloku anonimowego.
BEGIN
  adding_avatar('Airbender','Episode 1');
END;
/

-- Tworzenie procedury adding_avatar.
CREATE OR REPLACE PROCEDURE adding_avatar
( pv_avatar_name   IN     VARCHAR2
, pv_episode_name  IN     VARCHAR2
, pv_completion       OUT BOOLEAN) IS

  /* Deklaracja zmiennej lokalnej do zarządzania kolumną
     identyfikacyjną z kluczem sztucznym. */
  lv_avatar_id  NUMBER;
BEGIN
  /* Ustawianie zmiennej pv_completion. */
  pv_completion := FALSE;

  /* Ustawianie punktu zapisu. */
  SAVEPOINT all_or_none;

  /* Wstawianie wiersza do tabeli avatar. */
  INSERT INTO avatar (avatar_name)
  VALUES (pv_avatar_name)
  RETURNING avatar_id INTO lv_avatar_id;

  /* Wstawianie wiersza do tabeli episode. */
  INSERT INTO episode (avatar_id, episode_name)
  VALUES (lv_avatar_id, pv_episode_name);

  /* Zapisywanie zmian we własnym zasięgu transakcji. */
  COMMIT;

  /* Ustawianie zmiennej pv_completion. */
  pv_completion := TRUE;
EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK TO all_or_none;
END;
/

-- Tworzenie bloku anonimowego.
DECLARE
  /* Ustawianie zmiennej kontrolnej. */
  lv_completion_variable  BOOLEAN;
BEGIN
  adding_avatar('Airbender','Episode 1',lv_completion_variable);
  IF lv_completion_variable THEN
    dbms_output.put_line('Ukończono z powodzeniem.');
  ELSE
    dbms_output.put_line('Nie wykonano żadnej operacji.');
  END IF;
END;
/
