/* ================================================================
||   Program:    script_collection.sql
||   Data:       2013-12-02
||   Książka:    Oracle Database 12c. Programowanie w języku PL/SQL
||   Rozdział:   2
||   Autor:  Michael McLaughlin
|| ----------------------------------------------------------------
||   Zawartość:
||   ---------
||   Ten skrypt tworzy obiekty używane w kompletnym przykładzie ilustrującym
||   stosowanie kolekcji.
|| ================================================================*/

-- Umieść w komentarzu po zakończeniu debugowania skryptu.
SET ECHO ON
SET FEEDBACK ON
SET PAGESIZE 49999
SET SERVEROUTPUT ON SIZE UNLIMITED


-- Prosty skrypt typu "witaj, świecie".
BEGIN
  dbms_output.put_line('Witaj, świecie.');
END;
/

-- Blok anonimowy ze zmienną podstawianą.
BEGIN
  dbms_output.put_line('['||'&input'||']');
END;
/

-- Deklaracja zmiennej środowiska SQL*Plus.
VARIABLE bind_variable VARCHAR2(20)

-- Blok anonimowy ustawiający zmienną powiązaną.
BEGIN
  :bind_variable := 'Witaj, Krypton.';
  dbms_output.put_line('['||:bind_variable||']');
END;
/

-- ------------------------------------------------------------
--   Sekcja deklaracji
-- ------------------------------------------------------------

-- Blok anonimowy zarządzający wartością zmiennej powiązanej.
DECLARE
  lv_input VARCHAR2(30);
BEGIN
  lv_input := :bind_variable;
  dbms_output.put_line('['||lv_input||']');
END;
/

-- ------------------------------------------------------------
--   Sekcja deklaracji
-- ------------------------------------------------------------

-- Ten blok anonimowy zgłasza błąd po otrzymaniu łańcucha znaków, ponieważ
-- brakuje tu niezbędnych apostrofów. 
BEGIN
  dbms_output.put_line('['||&input||']');
EXCEPTION
  WHEN OTHERS THEN
  dbms_output.put_line(SQLERRM);
END;
/

-- ------------------------------------------------------------
--   Działanie zmiennych w blokach
-- ------------------------------------------------------------
--   Bloki anonimowe
-- ------------------------------------------------------------

-- Blok anonimowy bez deklaracji wartości zmiennej.
DECLARE
  lv_sample  NUMBER;
BEGIN
  dbms_output.put_line('Wartość to ['||lv_sample||']');
END;
/

-- Blok anonimowy z deklaracją stałej.
DECLARE
  lv_sample CONSTANT NUMBER := 1;
BEGIN
  dbms_output.put_line('Wartość to ['||lv_sample||']');
END;
/

-- Blok anonimowy ze zmienną podstawianą w sekcji deklaracji.
DECLARE
  lv_input VARCHAR2(10) := '&input';
BEGIN
  dbms_output.put_line('['||lv_input||']');
EXCEPTION
  WHEN OTHERS THEN
  dbms_output.put_line(SQLERRM);
END;
/

-- Blok anonimowy z podstawianiem w sekcji wykonawczej.
DECLARE
  lv_input VARCHAR2(10);
BEGIN
  lv_input := '&input';
  dbms_output.put_line('['||lv_input||']');
EXCEPTION
WHEN OTHERS THEN
  dbms_output.put_line(SQLERRM);
  END;
/

-- Blok anonimowy z rzutowaniem w dół z liczby rzeczywistej na liczbę całkowitą.
DECLARE
  lv_input INTEGER;
BEGIN
  lv_input := 4.67;
  dbms_output.put_line('['||lv_input||']');
EXCEPTION
  WHEN OTHERS THEN
  dbms_output.put_line('['||SQLERRM||']');
END;
/

-- ------------------------------------------------------------
--   Zagnieżdżony blok anonimowy
-- ------------------------------------------------------------

-- Zagnieżdżone bloki anonimowe.
DECLARE
  -- Deklaracja zmiennej lokalnej.
  lv_input VARCHAR2(30) DEFAULT 'ZEWNĘTRZNA';
BEGIN
  -- Wyświetlanie wartości przed blokiem wewnętrznym.
  dbms_output.put_line('Blok zewnętrzny ['||lv_input||']');

  -- Blok zagnieżdżony.
  BEGIN
    -- Wyświetlanie wartości przed przypisaniem.
    dbms_output.put_line('Blok wewnętrzny ['||lv_input||']');

    -- Przypisywanie nowej wartości do zmiennej.
    lv_input := 'WEWNĘTRZNA';

    -- Wyświetlanie wartości po przypisaniu.
    dbms_output.put_line('Blok wewnętrzny ['||lv_input||']');
  END;

  -- Wyświetlanie wartości po bloku zagnieżdżonym.
  dbms_output.put_line('Blok zewnętrzny ['||lv_input||']');
EXCEPTION
  WHEN OTHERS THEN
  dbms_output.put_line('Wyjątek ['||SQLERRM||']');
END;
/

-- Blok autonomiczny z zagnieżdżonym blokiem anonimowym.
DECLARE
  -- Deklaracja zmiennej lokalnej.
  lv_outer VARCHAR2(30) DEFAULT 'ZEWNĘTRZNA';
  lv_active VARCHAR2(30) DEFAULT 'ZEWNĘTRZNA';

BEGIN
  -- Wyświetlanie wartości przed blokiem wewnętrznym.
  dbms_output.put_line('Zewnętrzny ['||lv_outer||']['||lv_active||']');

  -- Blok zagnieżdżony.
  DECLARE
    -- Deklaracja zmiennej lokalnej.
    lv_active VARCHAR2(30) DEFAULT 'WEWNĘTRZNA';

  BEGIN
	-- Wyświetlanie wartości przed przypisaniem.
    dbms_output.put_line('Wewnętrzny ['||lv_outer||']['||lv_active||']');

	-- Przypisywanie nowej wartości do zmiennej.
    lv_outer := 'WEWNĘTRZNA';

	-- Wyświetlanie wartości po przypisaniu.
    dbms_output.put_line('Wewnętrzny ['||lv_outer||']['||lv_active||']');
  END;

  -- Wyświetlanie wartości po bloku zagnieżdżonym.
  dbms_output.put_line('Zewnętrzny ['||lv_outer||']['||lv_active||']');

EXCEPTION
  WHEN OTHERS THEN
  dbms_output.put_line('Wyjątek '||SQLERRM||']');
END;
/

-- ------------------------------------------------------------
--   Lokalne bloki nazwane
-- ------------------------------------------------------------

-- Zagnieżdżony blok nazwany w bloku anonimowym.
DECLARE
  -- Deklaracje zmiennych lokalnych.
  lv_outer VARCHAR2(30) DEFAULT 'ZEWNĘTRZNA';
  lv_active VARCHAR2(30) DEFAULT 'ZEWNĘTRZNA';

  -- Procedura lokalna bez parametrów formalnych.
  PROCEDURE local_named IS
  -- Deklaracja zmiennej lokalnej.
  lv_active VARCHAR2(30) DEFAULT 'WEWNĘTRZNA';

  BEGIN
	-- Wyświetlanie wartości przed przypisaniem.
    dbms_output.put_line(
     'Wewnętrzny ['||lv_outer||']['||lv_active||']');

	-- Przypisanie nowej wartości do zmiennej.
    lv_local := 'WEWNĘTRZNA';

	-- Wyświetlanie wartości po przypisaniu.
    dbms_output.put_line(
     'Wewnętrzny ['||lv_outer||']['||lv_active||']');
  END local_named;

BEGIN
  -- Wyświetlanie wartości przed blokiem wewnętrznym.
  dbms_output.put_line(
    'Zewnętrzny '||lv_outer||']['||lv_active||']');

  -- Wywołanie lokalnie zadeklarowanej procedury nazwanej.
  local_named;

  -- Wyświetlenie wartości po wywołaniu procedury.
  dbms_output.put_line(
   'Zewnętrzny ['||lv_outer||']['||lv_active||']');
EXCEPTION
  WHEN OTHERS THEN
  dbms_output.put_line('Wyjątek ['||SQLERRM||']');
END;
/

-- Problem z dostępem do zagnieżdżonych bloków nazwanych.
DECLARE
  PROCEDURE jack IS
  BEGIN
    dbms_output.put_line(hector||', świecie!');
  END jack;
  FUNCTION hector RETURN VARCHAR2 IS
  BEGIN
    RETURN 'Witaj';
  END hector;
BEGIN
  jack;
END;
/

-- Namiast z referencją uprzedzającą.
DECLARE
  PROCEDURE jack;
  FUNCTION hector RETURN VARCHAR2;
  PROCEDURE jack IS
  BEGIN
    dbms_output.put_line(b||', świecie!');
  END jack;
  FUNCTION hector RETURN VARCHAR2 IS
  BEGIN
    RETURN 'Witaj';
  END hector;
BEGIN
  jack;
END;
/

-- ------------------------------------------------------------
--   Składowany blok nazwany
-- ------------------------------------------------------------

-- Niezależna procedura nazwana.
CREATE OR REPLACE PROCEDURE local_named IS
  -- Deklaracja zmiennej lokalnej.
  lv_active VARCHAR2(30) DEFAULT 'WEWNĘTRZNA';
  lv_outer VARCHAR2(30) DEFAULT ' ';

BEGIN
  -- Wyświetlanie wartości przed przypisaniem.
  dbms_output.put_line(
   'Wewnętrzny ['||lv_outer||']['||lv_active||']');

  -- Przypisywanie nowej wartości do zmiennej.
  lv_outer := 'WEWNĘTRZNA';

  -- Print the value after the assignment.
  dbms_output.put_line(
   'Wewnętrzny ['||lv_outer||']['||lv_active||']');
END local_named;
/

-- Blok anonimowy wywołujący program nazwany.
DECLARE
  -- Deklaracja zmiennej lokalnej.
  lv_outer VARCHAR2(30) DEFAULT 'ZEWNĘTRZNA';
  lv_active VARCHAR2(30) DEFAULT 'ZEWNĘTRZNA';

BEGIN
  -- Wyświetlanie wartości przed blokiem wewnętrznym.
  dbms_output.put_line('Zewnętrzny ['||lv_outer||']['||lv_active||']');

  -- Wywołanie programu nazwanego.
  local_named;

  -- Wyświetlenie wartości po wywołaniu programu nazwanego.
  dbms_output.put_line('Zewnętrzny ['||lv_outer||']['||lv_active||']');
EXCEPTION
  WHEN OTHERS THEN
  dbms_output.put_line('Wyjątek ['||SQLERRM||']');
END;
/

-- ------------------------------------------------------------
--   Podstawowe typy skalarne i złożone
-- ------------------------------------------------------------
--   Typy skalarne
-- ------------------------------------------------------------

-- Typy łańcuchowe.
DECLARE
  lv_fixed CHAR(40) := 'Niezbyt długi tekst.';
  lv_variable VARCHAR2(40) := 'Niezbyt długi tekst.';
BEGIN
  dbms_output.put_line('Stała długość ['||LENGTH(lv_fixed)||']');
  dbms_output.put_line('Zmienna długość ['||LENGTH(lv_variable)||']');
END;
/

-- Typy danych związane z datami. 
DECLARE
  lv_date_1 DATE := '28-APR-75';
  lv_date_2 DATE := '29-APR-1975';
  lv_date_3 DATE := TO_DATE('19750430','YYYYMMDD');
BEGIN
  dbms_output.put_line('Niejawny ['||lv_date_1||']');
  dbms_output.put_line('Niejawny ['||lv_date_2||']');
  dbms_output.put_line('Jawny ['||lv_date_3||']');
END;
/

-- Typy liczbowe.
DECLARE
  lv_number1 NUMBER;
  lv_number2 NUMBER(4,2) := 99.99;
BEGIN
  lv_number1 := lv_number2;
  dbms_output.put_line(lv_number1);
END;
/

-- ------------------------------------------------------------
--   Typ UDT SQL-a
-- ------------------------------------------------------------

@hobbit.sql

-- ------------------------------------------------------------
--   Typ rekordowy języka PL/SQL
-- ------------------------------------------------------------

-- Tworzenie i zapełnianie rekordu.
DECLARE
  -- Deklaracja lokalnej, zdefiniowanej przez użytkownika struktury rekordowej.
  TYPE title_record IS RECORD
  ( title VARCHAR2(60)
  , subtitle VARCHAR2(60));

  -- Deklaracja zmiennej typu struktury rekordowej.
  lv_title_record TITLE_RECORD;
BEGIN
  -- Przypisanie wartości do struktury rekordowej.
  lv_title_record.title := 'Star Trek';
  lv_title_record.subtitle := 'Into Darkness';

  -- Wyświetlanie elementów struktury.
  dbms_output.put_line('['||lv_title_record.title||']'||
   '['||lv_title_record.subtitle||']');
END;
/

-- Tworzenie lub zastępowanie typu STRING_TABLE.
CREATE OR REPLACE
  TYPE string_table IS TABLE OF VARCHAR2(30);
/

-- Tworzenie lub zastępowanie typu STRING_VARRAY.
CREATE OR REPLACE
  TYPE string_varray IS VARRAY(3) OF VARCHAR2(30);
/

-- Blok anonimowy przechodzący w pętli po tablicy.
DECLARE
  -- Deklarowanie i inicjowanie kolekcji zbóż.
  lv_string_list STRING_TABLE := string_table('Kukurydza','Pszenica');
BEGIN
  -- Wyświetlanie pierwszego elementu tablicy.
  FOR i IN 1..2 LOOP
    dbms_output.put_line('['||i||']['||lv_string_list(i)||']');
  END LOOP;
END;
/

-- Blok anonimowy przenoszący dane z listy do tablicy.
DECLARE
  -- Deklarowanie i inicjowanie kolekcji zbóż.
  lv_string_table STRING_TABLE :=
    string_table('Kukurydza','Pszenica','Owies','Jęczmień');
  lv_string_varray STRING_VARRAY := string_varray();
BEGIN
  -- Wyświetlanie pierwszego elementu w tablicy.
  FOR i IN 1..lv_string_table.COUNT LOOP
    lv_string_varray.EXTEND;
    lv_string_varray(i) := lv_string_table(i);
  END LOOP;
END;
/

-- ------------------------------------------------------------
--   Kolekcja UDT
-- ------------------------------------------------------------

@hobbit.sql

-- ------------------------------------------------------------
--   Kolekcja z języka PL/SQL
-- ------------------------------------------------------------

-- Blok anonimowy z tabelą zagnieżdżoną.
DECLARE
  -- Deklaracja kolekcji z liczbami.
  TYPE number_table IS TABLE OF NUMBER;

  -- Deklaracja zmiennej typu kolekcji.
  lv_collection NUMBER_TABLE := number_type(1,2,3);
BEGIN
  -- Przejście w pętli po kolekcji i wyświetlenie wartości.
  FOR i IN 1..lv_collection.COUNT LOOP
    dbms_output.put_line(lv_collection(i));
  END LOOP;
END;
/

-- Blok anonimowy z kolekcją VARRAY.
DECLARE
  -- Deklaracja typu kolekcji zawierającej liczby.
  TYPE number_table IS VARRAY(3) OF NUMBER;

  -- Deklaracja zmiennej typu kolekcji.
  lv_collection NUMBER_TABLE := number_type(1,2,3);
BEGIN
  -- Przejście w pętli po kolekcji i wyświetlenie wartości.
  FOR i IN 1..lv_collection.COUNT LOOP
    dbms_output.put_line(lv_collection(i));
  END LOOP;
END;
/

-- Blok anonimowy z tablicą asocjacyjną.
DECLARE
  -- Deklaracja typu kolekcji z liczbami.
  TYPE numbers IS TABLE OF NUMBER INDEX BY BINARY_INTEGER;

  -- Deklaracja zmiennej typu kolekcji.
  lv_collection NUMBERS;
BEGIN
  -- Przypisanie wartości do kolekcji.
  lv_collection(0) := 1;
END;
/

-- Blok anonimowy z tablicą asocjacyjną.
DECLARE
  -- Deklaracja typu kolekcji z liczbami.
  TYPE numbers IS TABLE OF NUMBER INDEX BY VARCHAR2(10);

  -- Deklaracja zmiennej typu kolekcji.
  lv_collection NUMBERS;
BEGIN
  -- Przypisanie wartości do kolekcji.
  lv_collection('Jeden') := 1;
END;
/

-- Blok anonimowy z typem UDT SQL-a.
DECLARE
  -- Deklaracja lokalnej kolekcji hobbitów.
  TYPE hobbit_table IS TABLE OF HOBBIT;

  -- Deklarowanie i inicjowanie kolekcji hobbitów.
  lv_string_table STRING_TABLE :=
  string_table('Drogo Baggins','Frodo Baggins');
  lv_hobbit_table HOBBIT_TABLE := hobbit_table(
      hobbit('Bungo Baggins')
    , hobbit('Bilbo Baggins'));
BEGIN
  -- Wyświetlanie pierwszego elementu kolekcji.
  FOR i IN 1..lv_string_table.COUNT LOOP
    lv_hobbit_table.EXTEND;
    lv_hobbit_table(lv_hobbit_table.COUNT) :=
    hobbit(lv_string_table(i));
  END LOOP;
  -- Wyświetlanie elementów kolekcji hobbit.
  FOR i IN 1..lv_hobbit_table.COUNT LOOP
    dbms_output.put_line(
      lv_hobbit_table(i).to_string());
  END LOOP;
END;
/

-- Blok anonimowy z tablicą asocjacyjną ze zmiennymi złożonymi.
DECLARE
  -- Deklaracja lokalnej struktury rekordowej zdefiniowanej przez użytkownika.
  TYPE dwarf_record IS RECORD
  ( dwarf_name VARCHAR2(20)
  , dwarf_home VARCHAR2(20));

  -- Deklaracja lokalnej kolekcji krasnoludów.
  TYPE dwarf_table IS TABLE OF DWARF_RECORD
    INDEX BY PLS_INTEGER;

  -- Deklarowanie i inicjowanie kolekcji krasnoludów.
  list DWARF_TABLE;
BEGIN
  -- Dodawanie dwóch elementów do tablicy asocjacyjnej.
  list(1).dwarf_name := 'Gloin';
  list(1).dwarf_home := 'Durin''s Folk';
  list(2).dwarf_name := 'Gimli';
  list(2).dwarf_home := 'Durin''s Folk';

  -- Wyświetlanie pierwszego elementu tablicy.
  FOR i IN 1..list.COUNT LOOP
    dbms_output.put_line(
     '['||list(i).dwarf_name||']'||
     '['||list(i).dwarf_home||']');
  END LOOP;
END;
/

-- ------------------------------------------------------------
--   Struktury sterujące
-- ------------------------------------------------------------
--   Struktury warunkowe
-- ------------------------------------------------------------

-- Instrukcje if, elsif i else.
DECLARE
  lv_boolean BOOLEAN;
  lv_number NUMBER;
BEGIN
  IF NVL(lv_boolean,FALSE) THEN
    dbms_output.put_line('Wyświetlane, gdy zmienna ma wartość true.');
  ELSIF NVL((lv_number < 10),FALSE) THEN
    dbms_output.put_line('Wyświetlane, gdy wyrażenie ma wartość true.');
  ELSE
    dbms_output.put_line('Wyświetlane, gdy zmienne to null .');
  END IF;
END;
/

-- Instrukcje CASE.
DECLARE
  lv_selector VARCHAR2(20);
BEGIN
  lv_selector := '&input';
  CASE lv_selector
    WHEN 'Gruszka' THEN
      dbms_output.put_line('Czy to pyszna żółta gruszka?');
    WHEN 'Pomidor' THEN
      dbms_output.put_line('Czy to czerwony pomidor?');
    ELSE
      dbms_output.put_line('To ['||lv_selector||']?');
  END CASE;
END;
/

-- Instrukcje CASE sprawdzająca, czy wyrażenie ma wartość true.
BEGIN
  CASE
    WHEN (1 <> 1) THEN
      dbms_output.put_line('Niemożliwe!');
    WHEN (3 > 2) THEN
      dbms_output.put_line('Spełniony warunek.');
    ELSE
      dbms_output.put_line('Nigdy nie dochodzi do tego miejsca.');
  END CASE;
END;
/

-- Instrukcje CASE sprawdzająca, czy wyrażenie ma wartość false.
BEGIN
  CASE FALSE
    WHEN (1 <> 1) THEN
      dbms_output.put_line('Niemożliwe!');
    WHEN (3 > 2) THEN
      dbms_output.put_line('Spełniony warunek.');
    ELSE
      dbms_output.put_line('Nigdy nie dochodzi do tego miejsca.');
  END CASE;
END;
/

-- ------------------------------------------------------------
--   Struktury iteracyjne
-- ------------------------------------------------------------

-- Pętla for odliczająca od początku.
BEGIN
  FOR i IN 0..9 LOOP
    dbms_output.put_line('['||i||']['||TO_CHAR(i+1)||']');
  END LOOP;
END;
/

-- Pętla for odliczająca od tyłu.
BEGIN
  FOR i IN REVERSE 1..9 LOOP
    dbms_output.put_line('['||i||']['||TO_CHAR(i+1)||']');
  END LOOP;
END;
/

-- Pętla for z kursorem niejawnym.
BEGIN
  FOR i IN (SELECT item_title FROM item) LOOP
    dbms_output.put_line(i.item_title);
  END LOOP;
END;
/

-- Pętla for ze statycznym kursorem jawnym.
DECLARE
  CURSOR c IS
    SELECT item_title FROM item;
BEGIN
  FOR i IN c LOOP
    dbms_output.put_line(i.item_title);
  END LOOP;
END;
/

-- Pętla for z dynamicznym kursorem jawnym.
DECLARE
  lv_search_string VARCHAR2(60);
  CURSOR c (cv_search VARCHAR2) IS
    SELECT item_title
    FROM item
    WHERE REGEXP_LIKE(item_title,'^'||cv_search||'*+');
BEGIN
  FOR i IN c ('&input') LOOP
    dbms_output.put_line(i.item_title);
  END LOOP;
END;
/

-- ------------------------------------------------------------
--   Klauzula WHERE OF
-- ------------------------------------------------------------

-- Kursor statyczny dla bloku anonimowego.
DECLARE
  CURSOR c IS
    SELECT * FROM item
    WHERE item_id BETWEEN 1031 AND 1040
    FOR UPDATE;
BEGIN
  FOR I IN c LOOP 
    UPDATE item SET last_updated_by = 3
    WHERE CURRENT OF c;
  END LOOP;
END;
/

-- Kursor statyczny dla bloku anonimowego.
DECLARE
  TYPE update_record IS RECORD
  ( last_updated_by NUMBER
  , last_update_date DATE );
  TYPE update_table IS TABLE OF UPDATE_RECORD;
  updates UPDATE_TABLE;
  CURSOR c IS
    SELECT last_updated_by, last_update_date
    FROM item
    WHERE item_id BETWEEN 1031 AND 1040
    FOR UPDATE;
BEGIN
  OPEN c;
  LOOP
    FETCH c BULK COLLECT INTO updates LIMIT 5;
    EXIT WHEN updates.COUNT = 0;
    FORALL i IN updates.FIRST..updates.LAST
    UPDATE item
    SET last_updated_by = updates(i).last_updated_by
    , last_update_date = updates(i).last_update_date
    WHERE CURRENT OF c;
  END LOOP;
END;
/

-- Skorelowana instrukcja UPDATE.
UPDATE item i1
SET last_updated_by = 3
, last_update_date = TRUNC(SYSDATE)
WHERE EXISTS (SELECT NULL FROM item i2
              WHERE item_id BETWEEN 1031 AND 1040
              AND i1.ROWID = i2.ROWID);

-- ------------------------------------------------------------
--   Pętla While
-- ------------------------------------------------------------
-- Pętla WHILE z instrukcją CONTINUE.
DECLARE
  lv_counter NUMBER := 1;
BEGIN
  WHILE (lv_counter < 5) LOOP
    dbms_output.put('Indeks na początku ['||lv_counter||']');
    IF lv_counter >= 1 THEN
      IF MOD(lv_counter,2) = 0 THEN
        dbms_output.new_line();
        lv_counter := lv_counter + 1;
        CONTINUE;
      END IF;
      dbms_output.put_line('['||lv_counter||']');
    END IF;
    lv_counter := lv_counter + 1;
  END LOOP;
END;
/

-- Pętla WHILE z instrukcją GOTO.
DECLARE
  lv_counter NUMBER := 1;
BEGIN
  WHILE (lv_counter < 5) LOOP
    dbms_output.put('Indeks na początku ['||lv_counter||']');
    IF lv_counter >= 1 THEN
      IF MOD(lv_counter,2) = 0 THEN
        dbms_output.new_line();
        GOTO skippy;
      END IF;
      dbms_output.put_line('['||lv_counter||']');
    END IF;
    << skippy >>
    lv_counter := lv_counter + 1;
  END LOOP;
END;
/

-- Przechwytywanie aktualizacji wierszy.
BEGIN
  UPDATE system_user
  SET last_update_date = SYSDATE;
  IF SQL%FOUND THEN
    dbms_output.put_line('Zaktualizowano ['||SQL%ROWCOUNT||']');
  ELSE
    dbms_output.put_line('Bez aktualizacji!');
  END IF;
END;
/

-- Pobieranie aktualizacji wierszy.
DECLARE
  lv_id item.item_id%TYPE; -- Kotwiczenie typu.
  lv_title VARCHAR2(60);
  CURSOR c IS
    SELECT item_id, item_title
    FROM item;
BEGIN
  OPEN c;
  LOOP
    FETCH c INTO lv_id, lv_title;
    EXIT WHEN c%NOTFOUND;
    dbms_output.put_line('Tytuł ['||lv_title||']');
  END LOOP;
  CLOSE c;
END;
/

-- Pobieranie wierszy z kursora o zakotwiczonym typie rekordowym.
DECLARE
  lv_item_record item%ROWTYPE; -- Kotwiczenie typu.
  CURSOR c IS
    SELECT *
    FROM item;
BEGIN
  OPEN c;
  LOOP
    FETCH c INTO lv_item_record;
    EXIT WHEN c%NOTFOUND;
    dbms_output.put_line('Tytuł ['||lv_item_record.item_title||']');
  END LOOP;
  CLOSE c;
END;
/

-- Pobieranie wierszy z kursora o typie rekordowym.
DECLARE
  TYPE item_record IS RECORD
  ( id NUMBER
  , title VARCHAR2(60));
  lv_item_record ITEM_RECORD; -- Ustawiony typ rekordowy.
  CURSOR c IS
    SELECT item_id, item_title
    FROM item;
BEGIN
  OPEN c;
  LOOP
    FETCH c INTO lv_item_record;
    EXIT WHEN c%NOTFOUND;
    dbms_output.put_line('Tytuł ['||lv_item_record.title||']');
  END LOOP;
  CLOSE c;
END;
/

-- Pobieranie wierszy z kursora zakotwiczonego do kursora.
DECLARE
  CURSOR c IS
    SELECT *
    FROM item;
  lv_item_record c%ROWTYPE;
BEGIN
  OPEN c;
  LOOP
    FETCH c INTO lv_item_record;
    EXIT WHEN c%NOTFOUND;
    dbms_output.put_line('Tytuł ['||lv_item_record.item_title||']');
  END LOOP;
  CLOSE c;
END;
/

-- ------------------------------------------------------------
--   Wyjątki
-- ------------------------------------------------------------
--   Wprowadzenie
-- ------------------------------------------------------------

-- Prosty komunikat o błędzie.
DECLARE
  lv_letter VARCHAR2(1);
  lv_phrase VARCHAR2(2) := 'AB';
BEGIN
  lv_letter := lv_phrase;
EXCEPTION
  WHEN OTHERS THEN
    dbms_output.put_line('Błąd: '||CHR(10)||SQLERRM);
END;
/

-- Prosty komunikat o błędzie dla wyjątku nazwanego.
DECLARE
  lv_letter VARCHAR2(1);
  lv_phrase VARCHAR2(2) := 'AB';
BEGIN
  lv_letter := lv_phrase;
EXCEPTION
  WHEN VALUE_ERROR THEN -- Specyficzny blok obsługi wyjątków.
    dbms_output.put_line('Błąd: '||CHR(10)||SQLERRM);
  WHEN OTHERS THEN -- Ogólny blok obsługi wyjątków.
    dbms_output.put_line('Błąd: '||CHR(10)||SQLERRM);
END;
/

-- ------------------------------------------------------------
--   Wyjątki zdefiniowane przez użytkownika
-- ------------------------------------------------------------

-- Wyświetlanie wyjątku zdefiniowanego przez użytkownika w bloku anonimowym.
DECLARE
  lv_error EXCEPTION;
BEGIN
  RAISE lv_error;
  dbms_output.put_line('Niedostępne.');
EXCEPTION
  WHEN OTHERS THEN
    IF SQLCODE = 1 THEN
      dbms_output.put_line('To wyjątek ['||SQLERRM||']');
    END IF;
END;
/

-- Wyświetlanie wyjątku zdefiniowanego przez użytkownika.
DECLARE
  lv_sys_context VARCHAR2(20);
  lv_error EXCEPTION;
  PRAGMA EXCEPTION_INIT(lv_error,-2003);
BEGIN
  lv_sys_context := SYS_CONTEXT('USERENV','PROXY_PUSHER');
  RAISE lv_error;
  dbms_output.put_line('Niedostępne.');
EXCEPTION
  WHEN lv_error THEN
    dbms_output.put_line('To wyjątek ['||SQLERRM||']');
END;
/

-- Wyświetlanie wyjątku zdefiniowanego przez użytkownika 
-- za pomocą instrukcji RAISE_APPLICATION_ERROR.
DECLARE
  lv_error EXCEPTION;
  PRAGMA EXCEPTION_INIT(lv_error,-20001);
BEGIN
  RAISE_APPLICATION_ERROR(-20001,'Mało oryginalny komunikat.');
EXCEPTION
  WHEN lv_error THEN
    dbms_output.put_line('['||SQLERRM||']');
END;
/

-- ------------------------------------------------------------
--   Operacje masowe
-- ------------------------------------------------------------
--   Wprowadzenie
-- ------------------------------------------------------------

-- Masowe wyświetlanie kolekcji w bloku anonimowym
DECLARE
  TYPE title_record IS RECORD
  ( title VARCHAR2(60)
  , subtitle VARCHAR2(60)); 
  TYPE title_collection IS TABLE OF TITLE_RECORD;
  lv_title_collection TITLE_COLLECTION;
  CURSOR c IS
    SELECT item_title, item_subtitle
    FROM item;
BEGIN
  OPEN c;
  LOOP
    FETCH c BULK COLLECT INTO lv_title_collection LIMIT 20;
    EXIT WHEN lv_title_collection.COUNT = 0;
    FOR i IN 1..lv_title_collection.COUNT LOOP
      dbms_output.put_line('['||lv_title_collection(i).title||']');
    END LOOP;
  END LOOP;
  CLOSE c;
END;
/

-- Masowe wyświetlanie kolekcji za pomocą instrukcji FORALL.
DECLARE
  TYPE title_record IS RECORD
  ( id NUMBER
  , title VARCHAR2(60)
  , subtitle VARCHAR2(60));
  TYPE title_collection IS TABLE OF TITLE_RECORD;
  lv_title_collection TITLE_COLLECTION;
  CURSOR c IS
    SELECT item_id, item_title, item_subtitle
    FROM item;
BEGIN
  OPEN c;
  LOOP
    FETCH c BULK COLLECT INTO lv_title_collection LIMIT 20;
    EXIT WHEN lv_title_collection.COUNT = 0;
    FORALL i IN lv_title_collection.FIRST..lv_title_collection.LAST
      UPDATE item_temp
      SET    item_title = lv_title_collection(i).title
      ,      item_subtitle = lv_title_collection(i).subtitle
      WHERE  item_id = lv_title_collection(i).id;
  END LOOP;
END;
/

-- ------------------------------------------------------------
--   Funkcje, procedury i pakiety
-- ------------------------------------------------------------
--   Funkcje
-- ------------------------------------------------------------

-- Tworzenie lub zastępowanie funkcji JOIN_STRINGS.
CREATE OR REPLACE FUNCTION join_strings
( string1 VARCHAR2
, string2 VARCHAR2 ) RETURN VARCHAR2 IS
BEGIN
  RETURN string1 ||' '|| string2||'.';
END;
/

-- Zapytania używające tej funkcji.
SELECT join_strings('Witaj','świecie') FROM dual;
VARIABLE session_var VARCHAR2(30)
CALL join_strings('Witaj','świecie') INTO :session_var;
SELECT :session_var FROM dual;

-- Tworzenie lub zastępowanie procedury formatującej.
CREATE OR REPLACE PROCEDURE format_string
( string_in IN OUT VARCHAR2 ) IS
BEGIN
  string_in := '['||string_in||']';
END;
/

-- Testy funkcji JOIN_STRINGS i procedury FORMAT_STRING.
VARIABLE session_var VARCHAR2(30)
CALL join_strings('Witaj','świecie') INTO :session_var;
CALL format_string(:session_var);
EXECUTE format_string(:session_var);
SELECT :session_var FROM dual;

-- Tworzenie lub zastępowanie specyfikacji pakietu overloading.
CREATE OR REPLACE PACKAGE overloading IS

  -- Wymuszanie tworzenia świeżej kopii współużytkowanego kursora.
  PRAGMA SERIALLY_REUSABLE;

  -- Definicja domyślnej funkcji salutation.
  FUNCTION salutation
  ( pv_long_phrase VARCHAR2 DEFAULT 'Witaj'
  , pv_name VARCHAR2 ) RETURN VARCHAR2;

  -- Definicja przeciążonej funkcji salutation.
  FUNCTION salutation
  ( pv_long_phrase VARCHAR2 DEFAULT 'Witaj'
  , pv_name VARCHAR2
  , pv_language VARCHAR2 ) RETURN VARCHAR2;
END;
/

-- Tworzenie tabeli SALUTATION_TRANSLATION.
CREATE TABLE salutation_translation
( short_salutation VARCHAR2(4)
, long_salutation VARCHAR2(12)
, phrase_language VARCHAR2(12));

-- Wstawianie czterech rekordów do tabeli.
INSERT INTO salutation_translation VALUES ('Hi','HELLO','ENGLISH');
INSERT INTO salutation_translation VALUES ('Bye','GOODBYE','ENGLISH');
INSERT INTO salutation_translation VALUES ('Ciao','SALUTE','ITALIAN');
INSERT INTO salutation_translation VALUES ('Ciao','ADDIO','ITALIAN');

-- Tworzenie lub zastępowanie ciała pakietu overloading.
CREATE OR REPLACE PACKAGE BODY overloading IS

  -- Force fresh copy of shared cursor.
  -- Wymuszanie utworzenia świeżej kopii kursora współużytkowanego.
  PRAGMA SERIALLY_REUSABLE;
  -- Kursor współużytkowany.
  CURSOR c
  ( cv_long_phrase VARCHAR2
  , cv_language VARCHAR2 ) IS
  SELECT short_salutation
  , long_salutation
  FROM salutation_translation
  WHERE long_salutation = UPPER(cv_long_phrase)
  AND phrase_language = UPPER(cv_language);

  -- Deklaracja domyślnej funkcji salutation.
  FUNCTION salutation
  ( pv_long_phrase VARCHAR2 DEFAULT 'Hello'
  , pv_name VARCHAR2 ) RETURN VARCHAR2 IS

    -- Zmienna lokalna.
    lv_short_salutation VARCHAR2(4) := '';
    lv_language VARCHAR2(10) DEFAULT 'ENGLISH';
 
  BEGIN
	-- Wczytywanie współużytkowanego kursora i zwracania połączonego wyniku.
    FOR i IN c(pv_long_phrase, lv_language) LOOP
      lv_short_salutation := i.short_salutation;
    END LOOP;
    RETURN lv_short_salutation || ' ' || pv_name || '!';
  END;

  -- Deklaracja przeciążonej funkcji salutation.
  FUNCTION salutation
  ( pv_long_phrase VARCHAR2 DEFAULT 'Hello'
  , pv_name VARCHAR2
  , pv_language VARCHAR2) RETURN VARCHAR2 IS
 
    -- Zmienna lokalna.
    lv_short_salutation VARCHAR2(4) := '';

  BEGIN
	-- Wczytywanie współużytkowanego kursora i zwracania połączonego wyniku.
    FOR i IN c(pv_long_phrase, pv_language) LOOP
      lv_short_salutation := i.short_salutation;
    END LOOP;
    RETURN lv_short_salutation || ' ' || pv_name || '!';
  END;
END;
/

-- Testowanie pakietu OVERLOADING.
VARIABLE message VARCHAR2(30)
CALL overloading.salutation('Hello','Ringo') INTO :message;
CALL overloading.salutation('Addio','Lennon','Italian') INTO :message;
SELECT :message AS "Pożegnanie" FROM dual;
SELECT overloading.salutation('Addio','Lennon','Italian') AS "Komunikat" FROM dual;

-- ------------------------------------------------------------
--   Funkcje, procedury i pakiety
-- ------------------------------------------------------------
--   Funkcje
-- ------------------------------------------------------------


BEGIN
  -- Ustawianie punktu zapisu.
  SAVEPOINT all_or_nothing;

  -- Najpierw wstawianie.
  INSERT INTO member
  VALUES
  ( member_s1.NEXTVAL -- Sztuczny klucz główny.
  ,(SELECT   common_lookup_id
    FROM     common_lookup
    WHERE    common_lookup_table = 'MEMBER'
    AND      common_lookup_column = 'MEMBER_TYPE'  )
16 , SYSDATE);
17
18 -- Drugie wstawianie.
19 INSERT INTO contact
20 VALUES
21 ( contact_s1.NEXTVAL -- Sztuczny klucz główny.
22 , member_s1.CURRVAL -- Sztuczny klucz zewnętrzny.
...
30 , SYSDATE);
30
31 -- Zatwierdzanie rekordów.
32 COMMIT;
33
34 EXCEPTION
35 -- Wycofanie do punktu zapisu i zgłoszenie wyjątku.
36 WHEN others THEN
37 ROLLBACK TO all_or_nothing;
38 dbms_output.put_line(SQLERRM);
39 END;
40 /

