/* ================================================================
||   Program: load_clob_from_file.sql
||   Data:       2013-07-25
||   Książka:    Oracle Database 12c. Programowanie w języku PL/SQL
||   Rozdział:   10
||   Autor:  Michael McLaughlin
|| ----------------------------------------------------------------
||   Zawartość:
||   ---------
||   UWAGI:
||
||   Plik HarryPotter1.png należy umieścić w katalogu /tmp 
||   (Linux lub Unix) albo C:\WINDOWS\TEMP (Windows).
||   Inna możliwość to zdefiniowanie katalogu wirtualnego GENERIC powiązanego
||   z wybranym katalogiem. Upewnij się jednak, że baza Oracle
||   ma uprawnienia do odczytu danych z tego katalogu.
||
||   Ten skrypt przedstawia procedurę wczytującą duży plik tekstowy
||   do obiektu typu CLOB.
||
||   TRYB DEBUGOWANIA:
||   ==========
||   Zmień ustawienia sesji za pomocą poniższej instrukcji.
||
||   ALTER SESSION SET PLSQL_CCFLAGS = 'debug:1';
|| ================================================================*/

-- Po zakończeniu debugowania umieść w komentarzu.
SET ECHO ON
SET FEEDBACK ON
SET PAGESIZE 49999
SET SERVEROUTPUT ON SIZE 1000000

-- Procedura składowana wczytująca dane do obiektu CLOB.
CREATE OR REPLACE PROCEDURE load_clob_from_file
( src_file_name     IN VARCHAR2
, table_name        IN VARCHAR2
, column_name       IN VARCHAR2
, primary_key_name  IN VARCHAR2
, primary_key_value IN VARCHAR2 ) IS

  -- Definicje zmiennych lokalnych procedury DBMS_LOB.LOADCLOBFROMFILE.
  des_clob   CLOB;
  src_clob   BFILE := BFILENAME('GENERIC',src_file_name);
  des_offset NUMBER := 1;
  src_offset NUMBER := 1;
  ctx_lang   NUMBER := dbms_lob.default_lang_ctx;
  warning    NUMBER;
  
  -- Zmienna na wielkość obiektu CLOB.
  src_clob_size NUMBER;
  
  -- Definicja zmiennej lokalnej na potrzeby instrukcji NDS.
  stmt VARCHAR2(2000);

BEGIN

  -- Trzeba otworzyć plik źródłowy.
  IF dbms_lob.fileexists(src_clob) = 1 AND NOT dbms_lob.isopen(src_clob) = 1 THEN
    src_clob_size := dbms_lob.getlength(src_clob);
    dbms_lob.open(src_clob,DBMS_LOB.LOB_READONLY);
  END IF;
  
  -- Przypisywanie dynamicznego łańcucha znaków do instrukcji.
  stmt := 'UPDATE '||table_name||' '
       || 'SET    '||column_name||' = empty_clob() '
       || 'WHERE  '||primary_key_name||' = '||''''||primary_key_value||''' '
       || 'RETURNING '||column_name||' INTO :locator';

  -- Wykonywanie dynamicznej instrukcji.     
  EXECUTE IMMEDIATE stmt USING OUT des_clob;
 
  -- Odczyt pliku i zapis danych do obiektu CLOB.
  dbms_lob.loadclobfromfile( dest_lob     => des_clob
                           , src_bfile    => src_clob
                           , amount       => dbms_lob.getlength(src_clob)
                           , dest_offset  => des_offset
                           , src_offset   => src_offset
                           , bfile_csid   => dbms_lob.default_csid
                           , lang_context => ctx_lang
                           , warning      => warning );

  -- Zamknięcie otwartego pliku źródłowego.
  dbms_lob.close(src_clob);

  -- Zatwierdzanie zapisu i warunkowe potwierdzanie powodzenia.
  IF src_clob_size = dbms_lob.getlength(des_clob) THEN
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Powodzenie!');
    $END
    COMMIT;
  ELSE
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Niepowodzenie.');
    $END
    RAISE dbms_lob.operation_failed;
  END IF;
  
END load_clob_from_file;
/

SHOW ERRORS

-- Formatowanie kolumn w celu wyświetlenia danych.
COL item_id    FORMAT 9999
COL item_title FORMAT A50
COL size       FORMAT 9,999,990

-- Sprawdzanie danych przed wczytaniem.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_desc) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_desc) > 0;

-- Wstawianie opisu we wszystkich pasujących wierszach.
BEGIN
  FOR i IN (SELECT item_id
            FROM   item
            WHERE  item_title = 'The Lord of the Rings - Fellowship of the Ring'
            AND    item_type IN (SELECT common_lookup_id
                                 FROM   common_lookup
                                 WHERE  common_lookup_table = 'ITEM'
                                 AND    common_lookup_column = 'ITEM_TYPE'
                                 AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'))) LOOP
    -- Wywołanie procedury dla pasujących wierszy.
    load_clob_from_file( src_file_name     => 'LOTRFellowship.txt'
                       , table_name        => 'ITEM'
                       , column_name       => 'ITEM_DESC'
                       , primary_key_name  => 'ITEM_ID'
                       , primary_key_value => TO_CHAR(i.item_id) );
  END LOOP;
END;
/

-- Sprawdzanie danych po wczytaniu.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_desc) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_desc) > 0;