-- The SQL*Plus script object_schema.sql performs the following:
--   1. Creates object_user
--   2. Creates the database object types
--   3. Populates the database tables with example data

-- This script should be run by the system user (or the DBA)
CONNECT system/manager;

-- drop object_user
DROP USER object_user CASCADE;

-- create object_user
CREATE USER object_user IDENTIFIED BY object_password;

-- allow object_user to connect and create database objects
GRANT connect, resource, create public synonym TO object_user;

-- connect as object_user
CONNECT object_user/object_password;

-- tworzenie typw obiektowych
CREATE TYPE t_address AS OBJECT (
  street VARCHAR2(15),
  city VARCHAR2(15),
  state CHAR(3),
  zip VARCHAR2(5)
);
/

CREATE TYPE t_person AS OBJECT (
  id INTEGER,
  first_name VARCHAR2(10),
  last_name VARCHAR2(10),
  dob DATE,
  phone VARCHAR2(12),
  address t_address
);
/

CREATE TYPE t_product AS OBJECT (
  id INTEGER,
  name VARCHAR2(15),
  description VARCHAR2(22),
  price NUMBER(5, 2),
  days_valid INTEGER,

  -- get_sell_by_date() zwraca dat przed ktr
  -- produkt musi zosta sprzedany
  MEMBER FUNCTION get_sell_by_date RETURN DATE
);
/


CREATE TYPE BODY t_product AS
  -- get_sell_by_date() zwraca dat przed ktr
  -- produkt musi zosta sprzedany
  MEMBER FUNCTION get_sell_by_date RETURN DATE IS
    v_sell_by_date DATE;
  BEGIN
    -- oblicza termin sprzeday poprzez dodanie wartoci atrybutu days_valid
    -- do biecej daty (SYSDATE)
    SELECT days_valid + SYSDATE
    INTO v_sell_by_date
    FROM dual;

    -- zwraca ostateczny termin sprzeday 
    RETURN v_sell_by_date;
  END;
END;
/


CREATE TYPE t_person2 AS OBJECT (
  id INTEGER,
  first_name VARCHAR2(10),
  last_name VARCHAR2(10),
  dob DATE,
  phone VARCHAR2(12),
  address t_address,

  -- deklaracja funkcji odwzorowujcej get_string()
  -- zwracajcej napis typu VARCHAR2
  MAP MEMBER FUNCTION get_string RETURN VARCHAR2
);
/

CREATE TYPE BODY t_person2 AS
  -- definicja funkcji odwzorowujcej get_string()
  MAP MEMBER FUNCTION get_string RETURN VARCHAR2 IS
  BEGIN
    -- zwraca napis bdcy konkatenacj
    -- atrybutw last_name i first_name
    RETURN last_name || ' ' || first_name;
  END get_string;
END;
/




CREATE PUBLIC SYNONYM t_pub_product FOR t_product;

-- tworzenie tabel
CREATE TABLE products (
  product           t_product,
  quantity_in_stock INTEGER
);


CREATE TABLE object_products OF t_product;

CREATE TABLE object_customers OF t_person;

CREATE TABLE purchases (
  id       INTEGER PRIMARY KEY,
  customer_ref REF t_person SCOPE IS object_customers,
  product_ref  REF t_product SCOPE IS object_products
);

CREATE TABLE object_customers2 OF t_person2;


-- tworzenie pakietu PL/SQL
CREATE OR REPLACE PACKAGE product_package AS
  TYPE t_ref_cursor IS REF CURSOR;
  FUNCTION get_products RETURN t_ref_cursor;
  PROCEDURE display_product(
    p_id IN object_products.id%TYPE
  );
  PROCEDURE insert_product(
    p_id IN object_products.id%TYPE,
    p_name IN object_products.name%TYPE,
    p_description IN object_products.description%TYPE,
    p_price IN object_products.price%TYPE,
    p_days_valid IN object_products.days_valid%TYPE
  );
  PROCEDURE update_product_price(
    p_id IN object_products.id%TYPE,
    p_factor IN NUMBER
  );
  FUNCTION get_product(
    p_id IN object_products.id%TYPE
  ) RETURN t_product;
  PROCEDURE update_product(
    p_product t_product
  );
  FUNCTION get_product_ref(
    p_id IN object_products.id%TYPE
  ) RETURN REF t_product;
  PROCEDURE delete_product(
    p_id IN object_products.id%TYPE
  );
END product_package;
/


CREATE OR REPLACE PACKAGE BODY product_package AS

  FUNCTION get_products
  RETURN t_ref_cursor IS
    -- deklaracja obiektu t_ref_cursor 
    v_products_ref_cursor t_ref_cursor;
  BEGIN
    -- pobranie REF CURSOR
    OPEN v_products_ref_cursor FOR
      SELECT VALUE(op)
      FROM object_products op
       ORDER BY op.id;

    -- zwraca REF CURSOR
    RETURN v_products_ref_cursor;
  END get_products;

  PROCEDURE display_product(
    p_id IN object_products.id%TYPE
  ) AS
    -- deklaracja obiektu t_product o nazwie v_product
    v_product t_product;
  BEGIN
    -- prba uzyskania produktu i zapisania go w v_product
    SELECT VALUE(op)
    INTO v_product
    FROM object_products op
    WHERE id = p_id;

    -- wywietla atrybuty v_product
    DBMS_OUTPUT.PUT_LINE('v_product.id=' ||
      v_product.id);
    DBMS_OUTPUT.PUT_LINE('v_product.name=' ||
      v_product.name);
    DBMS_OUTPUT.PUT_LINE('v_product.description=' ||
      v_product.description);
    DBMS_OUTPUT.PUT_LINE('v_product.price=' ||
      v_product.price);
    DBMS_OUTPUT.PUT_LINE('v_product.days_valid=' ||
      v_product.days_valid);

    -- wywouje v_product.get_sell_by_date() i wywietla dat
    DBMS_OUTPUT.PUT_LINE('sell_by_date=' ||
      v_product.get_sell_by_date());
  END display_product;

  PROCEDURE insert_product(
    p_id IN object_products.id%TYPE,
    p_name IN object_products.name%TYPE,
    p_description IN object_products.description%TYPE,
    p_price IN object_products.price%TYPE,
    p_days_valid IN object_products.days_valid%TYPE
  ) AS
    -- tworzy obiekt t_product o nazwie v_product
    v_product t_product :=
      t_product(
        p_id, p_name, p_description, p_price, p_days_valid
    );
  BEGIN
    -- wstawia v_product do tabeli object_products
    INSERT INTO object_products VALUES (v_product);
    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
      ROLLBACK;
  END insert_product;

  PROCEDURE update_product_price(
    p_id IN object_products.id%TYPE,
    p_factor IN NUMBER
  ) AS
  -- deklaracja objektu t_product o nazwie v_product
    v_product t_product;
  BEGIN
    -- prba wybrania produktu do modyfikacji
    -- i zapisania go w v_product
    SELECT VALUE(op)
    INTO v_product
    FROM object_products op
    WHERE id = p_id
    FOR UPDATE;

    -- wywietla aktualn cen v_product
    DBMS_OUTPUT.PUT_LINE('v_product.price=' ||
      v_product.price);

    -- mnoy v_product.price przez p_factor
    v_product.price := v_product.price * p_factor;
    DBMS_OUTPUT.PUT_LINE('Nowa warto v_product.price=' ||
      v_product.price);

    -- aktualizuje produkt w tabeli object_products
    UPDATE object_products op
    SET op = v_product
    WHERE id = p_id;
    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
    ROLLBACK;
  END update_product_price;

  FUNCTION get_product(
    p_id IN object_products.id%TYPE
  )
  RETURN t_product IS
    -- deklaracja obiektu t_product o nazwie v_product
    v_product t_product;
  BEGIN
    -- pobiera produkt i zapisuje go w v_product
    SELECT VALUE(op)
    INTO v_product
    FROM object_products op
    WHERE op.id = p_id;

    -- zwraca v_product
    RETURN v_product;
  END get_product;

  PROCEDURE update_product(
    p_product IN t_product
  ) AS
  BEGIN
    -- aktualizuje produkt w tabeli object_products
    UPDATE object_products op
    SET op = p_product
    WHERE id = p_product.id;
    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
    ROLLBACK;
  END update_product;

  FUNCTION get_product_ref(
    p_id IN object_products.id%TYPE
  )
  RETURN REF t_product IS
    -- deklaracja odwoania do t_product
    v_product_ref REF t_product;
  BEGIN
    -- pobiera REF do produktu
    -- i zapisuje go w v_product_ref
    SELECT REF(op)
    INTO v_product_ref
    FROM object_products op
    WHERE op.id = p_id;

    -- zwraca v_product_ref
    RETURN v_product_ref;
  END get_product_ref;

  PROCEDURE delete_product(
    p_id IN object_products.id%TYPE
  ) AS
  BEGIN
    -- usu produkt
    DELETE FROM object_products op
    WHERE op.id = p_id;
    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
      ROLLBACK;
  END delete_product;


END product_package;
/


CREATE OR REPLACE PROCEDURE product_lifecycle AS
  -- deklaracja obiektu
  v_product t_product;
BEGIN
  -- wstawia nowy produkt
  product_package.insert_product(4, 'woowina',
    '0,5 kg paczka woowiny', 32, 10);

  -- wywietla produkt
  product_package.display_product(4);

  -- pobiera nowy produkt i zapisuje go w v_product
  SELECT product_package.get_product(4)
  INTO v_product
  FROM dual;

  -- zmienia kilka atrybutw v_product
  v_product.description := '0,4 kg paczka woowiny';
  v_product.price := 36;
  v_product.days_valid := 8;

  -- modyfikuje produkt
  product_package.update_product(v_product);

  -- wywietla produkt
  product_package.display_product(4);

  -- usuwa produkt
  product_package.delete_product(4);
END product_lifecycle;
/


CREATE OR REPLACE PROCEDURE product_lifecycle2 AS
  -- deklaracja obiektu
  v_product t_product;

  -- deklaracja odwoania obiektowego
  v_product_ref REF t_product;
BEGIN
  -- wstawia nowy produkt
  product_package.insert_product(4, 'woowina',
    '0,5 kg paczka woowiny', 32, 10);

  -- wywietla produkt
  product_package.display_product(4);

  -- pobiera odwoanie do nowego produktu i zapisuje je w v_product_ref
  SELECT product_package.get_product_ref(4)
  INTO v_product_ref
  FROM dual;

  -- dereferencja v_product_ref z uyciem nastpujcego zapytania
  SELECT DEREF(v_product_ref)
  INTO v_product
  FROM dual;

  -- zmiana niektrych atrybutw v_product
  v_product.description := '0,4 kg paczka woowiny';
  v_product.price := 36;
  v_product.days_valid := 8;

  -- aktualizacja produktu
  product_package.update_product(v_product);

  -- wywietla produkt
  product_package.display_product(4);

  -- usuwa produkt
  product_package.delete_product(4);
END product_lifecycle2;
/



-- wstawianie przykadowych danych do tabeli products

INSERT INTO products (
  product,
  quantity_in_stock
) VALUES (
  t_product(1, 'makaron', '0,5 kg paczka makaronu', 3.95, 10),
  50
);

INSERT INTO products (
  product,
  quantity_in_stock
) VALUES (
  t_product(2, 'sardynki', '100 g puszka sardynek', 2.99, 5),
  25
);

-- wstawianie przykadowych danych do tabeli object_products

INSERT INTO object_products VALUES (
  t_product(1, 'makaron', '0,5 kg paczka makaronu', 3.95, 10)
);

INSERT INTO object_products (
  id, name, description, price, days_valid
) VALUES (
  2, 'sardynki', '100 g puszka sardynek', 2.99, 5
);


-- wstawianie przykadowych danych do tabeli object_customers

INSERT INTO object_customers VALUES (
  t_person(1, 'Jan', 'Brzowy', '55/02/01', '800-555-1211',
    t_address('Kociuszki 23', 'Siemiatycze', 'MAL', '12345')
  )
);

INSERT INTO object_customers (
  id, first_name, last_name, dob, phone,
  address
) VALUES (
  2, 'Sylwia', 'Zielona', '68/02/05', '800-555-1212',
  t_address('Wolnoci 3', 'rdziemie', 'MAZ', '12345')
);

-- wstawianie przykadowych danych do tabeli purchases

INSERT INTO purchases (
  id,
  customer_ref,
  product_ref
) VALUES (
  1,
  (SELECT REF(oc) FROM object_customers oc WHERE oc.id = 1),
  (SELECT REF(op) FROM object_products  op WHERE op.id = 1)
);

-- wstawianie przykadowych danych do tabeli object_customers2
INSERT INTO object_customers2 VALUES (
  t_person2(1, 'Jan', 'Brzowy', '55/02/01', '800-555-1211',
      t_address('Kociuszki 23', 'Siemiatycze', 'MAL', '12345')
  )
);
INSERT INTO object_customers2 VALUES (
  t_person2(2, 'Sylwia', 'Zielona', '68/02/05', '800-555-1212',
    t_address('Wolnoci 3', 'rdziemie', 'MAZ', '12345')
  )
);



-- commit the transaction
COMMIT;
