-- The SQL*Plus script collection_user.sql performs the following:
--   1. Creates collection_user
--   2. Creates the database object and collection type
--   3. Populates the database tables with example data

-- This script should be run by the system user (or the DBA)
CONNECT system/manager;

-- drop collection_user
DROP USER collection_user CASCADE;

-- create collection_user
CREATE USER collection_user IDENTIFIED BY collection_password;

-- allow collection_user to connect and create database objects
GRANT connect, resource TO collection_user;

-- connect as collection_user
CONNECT collection_user/collection_password;

-- tworzenie obiektw i kolekcji
CREATE TYPE t_varray_address AS VARRAY(3) OF VARCHAR2(50);
/

CREATE TYPE t_address AS OBJECT (
  street VARCHAR2(15),
  city VARCHAR2(15),
  state CHAR(3),
  zip VARCHAR2(5)
);
/

CREATE TYPE t_nested_table_address AS TABLE OF t_address;
/

CREATE TYPE t_address2 AS OBJECT (
  street VARCHAR2(15),
  city VARCHAR2(15),
  state CHAR(3),
  zip VARCHAR2(5),

  -- deklaracja funkcji mapujcej get_string()
  -- zwracajcej napis VARCHAR2
  MAP MEMBER FUNCTION get_string RETURN VARCHAR2
);
/

CREATE TYPE BODY t_address2 AS
  -- definicja funkcji mapujcej get_string()
  MAP MEMBER FUNCTION get_string RETURN VARCHAR2 IS
  BEGIN
    -- zwraca skonkatenowany napis, zawierajcy wartoci
    -- atrybutw zip, state, city i street
    RETURN zip || ' ' || state || ' ' || city || ' ' || street;
  END get_string;
END;
/

CREATE TYPE t_nested_table_address2 AS TABLE OF t_address2;
/

CREATE TYPE t_varray_address2 AS VARRAY(3) OF t_address;
/

-- tworzenie tabel
CREATE TABLE customers_with_varray (
  id INTEGER PRIMARY KEY,
  first_name VARCHAR2(10),
  last_name VARCHAR2(10),
  addresses t_varray_address
);

CREATE TABLE customers_with_nested_table (
  id INTEGER PRIMARY KEY,
  first_name VARCHAR2(10),
  last_name VARCHAR2(10),
  addresses t_nested_table_address
)
NESTED TABLE
  addresses
STORE AS
  nested_addresses;

CREATE TABLE customers_with_nested_table2 (
  id INTEGER PRIMARY KEY,
  first_name VARCHAR2(10),
  last_name VARCHAR2(10),
  addresses t_nested_table_address2
)
NESTED TABLE
  addresses
STORE AS
  nested_addresses2;

CREATE TABLE customers_with_varray2 (
  id INTEGER PRIMARY KEY,
  first_name VARCHAR2(10),
  last_name VARCHAR2(10),
  addresses t_varray_address2
);


-- wstawianie wierszy do tabeli
INSERT INTO customers_with_varray VALUES (
  1, 'Stefan', 'Brzowy',
  t_varray_address(
    'Stanowa 2, Fasolowo, MAZ, 12345',
    'Grska 4, Rzeszotary, GDA, 54321'
  )
);

INSERT INTO customers_with_varray VALUES (
  2, 'Jan', 'Kowalski',
  t_varray_address(
    'Wysoka 1, Zae, MA, 12347',
    'Nowa 4, Byczyna, MAZ, 54323',
    'Handlowa 7, Zarzecze, POZ, 54323'
  )
);

INSERT INTO customers_with_nested_table VALUES (
  1, 'Stefan', 'Brzowy',
  t_nested_table_address(
    t_address('Stanowa 2', 'Fasolowo', 'MAZ', '12345'),
    t_address('Grska 4', 'Rzeszotary', 'GDA', '54321')
  )
);


INSERT INTO customers_with_nested_table VALUES (
  2, 'Jan', 'Kowalski',
  t_nested_table_address(
    t_address('Wysoka 1', 'Zae', 'MA', '12347'),
    t_address('Nowa 4', 'Byczyna', 'MAZ', '54323'),
    t_address('Handlowa 7', 'Zarzecze', 'POZ', '54323')
  )
);


INSERT INTO customers_with_nested_table2 VALUES (
  1, 'Stefan', 'Brzowy',
  t_nested_table_address2(
    t_address2('Stanowa 2', 'Fasolowo', 'MAZ', '12345'),
    t_address2('Wysoka 4', 'Byczyna', 'POZ', '54321')
  )
);

INSERT INTO customers_with_varray2 VALUES (
  1, 'Jan', 'Bond',
  t_varray_address2(
    t_address('Newtona 3', 'Inny Kcik', 'MA', '22123'),
    t_address('Wiosenna 6', 'Nowe Miasto', 'POZ', '77712')
  )
);

-- tworzenie pakietw
CREATE OR REPLACE PACKAGE varray_package AS
  TYPE t_ref_cursor IS REF CURSOR;
  FUNCTION get_customers RETURN t_ref_cursor;
  PROCEDURE insert_customer(
    p_id IN customers_with_varray.id%TYPE,
    p_first_name IN customers_with_varray.first_name%TYPE,
    p_last_name IN customers_with_varray.last_name%TYPE,
    p_addresses IN customers_with_varray.addresses%TYPE
  );
END varray_package;
/

CREATE PACKAGE BODY varray_package AS
  -- funkcja get_customers() zwraca REF CURSOR
  -- wskazujcy na wiersze w customers_with_varray
  FUNCTION get_customers
  RETURN t_ref_cursor IS
    -- deklaracja obiektu REF CURSOR
    v_customers_ref_cursor t_ref_cursor;
  BEGIN
    -- pobranie REF CURSOR
    OPEN v_customers_ref_cursor FOR
      SELECT *
      FROM customers_with_varray;
    -- zwraca REF CURSOR
    RETURN v_customers_ref_cursor;
  END get_customers;

  -- procedura insert_customer() wstawia wiersz
  -- do tabeli customers_with_varray
  PROCEDURE insert_customer(
    p_id IN customers_with_varray.id%TYPE,
    p_first_name IN customers_with_varray.first_name%TYPE,
    p_last_name IN customers_with_varray.last_name%TYPE,
    p_addresses IN customers_with_varray.addresses%TYPE
  ) IS
  BEGIN
    INSERT INTO customers_with_varray
    VALUES (p_id, p_first_name, p_last_name, p_addresses);
    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
      ROLLBACK;
  END insert_customer;
END varray_package;
/

CREATE PACKAGE nested_table_package AS
  TYPE t_ref_cursor IS REF CURSOR;
  FUNCTION get_customers RETURN t_ref_cursor;
  PROCEDURE insert_customer(
    p_id IN customers_with_nested_table.id%TYPE,
    p_first_name IN customers_with_nested_table.first_name%TYPE,
    p_last_name IN customers_with_nested_table.last_name%TYPE,
    p_addresses IN customers_with_nested_table.addresses%TYPE
  );
END nested_table_package;
/

CREATE PACKAGE BODY nested_table_package AS
  -- funkcja get_customers() zwraca REF CURSOR
  -- wskazujcy na wiersze tabeli customers_with_nested_table
  FUNCTION get_customers
  RETURN t_ref_cursor IS
    -- deklaracja obiektu REF CURSOR
    v_customers_ref_cursor t_ref_cursor;
  BEGIN
    -- pobranie REF CURSOR
    OPEN v_customers_ref_cursor FOR
      SELECT *
      FROM customers_with_nested_table;
    -- zwraca REF CURSOR
    RETURN v_customers_ref_cursor;
  END get_customers;

  -- procedura insert_customer() wstawia wiersz
  -- do tabeli customers_with_nested_table
  PROCEDURE insert_customer(
    p_id IN customers_with_nested_table.id%TYPE,
    p_first_name IN customers_with_nested_table.first_name%TYPE,
    p_last_name IN customers_with_nested_table.last_name%TYPE,
    p_addresses IN customers_with_nested_table.addresses%TYPE
  ) IS
  BEGIN
    INSERT INTO customers_with_nested_table
    VALUES (p_id, p_first_name, p_last_name, p_addresses);
    COMMIT;
  EXCEPTION
    WHEN OTHERS THEN
      ROLLBACK;
  END insert_customer;
END nested_table_package;
/

CREATE OR REPLACE PACKAGE collection_method_examples AS
  FUNCTION get_addresses(
    p_id customers_with_nested_table.id%TYPE
  ) RETURN t_nested_table_address;

  PROCEDURE display_addresses(p_addresses t_nested_table_address);
  PROCEDURE delete_address(p_address_num INTEGER);
  PROCEDURE exist_addresses;
  PROCEDURE extend_addresses;
  PROCEDURE first_address;
  PROCEDURE last_address;
  PROCEDURE next_address;
  PROCEDURE prior_address;
  PROCEDURE trim_addresses;

END collection_method_examples;
/

CREATE OR REPLACE PACKAGE BODY collection_method_examples AS
  FUNCTION get_addresses(
    p_id customers_with_nested_table.id%TYPE
  ) RETURN t_nested_table_address IS
    -- deklaracja obiektu o nazwie v_addresses, 
    -- w ktrym bdzie skadowana tabela zagniedona z adresami
    v_addresses t_nested_table_address;
  BEGIN
    -- pobranie tabeli zagniedonej z adresami do v_addresses
    SELECT addresses
    INTO v_addresses
    FROM customers_with_nested_table
    WHERE id = p_id;

    -- wywietla liczb adresw za pomoc v_addresses.COUNT
    DBMS_OUTPUT.PUT_LINE(
    'Liczba adresw = '|| v_addresses.COUNT
    );

    -- zwraca v_addresses
    RETURN v_addresses;
  END get_addresses;

  PROCEDURE display_addresses(
    p_addresses t_nested_table_address
  ) IS
    v_count INTEGER;
  BEGIN
    -- wywietla liczb adresw w p_addresses
    DBMS_OUTPUT.PUT_LINE(
    'Bieca liczba adresw = '|| p_addresses.COUNT
    );

    -- korzystajc z ptli wywietla adresy znajdujce si w p_addreses
    FOR v_count IN 1..p_addresses.COUNT LOOP
      DBMS_OUTPUT.PUT_LINE('Adres nr ' || v_count || ':');
      DBMS_OUTPUT.PUT(p_addresses(v_count).street || ', ');
      DBMS_OUTPUT.PUT(p_addresses(v_count).city || ', ');
      DBMS_OUTPUT.PUT(p_addresses(v_count).state || ', ');
      DBMS_OUTPUT.PUT_LINE(p_addresses(v_count).zip);
    END LOOP;
  END display_addresses;

  PROCEDURE delete_address(
    p_address_num INTEGER
  ) IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    display_addresses(v_addresses);
    DBMS_OUTPUT.PUT_LINE('Usuwanie adresu nr ' || p_address_num);

    -- usuwa adres okrelony przez p_address_num
    v_addresses.DELETE(p_address_num);

    display_addresses(v_addresses);
  END delete_address;

  PROCEDURE exist_addresses IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    DBMS_OUTPUT.PUT_LINE('Usuwanie adresu nr 1');
      v_addresses.DELETE(1);
    -- uywa EXISTS do sprawdzenia, czy adresy istniej
    IF v_addresses.EXISTS(1) THEN
      DBMS_OUTPUT.PUT_LINE('Adres nr 1 istnieje');
    ELSE
      DBMS_OUTPUT.PUT_LINE('Adres nr 1 nie istnieje');
    END IF;
    IF v_addresses.EXISTS(2) THEN
      DBMS_OUTPUT.PUT_LINE('Adres nr 2 istnieje');
    END IF;
  END exist_addresses;

  PROCEDURE extend_addresses IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    display_addresses(v_addresses);
    DBMS_OUTPUT.PUT_LINE('Rozszerzanie adresw');

    -- kopiuje dwukrotnie adres nr 1 na koniec v_addresses
    v_addresses.EXTEND(2, 1);
    display_addresses(v_addresses);
  END extend_addresses;

  PROCEDURE first_address IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    -- wywietla adres zwracany przez FIRST
    DBMS_OUTPUT.PUT_LINE('Pierwszy adres = ' || v_addresses.FIRST);
    DBMS_OUTPUT.PUT_LINE('Usuwanie adresu nr 1');
    v_addresses.DELETE(1);

    -- Ponownie wywietla adres zwracany przez FIRST
    DBMS_OUTPUT.PUT_LINE('Pierwszy adres = ' || v_addresses.FIRST);
  END first_address;

  PROCEDURE last_address IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    -- wywietla adres zwracany przez LAST
    DBMS_OUTPUT.PUT_LINE('Ostatni adres = ' || v_addresses.LAST);
    DBMS_OUTPUT.PUT_LINE('Usuwanie adresu nr 2');
    v_addresses.DELETE(2);
    -- ponownie wywietla adres zwracany przez LAST
    DBMS_OUTPUT.PUT_LINE('Ostatni adres = ' || v_addresses.LAST);
  END last_address;

  PROCEDURE next_address IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    -- uycie NEXT(1) do pobrania indeksu adresu
    -- znajdujcego si za adresem nr 1
    DBMS_OUTPUT.PUT_LINE(
      'v_addresses.NEXT(1) = ' || v_addresses.NEXT(1)
    );

    -- uycie NEXT(2) w celu pobrania indeksu
    -- adresu znajdujcego si za adresem nr 2
    -- (taki adres nie istnieje, wic zwracane jest NULL)
    DBMS_OUTPUT.PUT_LINE(
      'v_addresses.NEXT(2) = ' || v_addresses.NEXT(2)
    );
  END next_address;

  PROCEDURE prior_address IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    -- uycie PRIOR(2) do pobrania indeksu adresu
    -- znajdujcego si przed adresem nr 2
    DBMS_OUTPUT.PUT_LINE(
      'v_addresses.PRIOR(2) = ' || v_addresses.PRIOR(2)
    );

    -- uycie PRIOR(1) do pobrania indeksu
    -- adresu znajdujcego si przed adresem nr 1
    -- (taki adres nie istnieje, wic zwracana jest warto NULL)
    DBMS_OUTPUT.PUT_LINE(
      'v_addresses.PRIOR(1) = ' || v_addresses.PRIOR(1)
    );
  END prior_address;

  PROCEDURE trim_addresses IS
    v_addresses t_nested_table_address;
  BEGIN
    v_addresses := get_addresses(1);
    display_addresses(v_addresses);
    DBMS_OUTPUT.PUT_LINE('Rozszerzanie adresw');
    v_addresses.EXTEND(3, 1);
    display_addresses(v_addresses);
    DBMS_OUTPUT.PUT_LINE('Usuwanie 2 adresw z koca');

    -- usuwa 2 adresy z koca v_address,
    -- korzystajc z TRIM(2)
    v_addresses.TRIM(2);

    display_addresses(v_addresses);
  END trim_addresses;


END collection_method_examples;
/


-- commit the transaction
COMMIT;
