/*
 * LoadLobFromFile.sql
 * Rozdzia 16, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Ten skrypt testuje procedur DBMS_LOB.LOADLOBFROMFILE
 */

exec CLEAN_SCHEMA.TABLES
exec CLEAN_SCHEMA.OBJECTS
exec CLEAN_SCHEMA.PROCS

PROMPT
PROMPT ** Tworzenie tabeli book_samples
PROMPT

CREATE TABLE book_samples (
   book_sample_id     NUMBER (10) PRIMARY KEY,
   isbn               CHAR(10 CHAR),
   description        CLOB,
   nls_description    NCLOB,
   misc               BLOB,
   chapter_title      VARCHAR2(30 CHAR),
   bfile_description  BFILE
)  
   LOB (misc) 
      STORE AS blob_seg ( TABLESPACE blob_ts
                 CHUNK 8192
                 PCTVERSION 0
                 NOCACHE
                 NOLOGGING
                 DISABLE STORAGE IN ROW)
   LOB (description, nls_description) 
   STORE AS ( TABLESPACE clob_ts
                 CHUNK 8192
                 PCTVERSION 10
                 NOCACHE
                 LOGGING
                 ENABLE STORAGE IN ROW);

PROMPT
PROMPT ** Wstawianie rekordu do tabeli book_samples
PROMPT

INSERT INTO book_samples (
   book_sample_id,
   isbn,
   description,
   nls_description,
   misc,
   bfile_description)
 VALUES (
   1,
   '72230665', 
   EMPTY_CLOB(),
   EMPTY_CLOB(),
   EMPTY_BLOB(),
   BFILENAME('BOOK_SAMPLES_LOC', 'bfile_example.pdf'));

COMMIT;

set serveroutput on

DECLARE
   v_dest_blob BLOB;
   v_dest_clob CLOB;
   v_source_locator1 BFILE := BFILENAME('BOOK_SAMPLES_LOC', 'bfile_example.pdf');
   v_source_locator2 BFILE := BFILENAME('BOOK_SAMPLES_LOC', 'bfile_example.txt');
   v_source_offset NUMBER := 1;
   v_dest_offset NUMBER := 1;
   v_lang_context NUMBER := DBMS_LOB.DEFAULT_LANG_CTX;
   v_warning PLS_INTEGER;

BEGIN
   -- Oprnienie kolumn description i misc
   UPDATE book_samples
   SET description = EMPTY_CLOB(),
       misc = EMPTY_BLOB()
   WHERE book_sample_id = 1;
   
   -- Pobranie lokalizatorw kolumn docelowych
   SELECT description, misc
   INTO v_dest_clob, v_dest_blob
   FROM book_samples
   WHERE book_sample_id = 1
   FOR UPDATE;

   -- Otwarcie obiektw BFILE i docelowych obiektw LOB
   DBMS_LOB.OPEN(v_source_locator1, DBMS_LOB.LOB_READONLY);
   DBMS_LOB.OPEN(v_source_locator2, DBMS_LOB.LOB_READONLY);
   DBMS_LOB.OPEN(v_dest_blob, DBMS_LOB.LOB_READWRITE);
   DBMS_LOB.OPEN(v_dest_clob, DBMS_LOB.LOB_READWRITE);

   DBMS_OUTPUT.PUT_LINE('Rozmiar pliku BLOB to: '  
         ||DBMS_LOB.GETLENGTH(v_source_locator1));
   DBMS_OUTPUT.PUT_LINE('Rozmiar pliku CLOB to: '  
         ||DBMS_LOB.GETLENGTH(v_source_locator2));
   DBMS_OUTPUT.PUT_LINE('Rozmiar obiektu BLOB przed wczytywaniem: ' 
         ||DBMS_LOB.GETLENGTH(v_dest_blob));
   DBMS_OUTPUT.PUT_LINE('Rozmiar obiektu CLOB przed wczytywaniem: ' 
         ||DBMS_LOB.GETLENGTH(v_dest_clob));

   -- Zapenianie kolumn docelowych danymi rdowymi
   DBMS_LOB.LOADBLOBFROMFILE(v_dest_blob,
                              v_source_locator1, 
                              DBMS_LOB.LOBMAXSIZE, 
                              v_dest_offset, 
                              v_source_offset);
   DBMS_OUTPUT.PUT_LINE('Rozmiar obiektu BLOB po wczytywaniu: '||(v_dest_offset -1));

   v_dest_offset := 1;
   v_source_offset := 1;

   DBMS_LOB.LOADCLOBFROMFILE(v_dest_clob, 
                               v_source_locator2, 
                               DBMS_LOB.LOBMAXSIZE, 
                               v_dest_offset, 
                               v_source_offset, 
                               DBMS_LOB.DEFAULT_CSID, 
                               v_lang_context, 
                               v_warning);

   DBMS_OUTPUT.PUT_LINE('Rozmiar obiektu CLOB po wczytywaniu: '||(v_dest_offset -1));

   -- Zamknicie otwartych obiektw LOB
   DBMS_LOB.CLOSE(v_source_locator1);
   DBMS_LOB.CLOSE(v_source_locator2);
   DBMS_LOB.CLOSE(v_dest_blob);
   DBMS_LOB.CLOSE(v_dest_clob);

EXCEPTION
   WHEN OTHERS
   THEN
      DBMS_OUTPUT.PUT_LINE(SQLERRM);

      DBMS_LOB.CLOSE(v_source_locator1);
      DBMS_LOB.CLOSE(v_source_locator2);
      DBMS_LOB.CLOSE(v_dest_blob);
      DBMS_LOB.CLOSE(v_dest_clob);

END;
/



PROMPT
PROMPT ** Wywolanie instrukcji SELECT na kolumnie description
PROMPT

SET LONG 64000
SELECT description
FROM book_samples
WHERE book_sample_id = 1;
