/*
 * invokers.sql
 * Rozdzia 9, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Ten skrypt demonstruje uprawnienia osoby wywoujcej
 */

set echo on
set serveroutput on

-- Najpierw kod tworzy uytkownikw userA i userB z potrzebnymi
-- obiektami. Trzeba poczy si z kontem o niezbdnych uprawnieniach,
-- na przykad SYSTEM, aby to zrobi.
-- Moesz te chcie zmieni uprawnienia UNLIMITED TABLESPACE
-- przedstawione poniej, aby jawnie przyzna limity na przestrzenie
-- tabel w uywanej bazie danych
connect system/manager
DROP USER UserA CASCADE;
CREATE USER UserA IDENTIFIED BY UserA;
GRANT CREATE SESSION, CREATE TABLE, CREATE PROCEDURE,
      UNLIMITED TABLESPACE, CREATE ROLE, DROP ANY ROLE TO UserA;

DROP USER UserB CASCADE;
CREATE USER UserB IDENTIFIED BY UserB;
GRANT CREATE SESSION, CREATE TABLE, CREATE PROCEDURE,
      UNLIMITED TABLESPACE TO UserB;

connect UserA/UserA

-- ***********************************
-- Sytuacja z rysunku 9.8 - Wszystkie obiekty nale do uytkownika UserA
-- ***********************************

-- Najpierw naley utworzy tabel books. Kod nie tworzy tabeli authors,
-- dlatego nie naley uywa ogranicze spjnoci
CREATE TABLE books (
  isbn      CHAR(10) PRIMARY KEY,
  category  VARCHAR2(20),
  title     VARCHAR2(100),
  num_pages NUMBER,
  price     NUMBER,
  copyright NUMBER(4),
  author1   NUMBER,
  author2   NUMBER,
  author3   NUMBER 
);

-- Wstawianie danych. To jedynie podzbir wierszy z
-- kompletnej tabeli books. Warto zauway, e jedna ksika ma
-- trzech autorw, podczas gdy dwie pozostae - po dwch

INSERT INTO books (isbn, category, title, num_pages, price, copyright, author1, author2, author3)
  VALUES ('72121203', 'Oracle Basics', 'Oracle DBA 101', 563, 39.99, 1999, 1, 2, 3);

INSERT INTO books (isbn, category, title, num_pages, price, copyright, author1, author2)
  VALUES ('72122048', 'Oracle Basics', 'Oracle8i: A Beginner''s Guide', 765, 44.99, 1999, 4, 5);

-- Tworzenie tabeli temp_table
CREATE TABLE temp_table (
  num_col    NUMBER,
  char_col   VARCHAR2(60)
);


-- Potrzebna jest take funkcja ThreeAuthors:
CREATE OR REPLACE FUNCTION ThreeAuthors(p_ISBN IN books.isbn%TYPE)
  RETURN BOOLEAN AS

  v_Author3 books.author3%TYPE;
BEGIN
  -- Pobranie trzeciego autora podanej ksiki do zmiennej v_Author3
  SELECT author3
    INTO v_Author3
    FROM books
    WHERE isbn = p_ISBN;

  -- Jeli zmienna v_Author3 to NULL, oznacza to, e ksika ma mniej ni
  -- trzech autorw, dlatego mona zwrci warto false.  W przeciwnym 
  -- razie trzeba zwrci true
  IF v_Author3 IS NULL THEN
    RETURN FALSE;
  ELSE
    RETURN TRUE;
  END IF;
END ThreeAuthors;
/
 
-- Wersja procedury RecordThreeAuthors z uprawnieniami osoby wywoujcej. Ta wersja
-- dziaa z uprawnieniami osoby wywoujcej, a nie waciciela
CREATE OR REPLACE PROCEDURE RecordThreeAuthors
  AUTHID CURRENT_USER AS
  CURSOR c_Books IS
    SELECT *
      FROM UserA.books;
BEGIN
  FOR v_BookRecord in c_Books LOOP
    -- Rejestruje w tabeli temp_table wszystkie ksiki
    -- majce trzech autorw
    IF UserA.ThreeAuthors(v_BookRecord.ISBN) THEN
      INSERT INTO temp_table (char_col) VALUES
        (v_BookRecord.title || ' ma trzech autorow');
    END IF;
  END LOOP;
END RecordThreeAuthors;
/
show errors

-- ***********************************
-- Sytuacja z rysunku 9.12: UserB z uprawnieniami SELECT
--                          do tabeli books
-- ***********************************

-- Przyznanie niezbdnych uprawnie uytkownikowi UserB
GRANT EXECUTE ON RecordThreeAuthors TO UserB;
GRANT SELECT ON books TO UserB;

-- Wywoanie z konta UserA. Wstawia dane do tabeli UserA.temp_table
BEGIN
  RecordThreeAuthors;
  COMMIT;
END;
/

-- Zapytanie do tabeli temp_table. Powinno zwrci jeden wiersz
SELECT * FROM temp_table;

-- Poczenie jako uytkownik UserB i utworzenie tabeli temp_table
connect UserB/UserB
CREATE TABLE temp_table (
  num_col    NUMBER,
  char_col   VARCHAR2(60)
  );

-- Teraz wywoanie procedury RecordThreeAuthors wstawi dane do tabeli
-- UserB.temp_table
BEGIN
  UserA.RecordThreeAuthors;
  COMMIT;
END;
/

-- Take w tym przypadku zapytanie powinno zwrci jeden wiersz
SELECT * FROM temp_table;

-- ***********************************
-- Sytuacja z rysunku 9.13: UserB bez uprawnie SELECT
--                          do tabeli books
-- ***********************************

-- Poczenie z konta UserA i odebranie uprawnie
connect UserA/UserA
REVOKE SELECT ON books FROM UserB;

-- Wywoanie z konta UserA wci dziaa.
-- Wywoanie jako uytkownik UserA. Powoduje wstawienie danych do tabeli UserA.temp_table.
BEGIN
  RecordThreeAuthors;
  COMMIT;
END;
/

-- Zapytanie do temp_table. Powinno zwrci dwa wiersze, po jednym
-- dla kadego wywoania RecordThreeAuthors
SELECT * FROM temp_table;

-- Poczenie jako uytkownik UserB i wywoanie
connect UserB/UserB

-- Teraz wywoanie procedury RecordThreeAuthors nie powiedzie si
BEGIN
  UserA.RecordThreeAuthors;
END;
/

-- ***********************************
-- Sytuacja z rysunku 9.14: uprawnienia SELECT do tabeli books
--                          przyznane przez rol
-- ***********************************

-- Poczenie jako UserA i utworzenie roli
connect UserA/UserA
DROP ROLE UserA_Role;
CREATE ROLE UserA_Role;
GRANT SELECT ON books TO UserA_Role;
GRANT UserA_Role TO UserB;

-- Poczenie jako UserB i wywoanie
connect UserB/UserB

-- Teraz wywoanie procedury RecordThreeAuthors powiedzie si
BEGIN
  UserA.RecordThreeAuthors;
  COMMIT;
END;
/

-- Teraz zapytanie powinno zwrci dwa wiersze
SELECT * FROM temp_table;
