/*
 * RestrictReferences.sql
 * Rozdzia 9, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Ten skrypt demonstruje pakiet InventoryOps
 */

CREATE OR REPLACE PACKAGE InventoryOps AS
  -- Modyfikuje dane okrelonej ksiki
  PROCEDURE UpdateISBN(p_ISBN IN inventory.isbn%TYPE,
                       p_Status IN inventory.status%TYPE,
                       p_StatusDate IN inventory.status_date%TYPE,
                       p_Amount IN inventory.amount%TYPE);
  PRAGMA RESTRICT_REFERENCES(UpdateISBN, RNPS, WNPS);
  
  -- Usuwa dane okrelonej ksiki
  PROCEDURE DeleteISBN(p_ISBN IN inventory.isbn%TYPE);
  PRAGMA RESTRICT_REFERENCES(DeleteISBN, RNPS, WNPS);
  
  -- Wyjtek zgaszany w funkcjach UpdateISBN i DeleteISBN, jeli
  -- danego numeru ISBN nie ma w tabeli inventory
  e_ISBNNotFound EXCEPTION;
  
  TYPE t_ISBNTable IS TABLE OF inventory.isbn%TYPE
    INDEX BY BINARY_INTEGER;
    
  -- Zwraca tabel PL/SQL zawierajc ksiki o okrelonym
  -- stanie
  PROCEDURE StatusList(p_Status IN inventory.status%TYPE,
                       p_Books OUT t_ISBNTable,
                       p_NumBooks OUT BINARY_INTEGER);
  PRAGMA RESTRICT_REFERENCES(StatusList, RNPS, WNPS, WNDS);

END InventoryOps;
/
show errors

CREATE OR REPLACE PACKAGE BODY InventoryOps AS
  -- Modyfikuje dane okrelonej ksiki
  PROCEDURE UpdateISBN(p_ISBN IN inventory.isbn%TYPE,
                       p_Status IN inventory.status%TYPE,
                       p_StatusDate IN inventory.status_date%TYPE,
                       p_Amount IN inventory.amount%TYPE) IS
  BEGIN
    UPDATE inventory
      SET status = p_Status, status_date = p_StatusDate, amount = p_Amount
      WHERE isbn = p_ISBN;
      
    -- Sprawdza, czy ksiki zostay zaktualizowane, a jeli nie - zgasza wyjtek
    IF SQL%ROWCOUNT = 0 THEN
      RAISE e_ISBNNotFound;
    END IF;
  END UpdateISBN;

  -- Usuwa dane okrelonej ksiki
  PROCEDURE DeleteISBN(p_ISBN IN inventory.isbn%TYPE) IS
  BEGIN
    DELETE FROM inventory
      WHERE isbn = p_ISBN;

    -- Sprawdza, czy ksiki zostay usunite, a jeli nie - zgasza wyjtek
    IF SQL%ROWCOUNT = 0 THEN
      RAISE e_ISBNNotFound;
    END IF;
  END DeleteISBN;

  -- Zwraca tabel PL/SQL zawierajc ksiki o okrelonym
  -- stanie
  PROCEDURE StatusList(p_Status IN inventory.status%TYPE,
                       p_Books OUT t_ISBNTable,
                       p_NumBooks OUT BINARY_INTEGER) IS
    v_ISBN inventory.isbn%TYPE;
    CURSOR c_Books IS
      SELECT isbn
        FROM inventory
        WHERE status = p_Status;
        
  BEGIN
    /* Parametr p_NumBooks to indeks tabeli. Jego warto pocztkowa to
     * 0 i zwiksza si po kadym wykonaniu ptli pobierajcej dane.
     * Na kocu ptli warto tego parametru to liczba pobranych wierszy,
     * ktra jest jednoczenie liczb wierszy zwrconych do
     * p_Books */
     p_NumBooks := 0;
     OPEN c_Books;
     LOOP
       FETCH c_Books INTO v_ISBN;
       EXIT WHEN c_Books%NOTFOUND;

       p_NumBooks := p_NumBooks + 1;       
       p_Books(p_NumBooks) := v_ISBN;
     END LOOP;
     CLOSE c_Books;
   END StatusList;

  
END InventoryOps;
/
show errors


-- W tej wersji pakietu InventoryOps zmodyfikowano procedur StatusList tak,
-- aby wstawiaa dane do tabeli temp_table, co narusza dyrektyw
CREATE OR REPLACE PACKAGE BODY InventoryOps AS
  -- Modyfikuje dane okrelonej ksiki
  PROCEDURE UpdateISBN(p_ISBN IN inventory.isbn%TYPE,
                       p_Status IN inventory.status%TYPE,
                       p_StatusDate IN inventory.status_date%TYPE,
                       p_Amount IN inventory.amount%TYPE) IS
  BEGIN
    UPDATE inventory
      SET status = p_Status, status_date = p_StatusDate, amount = p_Amount
      WHERE isbn = p_ISBN;
      
    -- Sprawdza, czy ksiki zostay zaktualizowane, a jeli nie - zgasza wyjtek
    IF SQL%ROWCOUNT = 0 THEN
      RAISE e_ISBNNotFound;
    END IF;
  END UpdateISBN;

  -- Usuwa dane okrelonej ksiki
  PROCEDURE DeleteISBN(p_ISBN IN inventory.isbn%TYPE) IS
  BEGIN
    DELETE FROM inventory
      WHERE isbn = p_ISBN;

    -- Sprawdza, czy ksiki zostay usunite, a jeli nie - zgasza wyjtek
    IF SQL%ROWCOUNT = 0 THEN
      RAISE e_ISBNNotFound;
    END IF;
  END DeleteISBN;

  -- Zwraca tabel PL/SQL zawierajc ksiki o okrelonym
  -- stanie
  PROCEDURE StatusList(p_Status IN inventory.status%TYPE,
                       p_Books OUT t_ISBNTable,
                       p_NumBooks OUT BINARY_INTEGER) IS
    v_ISBN inventory.isbn%TYPE;
    CURSOR c_Books IS
      SELECT isbn
        FROM inventory
        WHERE status = p_Status;
        
  BEGIN
    INSERT INTO temp_table (char_col)
      VALUES ('Pozdrowienia ze StatusList!');
    
    /* Parametr p_NumBooks to indeks tabeli. Jego warto pocztkowa to
     * 0 i zwiksza si po kadym wykonaniu ptli pobierajcej dane.
     * Na kocu ptli warto tego parametru to liczba pobranych wierszy,
     * ktra jest jednoczenie liczb wierszy zwrconych do
     * p_Books */
     p_NumBooks := 0;
     OPEN c_Books;
     LOOP
       FETCH c_Books INTO v_ISBN;
       EXIT WHEN c_Books%NOTFOUND;

       p_NumBooks := p_NumBooks + 1;       
       p_Books(p_NumBooks) := v_ISBN;
     END LOOP;
     CLOSE c_Books;
   END StatusList;
END InventoryOps;
/
show errors
