package com.apress.proandroid;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;

/**
 * Zmodyfikowana wersja klasy BitSet Androida (z wartociami typu int zamiast long)
 */
public class MyBitSet implements  Serializable, Cloneable {
	private static final long serialVersionUID = 7889876080059670889L;

	private static final int OFFSET = 5;

    private static final int ELM_SIZE = 1 << OFFSET;

    private static final int RIGHT_BITS = ELM_SIZE - 1;

    private static final int[] TWO_N_ARRAY = new int[] {
    	0x1, 0x2,
    	0x4, 0x8, 0x10, 0x20, 0x40, 0x80, 0x100, 0x200,
    	0x400, 0x800, 0x1000, 0x2000, 0x4000, 0x8000,
    	0x10000, 0x20000, 0x40000, 0x80000, 0x100000,
    	0x200000, 0x400000, 0x800000, 0x1000000, 0x2000000,
    	0x4000000, 0x8000000, 0x10000000, 0x20000000,
    	0x40000000, 0x80000000,

    };

    private int[] bits;

    private transient boolean needClear;

    private transient int actualArrayLength;

    private transient boolean isLengthActual;

    /**
     * Tworzenie nowego {@code BitSet} o wielkoci 64 bitw.
     *
     * @see #clear(int)
     * @see #set(int)
     * @see #clear()
     * @see #clear(int, int)
     * @see #set(int, boolean)
     * @see #set(int, int)
     * @see #set(int, int, boolean)
     */
    public MyBitSet() {
        bits = new int[1];
        actualArrayLength = 0;
        isLengthActual = true;
    }

    /**
     * Tworzenie nowego {@code BitSet} o wielkoci nbits. Jeli nbits nie jest
     * wielokrotnoci 64, naley utworzy {@code BitSet} o wielkoci nbits zaokrglonej
     * w gr do najbliszej wielokrotnoci 64.
     *
     * @param nbits
     *            wielko zbioru bitw.
     * @throws NegativeArraySizeException
     *             jeli {@code nbits} ma warto ujemn.
     * @see #clear(int)
     * @see #set(int)
     * @see #clear()
     * @see #clear(int, int)
     * @see #set(int, boolean)
     * @see #set(int, int)
     * @see #set(int, int, boolean)
     */
    public MyBitSet(int nbits) {
        if (nbits < 0) {
            throw new NegativeArraySizeException();
        }
        bits = new int[(nbits >> OFFSET)
                + ((nbits & RIGHT_BITS) > 0 ? 1 : 0)];
        actualArrayLength = 0;
        isLengthActual = true;
    }

    /**
     * Prywatny konstruktor wywoywany w metodzie get(int, int)
     *
     * @param bits
     *            wielko zbioru bitw
     */
    private MyBitSet(int[] bits, boolean needClear,
            int actualArrayLength, boolean isLengthActual) {
        this .bits = bits;
        this .needClear = needClear;
        this .actualArrayLength = actualArrayLength;
        this .isLengthActual = isLengthActual;
    }

    /**
     * Tworzy kopi danego {@code BitSet}.
     *
     * @return kopi danego {@code BitSet}.
     */
    @Override
    public Object clone() {
        try {
            MyBitSet clone = (MyBitSet) super .clone();
            clone.bits = bits.clone();
            return clone;
        } catch (CloneNotSupportedException e) {
            throw new AssertionError(e); // Zmodyfikowane pod ktem Androida
        }
    }

    /**
     * Porwnuje argument z danym {@code BitSet} i zwraca informacj o tym, czy
     * obiekty s rwne. Obiekt musi by egzemplarzem {@code BitSet} z tym samym
     * zbiorem bitw.
     *
     * @param obj
     *            porwnywany obiekt {@code BitSet}.
     * @return {@code boolean} okrelajcy, czy dany {@code BitSet} i obiekt
     *         {@code obj} s sobie rne.
     * @see #hashCode
     */
    @Override
    public boolean equals(Object obj) {
        if (this  == obj) {
            return true;
        }
        if (obj instanceof  MyBitSet) {
            int[] bsBits = ((MyBitSet) obj).bits;
            int length1 = this .actualArrayLength, length2 = ((MyBitSet) obj).actualArrayLength;
            if (this .isLengthActual && ((MyBitSet) obj).isLengthActual
                    && length1 != length2) {
                return false;
            }
            // Jeli jeden z obiektw BitSet jest wikszy od drugiego, naley sprawdzi,
            // czy ktry z dodatkowych bitw jest ustawiony. Jeli tak, naley zwrci false.
            if (length1 <= length2) {
                for (int i = 0; i < length1; i++) {
                    if (bits[i] != bsBits[i]) {
                        return false;
                    }
                }
                for (int i = length1; i < length2; i++) {
                    if (bsBits[i] != 0) {
                        return false;
                    }
                }
            } else {
                for (int i = 0; i < length2; i++) {
                    if (bits[i] != bsBits[i]) {
                        return false;
                    }
                }
                for (int i = length2; i < length1; i++) {
                    if (bits[i] != 0) {
                        return false;
                    }
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Zwiksza wielko wewntrznej tablicy, aby pomieci {@code len} bitw.
     * Maksymalny indeks nowej tablicy to wielokrotno 64.
     *
     * @param len
     *            Indeks, ktry musi by dostpny w nowej tablicy.
     */
    private final void growLength(int len) {
        int[] tempBits = new int[Math.max(len, bits.length * 2)];
        System.arraycopy(bits, 0, tempBits, 0, this .actualArrayLength);
        bits = tempBits;
    }

    /**
     * Oblicza skrt dla danego {@code BitSet}. Jeli dwa obiekty {@code BitSet} s sobie rwne,
     * {@code hashCode()} zwraca dla nich t sam warto.
     *
     * @return warto {@code int} reprezentujc skrt dla danego zbioru bitw.
     * @see #equals
     * @see java.util.Hashtable
     */
    @Override
    public int hashCode() {
        long x = 1234;
        for (int i = 0, length = actualArrayLength; i < length; i++) {
            x ^= bits[i] * (i + 1);
        }
        return (int) ((x >> 32) ^ x);
    }

    /**
     * Pobiera bit o indeksie {@code index}. Powiksza {@code BitSet}, jeli
     * {@code index > size}.
     *
     * @param index
     *            Indeks pobieranego bitu.
     * @return {@code true}, jeli bit o indeksie {@code index} jest ustawiony.
     *         W przeciwnym razie {@code false}.
     * @throws IndexOutOfBoundsException
     *             Jeli {@code index} ma warto ujemn.
     * @see #clear(int)
     * @see #set(int)
     * @see #clear()
     * @see #clear(int, int)
     * @see #set(int, boolean)
     * @see #set(int, int)
     * @see #set(int, int, boolean)
     */
    public boolean get(int index) {
        checkIndex(index);
        int arrayPos = index >> OFFSET;
        if (arrayPos < actualArrayLength) {
            return (bits[arrayPos] & TWO_N_ARRAY[index & RIGHT_BITS]) != 0;
        }
        return false;
    }

    private void checkIndex(int index) {
        if (index < 0) {
            throw new IndexOutOfBoundsException("index < 0");
        }
    }

    private void checkRange(int fromIndex, int toIndex) {
        if (fromIndex < 0 || toIndex < 0 || toIndex < fromIndex) {
            throw new IndexOutOfBoundsException("fromIndex="
                    + fromIndex + " toIndex=" + toIndex);
        }
    }

    /**
     * Pobiera bity z przedziau od {@code fromIndex} do {@code toIndex} i zwraca
     * nowy oparty na nich zbir bitw. Powiksza {@code BitSet}, jeli {@code toIndex &gt; size}.
     *
     * @param fromIndex
     *            Pocztkowa pozycja (podany element jest uwzgldniany).
     * @param toIndex
     *            Kocowa pozycja (podany element jest pomijany).
     * @return Nowy zestaw bitw oparty na podanym przedziale.
     * @throws IndexOutOfBoundsException
     *             Jeli {@code fromIndex} lub {@code toIndex} ma warto ujemn albo
     *             {@code toIndex} jest mniejsze ni {@code fromIndex}.
     * @see #get(int)
     */
    public MyBitSet get(int fromIndex, int toIndex) {
        checkRange(fromIndex, toIndex);

        int last = actualArrayLength << OFFSET;
        if (fromIndex >= last || fromIndex == toIndex) {
            return new MyBitSet(0);
        }
        if (toIndex > last) {
            toIndex = last;
        }

        int idx1 = fromIndex >> OFFSET;
        int idx2 = (toIndex - 1) >> OFFSET;
        int factor1 = (~0) << (fromIndex & RIGHT_BITS);
        int factor2 = (~0) >>> (ELM_SIZE - (toIndex & RIGHT_BITS));

        if (idx1 == idx2) {
            int result = (bits[idx1] & (factor1 & factor2)) >>> (fromIndex % ELM_SIZE);
            if (result == 0) {
                return new MyBitSet(0);
            }
            return new MyBitSet(new int[] { result }, needClear, 1, true);
        }
        int[] newbits = new int[idx2 - idx1 + 1];
        // Najpierw naley okreli wartoci pierwszego i ostatniego indeksu z zestawu bitw
        newbits[0] = bits[idx1] & factor1;
        newbits[newbits.length - 1] = bits[idx2] & factor2;

        // Uzupenianie rodkowych bitw
        for (int i = 1; i < idx2 - idx1; i++) {
            newbits[i] = bits[idx1 + i];
        }

        // Przesunicie wszystkie elementw nowebo zbioru bitw w prawo o fromIndex % ELM_SIZE
        int numBitsToShift = fromIndex & RIGHT_BITS;
        int actualLen = newbits.length;
        if (numBitsToShift != 0) {
            for (int i = 0; i < newbits.length; i++) {
                // Przesunicie biecego elementu w prawo niezalenie od 
                // znaku
                newbits[i] = newbits[i] >>> (numBitsToShift);

                // Zastosowanie ostatnich x bitw z newbits[i+1] do biecego
                // elementu
                if (i != newbits.length - 1) {
                    newbits[i] |= newbits[i + 1] << (ELM_SIZE - (numBitsToShift));
                }
                if (newbits[i] != 0) {
                    actualLen = i + 1;
                }
            }
        }
        return new MyBitSet(newbits, needClear, actualLen,
                newbits[actualLen - 1] != 0);
    }

    /**
     * Ustawia bit o indeksie {@code index} to 1. Powiksza {@code BitSet}, jeli
     * {@code index > size}.
     *
     * @param index
     *            Indeks ustawianego bitu.
     * @throws IndexOutOfBoundsException
     *             jeli {@code index} ma warto ujemn.
     * @see #clear(int)
     * @see #clear()
     * @see #clear(int, int)
     */
    public void set(int index) {
        checkIndex(index);
        int len = (index >> OFFSET) + 1;
        if (len > bits.length) {
            growLength(len);
        }
        bits[len - 1] |= TWO_N_ARRAY[index & RIGHT_BITS];
        if (len > actualArrayLength) {
            actualArrayLength = len;
            isLengthActual = true;
        }
        needClear();
    }

    /**
     * Ustawia bit o indeksie {@code index} na {@code val}. Powiksza
     * {@code BitSet}, jeli {@code index > size}.
     *
     * @param index
     *            Indeks ustawianego bitu.
     * @param val
     *            Warto, na ktr naley ustawi bit.
     * @throws IndexOutOfBoundsException
     *             Jeli {@code index} ma warto ujemn.
     * @see #set(int)
     */
    public void set(int index, boolean val) {
        if (val) {
            set(index);
        } else {
            clear(index);
        }
    }

    /**
     * Ustawia bity od {@code fromIndex} do {@code toIndex}. Powiksza
     * {@code BitSet}, jeli {@code toIndex > size}.
     *
     * @param fromIndex
     *            Pocztkowa pozycja (z uwzgldnieniem podanego elementu).
     * @param toIndex
     *            Kocowa pozycja (z pominiciem podanego elementu).
     * @throws IndexOutOfBoundsException
     *             Jeli {@code fromIndex} lub {@code toIndex} ma warto ujemn albo
     *             {@code toIndex} jest mniejsze od {@code fromIndex}.
     * @see #set(int)
     */
    public void set(int fromIndex, int toIndex) {
        checkRange(fromIndex, toIndex);

        if (fromIndex == toIndex) {
            return;
        }
        int len2 = ((toIndex - 1) >> OFFSET) + 1;
        if (len2 > bits.length) {
            growLength(len2);
        }

        int idx1 = fromIndex >> OFFSET;
        int idx2 = (toIndex - 1) >> OFFSET;
        long factor1 = (~0L) << (fromIndex & RIGHT_BITS);
        long factor2 = (~0L) >>> (ELM_SIZE - (toIndex & RIGHT_BITS));

        if (idx1 == idx2) {
            bits[idx1] |= (factor1 & factor2);
        } else {
            bits[idx1] |= factor1;
            bits[idx2] |= factor2;
            for (int i = idx1 + 1; i < idx2; i++) {
                bits[i] |= (~0L);
            }
        }
        if (idx2 + 1 > actualArrayLength) {
            actualArrayLength = idx2 + 1;
            isLengthActual = true;
        }
        needClear();
    }

    private void needClear() {
        this .needClear = true;
    }

    /**
     * Ustawia bity od {@code fromIndex} do {@code toIndex} na warto
     * {@code val}. Powiksza {@code BitSet}, jeli {@code toIndex > size}.
     *
     * @param fromIndex
     *            Pocztkowa pozycja (z uwzgldnieniem podanego elementu).
     * @param toIndex
     *            Kocowa pozycja (z pominiciem podanego elementu).
     * @param val
     *            Warto, na ktr naley ustawi bity.
     * @throws IndexOutOfBoundsException
     *             Jeli {@code fromIndex} lub {@code toIndex} ma warto ujemn albo
     *             {@code toIndex} jest mniejsze ni {@code fromIndex}.
     * @see #set(int,int)
     */
    public void set(int fromIndex, int toIndex, boolean val) {
        if (val) {
            set(fromIndex, toIndex);
        } else {
            clear(fromIndex, toIndex);
        }
    }

    /**
     * Zeruje wszystkie bity w {@code BitSet}.
     *
     * @see #clear(int)
     * @see #clear(int, int)
     */
    public void clear() {
        if (needClear) {
            for (int i = 0; i < bits.length; i++) {
                bits[i] = 0;
            }
            actualArrayLength = 0;
            isLengthActual = true;
            needClear = false;
        }
    }

    /**
     * Zeruje bit o indeksie {@code index}. Powiksza {@code BitSet}, jeli
     * {@code index > size}.
     *
     * @param index
     *            Indeks zerowanego bitu.
     * @throws IndexOutOfBoundsException
     *             if {@code index} is negative.
     * @see #clear(int, int)
     */
    public void clear(int index) {
        checkIndex(index);
        if (!needClear) {
            return;
        }
        int arrayPos = index >> OFFSET;
        if (arrayPos < actualArrayLength) {
            bits[arrayPos] &= ~(TWO_N_ARRAY[index & RIGHT_BITS]);
            if (bits[actualArrayLength - 1] == 0) {
                isLengthActual = false;
            }
        }
    }

    /**
     * Zaruje bity od {@code fromIndex} do {@code toIndex}. Powiksza
     * {@code BitSet}, jeli {@code toIndex > size}.
     *
     * @param fromIndex
     *            Pocztkowa pozycja (z uwzgldnieniem podanego elementu).
     * @param toIndex
     *            Kocowa pozycja (z pominiciem podanego elementu).
     * @throws IndexOutOfBoundsException
     *             Jeli {@code fromIndex} lub {@code toIndex} ma warto ujemn albo
     *             {@code toIndex} jest mniejsze ni {@code fromIndex}.
     * @see #clear(int)
     */
    public void clear(int fromIndex, int toIndex) {
        checkRange(fromIndex, toIndex);

        if (!needClear) {
            return;
        }
        int last = (actualArrayLength << OFFSET);
        if (fromIndex >= last || fromIndex == toIndex) {
            return;
        }
        if (toIndex > last) {
            toIndex = last;
        }

        int idx1 = fromIndex >> OFFSET;
        int idx2 = (toIndex - 1) >> OFFSET;
        long factor1 = (~0L) << (fromIndex & RIGHT_BITS);
        long factor2 = (~0L) >>> (ELM_SIZE - (toIndex & RIGHT_BITS));

        if (idx1 == idx2) {
            bits[idx1] &= ~(factor1 & factor2);
        } else {
            bits[idx1] &= ~factor1;
            bits[idx2] &= ~factor2;
            for (int i = idx1 + 1; i < idx2; i++) {
                bits[i] = 0;
            }
        }
        if ((actualArrayLength > 0)
                && (bits[actualArrayLength - 1] == 0)) {
            isLengthActual = false;
        }
    }

    /**
     * Zmienia warto bitu o indeksie {@code index}. Powiksza {@code BitSet}, jeli
     * {@code index > size}.
     *
     * @param index
     *            Indeks bitu, ktrego warto jest zmieniana.
     * @throws IndexOutOfBoundsException
     *             Jeli {@code index} ma warto ujemn.
     * @see #flip(int, int)
     */
    public void flip(int index) {
        checkIndex(index);
        int len = (index >> OFFSET) + 1;
        if (len > bits.length) {
            growLength(len);
        }
        bits[len - 1] ^= TWO_N_ARRAY[index & RIGHT_BITS];
        if (len > actualArrayLength) {
            actualArrayLength = len;
        }
        isLengthActual = !((actualArrayLength > 0) && (bits[actualArrayLength - 1] == 0));
        needClear();
    }

    /**
     * Zmienia warto bitw od {@code fromIndex} do {@code toIndex}. Powiksza
     * {@code BitSet}, jeli {@code toIndex > size}.
     *
     * @param fromIndex
     *            Pocztkowa pozycja (z uwzgldnieniem podanego elementu).
     * @param toIndex
     *            Kocowa pozycja (z pominiciem podanego elementu).
     * @throws IndexOutOfBoundsException
     *             Jeli {@code fromIndex} lub {@code toIndex} ma warto ujemn albo
     *             {@code toIndex} jest mniejsze ni {@code fromIndex}.
     * @see #flip(int)
     */
    public void flip(int fromIndex, int toIndex) {
        checkRange(fromIndex, toIndex);

        if (fromIndex == toIndex) {
            return;
        }
        int len2 = ((toIndex - 1) >> OFFSET) + 1;
        if (len2 > bits.length) {
            growLength(len2);
        }

        int idx1 = fromIndex >> OFFSET;
        int idx2 = (toIndex - 1) >> OFFSET;
        long factor1 = (~0L) << (fromIndex & RIGHT_BITS);
        long factor2 = (~0L) >>> (ELM_SIZE - (toIndex & RIGHT_BITS));

        if (idx1 == idx2) {
            bits[idx1] ^= (factor1 & factor2);
        } else {
            bits[idx1] ^= factor1;
            bits[idx2] ^= factor2;
            for (int i = idx1 + 1; i < idx2; i++) {
                bits[i] ^= (~0L);
            }
        }
        if (len2 > actualArrayLength) {
            actualArrayLength = len2;
        }
        isLengthActual = !((actualArrayLength > 0) && (bits[actualArrayLength - 1] == 0));
        needClear();
    }

    /**
     * Sprawdza, czy dwa obiekty {@code BitSet} maj przynajmniej jeden bit ustawiony na true
     * na tej samej pozycji.
     *
     * @param bs
     *            {@code BitSet} uywany do wyznaczenia czci wsplnej.
     * @return {@code true} jeli bs ma cz wspln z {@code BitSet}.
     *         W przeciwnym razie {@code false}.
     */
    public boolean intersects(MyBitSet bs) {
        int[] bsBits = bs.bits;
        int length1 = actualArrayLength, length2 = bs.actualArrayLength;

        if (length1 <= length2) {
            for (int i = 0; i < length1; i++) {
                if ((bits[i] & bsBits[i]) != 0L) {
                    return true;
                }
            }
        } else {
            for (int i = 0; i < length2; i++) {
                if ((bits[i] & bsBits[i]) != 0L) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Wykonuje logiczn operacj AND na danym obiektu {@code BitSet} i innym obiekcie
     * {@code BitSet}. Wartoci danego obiektu {@code BitSet} s odpowiednio modyfikowane.
     *
     * @param bs
     *            {@code BitSet} bdcy drugim operandem AND.
     * @see #or
     * @see #xor
     */
    public void and(MyBitSet bs) {
        int[] bsBits = bs.bits;
        if (!needClear) {
            return;
        }
        int length1 = actualArrayLength, length2 = bs.actualArrayLength;
        if (length1 <= length2) {
            for (int i = 0; i < length1; i++) {
                bits[i] &= bsBits[i];
            }
        } else {
            for (int i = 0; i < length2; i++) {
                bits[i] &= bsBits[i];
            }
            for (int i = length2; i < length1; i++) {
                bits[i] = 0;
            }
            actualArrayLength = length2;
        }
        isLengthActual = !((actualArrayLength > 0) && (bits[actualArrayLength - 1] == 0));
    }

    /**
     * Zeruje wszystkie bity, ktre s ustawione take w drugim obiekcie
     * {@code BitSet}. Wartoci danego obiektu {@code BitSet} s odpowiednio zmieniane.
     *
     * @param bs
     *            {@code BitSet} bdcy drugim operandem operacji ANDNOT.
     */
    public void andNot(MyBitSet bs) {
        int[] bsBits = bs.bits;
        if (!needClear) {
            return;
        }
        int range = actualArrayLength < bs.actualArrayLength ? actualArrayLength
                : bs.actualArrayLength;
        for (int i = 0; i < range; i++) {
            bits[i] &= ~bsBits[i];
        }

        if (actualArrayLength < range) {
            actualArrayLength = range;
        }
        isLengthActual = !((actualArrayLength > 0) && (bits[actualArrayLength - 1] == 0));
    }

    /**
     * Wykonuje logiczn operacj OR na danym {@code BitSet} i innym {@code BitSet}.
     * Wartoci danego {@code BitSet} s odpowiednio modyfikowane.
     *
     * @param bs
     *            {@code BitSet} bdcy drugim oparandem operacji OR.
     * @see #xor
     * @see #and
     */
    public void or(MyBitSet bs) {
        int bsActualLen = bs.getActualArrayLength();
        if (bsActualLen > bits.length) {
            int[] tempBits = new int[bsActualLen];
            System.arraycopy(bs.bits, 0, tempBits, 0,
                    bs.actualArrayLength);
            for (int i = 0; i < actualArrayLength; i++) {
                tempBits[i] |= bits[i];
            }
            bits = tempBits;
            actualArrayLength = bsActualLen;
            isLengthActual = true;
        } else {
            int[] bsBits = bs.bits;
            for (int i = 0; i < bsActualLen; i++) {
                bits[i] |= bsBits[i];
            }
            if (bsActualLen > actualArrayLength) {
                actualArrayLength = bsActualLen;
                isLengthActual = true;
            }
        }
        needClear();
    }

    /**
     * Wykonuje logiczn operacj XOR na danym {@code BitSet} i innym {@code BitSet}.
     * Wartoci w danym {@code BitSet} s odpowiednio modyfikowane.
     *
     * @param bs
     *            {@code BitSet} bdcy drugim oparandem operacji XOR.
     * @see #or
     * @see #and
     */
    public void xor(MyBitSet bs) {
        int bsActualLen = bs.getActualArrayLength();
        if (bsActualLen > bits.length) {
            int[] tempBits = new int[bsActualLen];
            System.arraycopy(bs.bits, 0, tempBits, 0,
                    bs.actualArrayLength);
            for (int i = 0; i < actualArrayLength; i++) {
                tempBits[i] ^= bits[i];
            }
            bits = tempBits;
            actualArrayLength = bsActualLen;
            isLengthActual = !((actualArrayLength > 0) && (bits[actualArrayLength - 1] == 0));
        } else {
            int[] bsBits = bs.bits;
            for (int i = 0; i < bsActualLen; i++) {
                bits[i] ^= bsBits[i];
            }
            if (bsActualLen > actualArrayLength) {
                actualArrayLength = bsActualLen;
                isLengthActual = true;
            }
        }
        needClear();
    }

    /**
     * Zwraca liczb bitw z danego {@code BitSet}.
     *
     * @return Liczba bitw w danym {@code BitSet}.
     * @see #length
     */
    public int size() {
        return bits.length << OFFSET;
    }

    /**
     * Zwraca liczb bitw do najwyszego ustawionego bitu (z jego uwzgldnieniem).
     *
     * @return Dugo danego {@code BitSet}.
     */
    public int length() {
        int idx = actualArrayLength - 1;
        while (idx >= 0 && bits[idx] == 0) {
            --idx;
        }
        actualArrayLength = idx + 1;
        if (idx == -1) {
            return 0;
        }
        int i = ELM_SIZE - 1;
        long val = bits[idx];
        while ((val & (TWO_N_ARRAY[i])) == 0 && i > 0) {
            i--;
        }
        return (idx << OFFSET) + i + 1;
    }

    private final int getActualArrayLength() {
        if (isLengthActual) {
            return actualArrayLength;
        }
        int idx = actualArrayLength - 1;
        while (idx >= 0 && bits[idx] == 0) {
            --idx;
        }
        actualArrayLength = idx + 1;
        isLengthActual = true;
        return actualArrayLength;
    }

    /**
     * Zwraca acuch ze zwizym opisem zbioru.
     *
     * @return Rozdzielona przecinkami lista indeksw ustawionych bitw.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(bits.length / 2);
        int bitCount = 0;
        sb.append('{');
        boolean comma = false;
        for (int i = 0; i < bits.length; i++) {
            if (bits[i] == 0) {
                bitCount += ELM_SIZE;
                continue;
            }
            for (int j = 0; j < ELM_SIZE; j++) {
                if (((bits[i] & (TWO_N_ARRAY[j])) != 0)) {
                    if (comma) {
                        sb.append(", ");
                    }
                    sb.append(bitCount);
                    comma = true;
                }
                bitCount++;
            }
        }
        sb.append('}');
        return sb.toString();
    }

    /**
     * Zwraca pozycj pierwszego bitu o wartoci {@code true} na pozycji {@code index} lub za ni.
     *
     * @param index
     *            Pocztkowa pozycja (z uwzgldnieniem podanego elementu).
     * @return -1, jeli nie ma bitw {@code true} na pozycji {@code index} lub za ni.
     */
    public int nextSetBit(int index) {
        checkIndex(index);

        if (index >= actualArrayLength << OFFSET) {
            return -1;
        }

        int idx = index >> OFFSET;
        
        if (bits[idx] != 0L) {
            for (int j = index & RIGHT_BITS; j < ELM_SIZE; j++) {
                if (((bits[idx] & (TWO_N_ARRAY[j])) != 0)) {
                    return (idx << OFFSET) + j;
                }
            }

        }
        idx++;
        while (idx < actualArrayLength && bits[idx] == 0L) {
            idx++;
        }
        if (idx == actualArrayLength) {
            return -1;
        }

        // Wiadomo, e istnieje bit o wartoci true, poniewa
        // warto zbioru bitw jest rna od 0L
        for (int j = 0; j < ELM_SIZE; j++) {
            if (((bits[idx] & (TWO_N_ARRAY[j])) != 0)) {
                return (idx << OFFSET) + j;
            }
        }

        return -1;
    }

    /**
     * Zwraca pozycj pierwszego bitu o wartoci {@code false} na pozycji {@code index} lub wikszej.
     *
     * @param index
     *            Pocztkowa pozycja (z uwzgldnieniem podanego elementu).
     * @return Pozycja nastpnego bitu o wartoci {@code false}, nawet jeli wykracza
     *         poza rozmiar obiektu {@code BitSet}.
     */
    public int nextClearBit(int index) {
        checkIndex(index);

        int length = actualArrayLength;
        int bssize = length << OFFSET;
        if (index >= bssize) {
            return index;
        }

        int idx = index >> OFFSET;
        
        if (bits[idx] != (~0L)) {
            for (int j = index % ELM_SIZE; j < ELM_SIZE; j++) {
                if (((bits[idx] & (TWO_N_ARRAY[j])) == 0)) {
                    return idx * ELM_SIZE + j;
                }
            }
        }
        idx++;
        while (idx < length && bits[idx] == (~0L)) {
            idx++;
        }
        if (idx == length) {
            return bssize;
        }

        // Wiadomo, e istnieje bit o wartoci true, 
        // poniewa warto zbioru bitw to nie 0L
        for (int j = 0; j < ELM_SIZE; j++) {
            if (((bits[idx] & (TWO_N_ARRAY[j])) == 0)) {
                return (idx << OFFSET) + j;
            }
        }

        return bssize;
    }

    /**
     * Zwraca true, jeli wszystkie bity w {@code BitSet} maj warto false.
     *
     * @return {@code true}, jeli {@code BitSet} jest pusty.
     *         W przeciwnym razie {@code false}.
     */
    public boolean isEmpty() {
        if (!needClear) {
            return true;
        }
        int length = bits.length;
        for (int idx = 0; idx < length; idx++) {
            if (bits[idx] != 0L) {
                return false;
            }
        }
        return true;
    }

    /**
     * Zwraca liczb bitw o wartoci {@code true} ze zbioru {@code BitSet}.
     *
     * @return Liczba bitw w zbiorze o wartoci {@code true}.
     */
    public int cardinality() {
        if (!needClear) {
            return 0;
        }
        int count = 0;
        int length = bits.length;
        // FIXME: naley przetestowa wydajno; jeli jest niska, naley zastosowa
        // tabel 256-bitow
        for (int idx = 0; idx < length; idx++) {
            count += pop(bits[idx] & 0xffffffffL);
            count += pop(bits[idx] >>> 32);
        }
        return count;
    }

    private final int pop(long x) {
        // BEGIN android-note
        // Oddelegowanie zada do Integer.bitCount(i); rozway zastosowanie kodu natywnego
        // END android-note
        x = x - ((x >>> 1) & 0x55555555);
        x = (x & 0x33333333) + ((x >>> 2) & 0x33333333);
        x = (x + (x >>> 4)) & 0x0f0f0f0f;
        x = x + (x >>> 8);
        x = x + (x >>> 16);
        return (int) x & 0x0000003f;
    }

    private void readObject(ObjectInputStream ois) throws IOException,
            ClassNotFoundException {
        ois.defaultReadObject();
        this .isLengthActual = false;
        this .actualArrayLength = bits.length;
        this .needClear = this .getActualArrayLength() != 0;
    }
}
