//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <iostream>
#include <cstdlib>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"
#include "models.h"

//////////////////////////////////////////////////////////////////////
// nazwa pliku podana w wierszu polecenia
//////////////////////////////////////////////////////////////////////
char *fileName = NULL;

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 4.0f;
GLfloat far = 100.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu skybox'a
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie skybox'a
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateZ = 0.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat modelRotateX = 0.0f;
GLfloat modelRotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    SKYBOX,
    OBJECT_3D,
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    SKYBOX_POSITION,
    SKYBOX_TEX_COORD,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLfloat position[8*3] =
{
    20.0f, 20.0f, 20.0f,
    -20.0f, 20.0f, 20.0f,
    -20.0f, -20.0f, 20.0f,
    20.0f, -20.0f, 20.0f,
    20.0f, 20.0f, -20.0f,
    -20.0f, 20.0f, -20.0f,
    -20.0f, -20.0f, -20.0f,
    20.0f, -20.0f, -20.0f
};

//////////////////////////////////////////////////////////////////////
// skadowe wsprzdnych tekstury szeciennej
//////////////////////////////////////////////////////////////////////
const GLfloat texCoord[8*3] =
{
    1.0f, 1.0f, 1.0f,
    -1.0f, 1.0f, 1.0f,
    -1.0f, -1.0f, 1.0f,
    1.0f, -1.0f, 1.0f,
    1.0f, 1.0f, -1.0f,
    -1.0f, 1.0f, -1.0f,
    -1.0f, -1.0f, -1.0f,
    1.0f, -1.0f, -1.0f
};

//////////////////////////////////////////////////////////////////////
// dane indeksw wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLuint indices[12*3] =
{
    5, 0, 1,
    5, 4, 0,
    2, 0, 3,
    2, 1, 0,
    7, 0, 4,
    7, 3, 0,
    3, 6, 2,
    3, 7, 6,
    1, 2, 6,
    1, 6, 5,
    4, 5, 6,
    4, 6, 7
};

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    CUBEMAP_MASKONAIVE,
    CUBEMAP_TEIDE,
    CUBEMAP_TENERIFE,
    CUBEMAP_VASA,
    TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury
//////////////////////////////////////////////////////////////////////
GLuint texture[TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// numer wybranego obiektu tekstury
//////////////////////////////////////////////////////////////////////
int texNumber = CUBEMAP_MASKONAIVE;

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION  0
#define TEX_COORD 2

//////////////////////////////////////////////////////////////////////
// wspczynnik refrakcji
//////////////////////////////////////////////////////////////////////
GLfloat eta = 0.9f;

//////////////////////////////////////////////////////////////////////
// znacznik odwrcenia wartoci mapy wysokoci
//////////////////////////////////////////////////////////////////////
bool reverseHeightMap = true;

//////////////////////////////////////////////////////////////////////
// obiekt klasy obsugujcej odczyt i rendering plikw 3D
//////////////////////////////////////////////////////////////////////
ModelAndTextureLoader model;

//////////////////////////////////////////////////////////////////////
// funkcja rysujca skybox'a
// modelViewMatrix - macierz modelu-widoku
//////////////////////////////////////////////////////////////////////
void DrawSkybox( glm::mat4x4 &modelViewMatrix )
{
    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // wybr aktywnej jednostki teksturujcej
    glActiveTexture( GL_TEXTURE6 );

    // wybr obiektu tekstury
    glBindTexture( GL_TEXTURE_CUBE_MAP, texture[texNumber] );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program[SKYBOX] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[SKYBOX], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program[SKYBOX], "tex" ), 6 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, 12 * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_CUBE_MAP, 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie obiektu 3D
// objectModelViewMatrix - macierz modelu-widoku obiektu
// skyboxModelViewMatrix - macierz modelu-widoku skybox-a
//////////////////////////////////////////////////////////////////////
void DrawObject3D( glm::mat4x4 &objectModelViewMatrix, glm::mat4x4 &skyboxModelViewMatrix )
{
    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // wybr aktywnej jednostki teksturujcej
    glActiveTexture( GL_TEXTURE6 );

    // wybr obiektu tekstury
    glBindTexture( GL_TEXTURE_CUBE_MAP, texture[texNumber] );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( objectModelViewMatrix ) );

    // przeksztacenie wektora kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition = modelViewMatrixInverse * lightPosition;
    inverseLightPosition = glm::normalize( inverseLightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie programu
    glUseProgram( program[OBJECT_3D] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * objectModelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[OBJECT_3D], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - macierzy modelu-widoku do oblicze odbicia
    glm::mat4x4 modelViewMatrix( glm::inverse( skyboxModelViewMatrix ) * objectModelViewMatrix );
    glUniformMatrix4fv( glGetUniformLocation( program[OBJECT_3D], "modelViewMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewMatrix ) );

    // zaadowanie zmiennej jednorodnej - macierzy przeksztacenia wektora normalnego
    // do oblicze odbicia; przeksztacenia macierzy wektora normalnego, tj. transponowanie
    // i odwracanie, nie s potrzebne z uwagi na to, e jest to macierz ortogonalna
    glm::mat3x3 normalMatrix( modelViewMatrix );
    glUniformMatrix3fv( glGetUniformLocation( program[OBJECT_3D], "normalMatrix" ), 1, GL_FALSE, glm::value_ptr( normalMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[OBJECT_3D], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[OBJECT_3D], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // wspczynnik skalowania wielkoci przemieszczenia
    GLfloat displacement = 0.015f * model.GetSize();
    glUniform1f( glGetUniformLocation( program[OBJECT_3D], "displacement" ), displacement );

    // znacznik odwrcenia wartoci mapy wysokoci
    glUniform1i( glGetUniformLocation( program[OBJECT_3D], "reverseHeightMap" ), reverseHeightMap );

    // przeksztacenie wsprzdnych oka do ukadu wsprzdnych obserwatora
    glm::vec3 eye( 0.0f );
    glUniform3fv( glGetUniformLocation( program[OBJECT_3D], "eye" ), 1, glm::value_ptr( eye ) );

    // wspczynnik refrakcji
    glUniform1f( glGetUniformLocation( program[OBJECT_3D], "eta" ), eta );

    // rendering obiektu 3D z pliku
    model.MeshRendering();

    // wyczenie programu
    glUseProgram( 0 );

    // wybr aktywnej jednostki teksturujcej
    glActiveTexture( GL_TEXTURE6 );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_CUBE_MAP, 0 );
}

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // macierz modelu-widoku skybox'a - macierz jednostkowa
    glm::mat4x4 skyboxModelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora do punktu pocztkowego wewntrz skybox'a
    skyboxModelViewMatrix = glm::translate( skyboxModelViewMatrix, glm::vec3( translateX, 0.0f, translateZ ) );

    // obroty skybox'a
    skyboxModelViewMatrix = glm::rotate( skyboxModelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    skyboxModelViewMatrix = glm::rotate( skyboxModelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // macierz modelu-widoku obiektu = macierz jednostkowa
    glm::mat4x4 objectModelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    objectModelViewMatrix = glm::translate( objectModelViewMatrix, glm::vec3( 0.0f, 0.0f, -10.0f ) );

    // skalowanie obiektu do wielkoci bryy obcinania
    objectModelViewMatrix = glm::scale( objectModelViewMatrix, glm::vec3( (right - left) * 0.8f / model.GetSize() ) );

    // skalowanie obiektu
    objectModelViewMatrix = glm::scale( objectModelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    objectModelViewMatrix = glm::rotate( objectModelViewMatrix, modelRotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    objectModelViewMatrix = glm::rotate( objectModelViewMatrix, modelRotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wycentowanie obiektu
    objectModelViewMatrix = glm::translate( objectModelViewMatrix, -model.GetCenter() );

    // narysowanie skybox'a
    DrawSkybox( skyboxModelViewMatrix );

    // narysowanie obiektu
    DrawObject3D( objectModelViewMatrix, skyboxModelViewMatrix );

    // wypisanie wspczynnika refrakcji materiau
    std::ostringstream txt;
    txt << std::setprecision( 2 ) << std::fixed << "wspczynnik refrakcji materiau: " << eta;
    DrawText8x16( 3, 3, txt.str(), glm::vec4( 1.0f ) );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // zaadowanie danych z pliku 3D
    if( !model.Load( fileName ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << fileName << std::endl;
        exit( 0 );
    }

    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SKYBOX] = glCreateProgram();
    glAttachShader( program[SKYBOX], LoadShader( GL_VERTEX_SHADER, "skybox_vs.glsl" ) );
    glAttachShader( program[SKYBOX], LoadShader( GL_FRAGMENT_SHADER, "skybox_fs.glsl" ) );
    LinkValidateProgram( program[SKYBOX] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[OBJECT_3D] = glCreateProgram();
    glAttachShader( program[OBJECT_3D], LoadShader( GL_VERTEX_SHADER, "dyspersja_vs.glsl" ) );
    glAttachShader( program[OBJECT_3D], LoadShader( GL_TESS_CONTROL_SHADER, "dyspersja_tcs.glsl" ) );
    glAttachShader( program[OBJECT_3D], LoadShader( GL_TESS_EVALUATION_SHADER, "dyspersja_tes.glsl" ) );
    glAttachShader( program[OBJECT_3D], LoadShader( GL_FRAGMENT_SHADER, "dyspersja_fs.glsl" ) );
    LinkValidateProgram( program[OBJECT_3D] );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[SKYBOX_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[SKYBOX_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[SKYBOX_TEX_COORD] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[SKYBOX_TEX_COORD] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoord ), texCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( indices ), indices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // nazwy plikw z obrazami trzech tekstur szeciennych
    // dane pochodz ze strony WWW: http://www.humus.name
    const std::string fileNames[6*TEXTURE_SIZE] =
    {
        "../../media/www.humus.name/maskonaive/posx.jpg",
        "../../media/www.humus.name/maskonaive/negx.jpg",
        "../../media/www.humus.name/maskonaive/posy.jpg",
        "../../media/www.humus.name/maskonaive/negy.jpg",
        "../../media/www.humus.name/maskonaive/posz.jpg",
        "../../media/www.humus.name/maskonaive/negz.jpg",

        "../../media/www.humus.name/teide/posx.jpg",
        "../../media/www.humus.name/teide/negx.jpg",
        "../../media/www.humus.name/teide/posy.jpg",
        "../../media/www.humus.name/teide/negy.jpg",
        "../../media/www.humus.name/teide/posz.jpg",
        "../../media/www.humus.name/teide/negz.jpg",

        "../../media/www.humus.name/tenerife/posx.jpg",
        "../../media/www.humus.name/tenerife/negx.jpg",
        "../../media/www.humus.name/tenerife/posy.jpg",
        "../../media/www.humus.name/tenerife/negy.jpg",
        "../../media/www.humus.name/tenerife/posz.jpg",
        "../../media/www.humus.name/tenerife/negz.jpg",

        "../../media/www.humus.name/vasa/posx.jpg",
        "../../media/www.humus.name/vasa/negx.jpg",
        "../../media/www.humus.name/vasa/posy.jpg",
        "../../media/www.humus.name/vasa/negy.jpg",
        "../../media/www.humus.name/vasa/posz.jpg",
        "../../media/www.humus.name/vasa/negz.jpg"
    };

    // kolejne obrazy tekstury szeciennej
    const GLenum targets[6] =
    {
        GL_TEXTURE_CUBE_MAP_POSITIVE_X,
        GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
        GL_TEXTURE_CUBE_MAP_POSITIVE_Y,
        GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
        GL_TEXTURE_CUBE_MAP_POSITIVE_Z,
        GL_TEXTURE_CUBE_MAP_NEGATIVE_Z
    };

    // generowanie trzech identyfikatorw obiektw tekstury
    glGenTextures( TEXTURE_SIZE, texture );

    // utworzenie trzech tekstur szeciennych
    for( int i = 0; i < TEXTURE_SIZE; i++ )
    {
        // utworzenie obiektu tekstury
        glBindTexture( GL_TEXTURE_CUBE_MAP, texture[i] );

        // odczyt kolejnych plikw z tekstur i ewentualny komunikat o bdzie
        for( int j = 0; j < 6; j++ )
            if( !LoadTexture( fileNames[i*6+j].c_str(), targets[j] ) )
            {
                std::cout << "Niepoprawny odczyt pliku " << fileNames[i*6+j] << std::endl;
                exit( 0 );
            }

        // filtr powikszajcy i pomniejszajcy
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

        // tryb zawijania
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE );
    }

    // przeczenie na tekstur domyln
    glBindTexture( GL_TEXTURE_CUBE_MAP, 0 );

    // bezszwowa filtracja tekstury szeciennej
    glEnable( GL_TEXTURE_CUBE_MAP_SEAMLESS );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[SKYBOX] );
    glDeleteProgram( program[OBJECT_3D] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteTextures( TEXTURE_SIZE, texture );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
