//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "newell_teapot.h"
#include "newell_teacup.h"
#include "newell_teaspoon.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 10.0f;
GLfloat far = 18.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu kierunku wiata
//////////////////////////////////////////////////////////////////////
GLfloat rotateLightX = 0.0f;
GLfloat rotateLightY = 0.0f;

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy wierzchokw ze
// wsprzdnymi wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablic
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// parametr regulujcy jako generowanej powierzchni
//////////////////////////////////////////////////////////////////////
float DENSITY = 8;

//////////////////////////////////////////////////////////////////////
// numer rysowanego obiektu: 0 - czajnik, 1 - filiaka, 2 - yeczka
//////////////////////////////////////////////////////////////////////
int object = 0;

//////////////////////////////////////////////////////////////////////
// liczba wierzchokw uywanych przy renderingu wybranego obiektu
//////////////////////////////////////////////////////////////////////
const int objectIndicesCount[3] =
{
    TEAPOT_PATCHES_COUNT,
    TEACUP_PATCHES_COUNT,
    TEASPOON_PATCHES_COUNT
};

//////////////////////////////////////////////////////////////////////
// przesunicie danych wsprzdnych punktw kontrolnych
// w obiekcie bufora wierzchokw
//////////////////////////////////////////////////////////////////////
const int vertexDataOffset[3] =
{
    0,
    TEAPOT_VERTICES_COUNT / 3,
    (TEAPOT_VERTICES_COUNT + TEACUP_VERTICES_COUNT) / 3
};

//////////////////////////////////////////////////////////////////////
// przesunicie danych indeksw wsprzdnych punktw kontrolnych
// w obiekcie bufora indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
const int indicesDataOffset[3] =
{
    0,
    TEAPOT_PATCHES_COUNT,
    TEAPOT_PATCHES_COUNT + TEACUP_PATCHES_COUNT
};

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // obroty wektora kierunku wiata
    glm::mat4x4 rotateLightDir = glm::mat4x4( 1.0 );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightY, glm::vec3( 0.0f, 1.0f, 0.0f ) );
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = rotateLightDir * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition( modelViewMatrixInverse * lightPosition );
    inverseLightPosition = glm::normalize( inverseLightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // okrelenie poziomw teselacji
    GLfloat tessLevel[4] = { DENSITY, DENSITY, DENSITY, DENSITY };
    glPatchParameterfv( GL_PATCH_DEFAULT_OUTER_LEVEL, tessLevel );
    glPatchParameterfv( GL_PATCH_DEFAULT_INNER_LEVEL, tessLevel );
 
    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program, "lightSource[0].position" ), 1, glm::value_ptr( inverseLightPosition ) );
    glUniform4fv( glGetUniformLocation( program, "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program, "material" ), material );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElementsBaseVertex( GL_PATCHES, objectIndicesCount[object], GL_UNSIGNED_INT,
                              reinterpret_cast< GLvoid* >( indicesDataOffset[object] * sizeof( GLuint ) ),
                              vertexDataOffset[object] - 1 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wypisanie stopnia teselacji
    std::ostringstream txt;
    txt << "poziom teselacji: " << (int)DENSITY << "" << (int)DENSITY;
    DrawText8x16( 3, 21, txt.str() );

    // wypisanie nazwy materiau
    DrawText8x16( 3, 3, std::string( "materia: " ) + GetMaterialName( material ) );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "powierzchnia_beziera_teselacja_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_TESS_EVALUATION_SHADER, "powierzchnia_beziera_teselacja_tes.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "powierzchnia_beziera_teselacja_fs.glsl" ) );
    LinkValidateProgram( program );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
    glBufferData( GL_ARRAY_BUFFER, (TEAPOT_VERTICES_COUNT + TEACUP_VERTICES_COUNT + TEASPOON_VERTICES_COUNT) * sizeof( GLfloat ),
                  NULL, GL_STATIC_DRAW );
    glBufferSubData( GL_ARRAY_BUFFER, 0, TEAPOT_VERTICES_COUNT * sizeof( GLfloat ), teapotVertices );
    glBufferSubData( GL_ARRAY_BUFFER, TEAPOT_VERTICES_COUNT * sizeof( GLfloat ), TEACUP_VERTICES_COUNT * sizeof( GLfloat ), teacupVertices );
    glBufferSubData( GL_ARRAY_BUFFER, (TEAPOT_VERTICES_COUNT + TEACUP_VERTICES_COUNT) * sizeof( GLfloat ),
                     TEASPOON_VERTICES_COUNT * sizeof( GLfloat ), teaspoonVertices );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, (TEAPOT_PATCHES_COUNT + TEACUP_PATCHES_COUNT + TEASPOON_PATCHES_COUNT) * sizeof( GLuint ),
                  NULL, GL_STATIC_DRAW );
    glBufferSubData( GL_ELEMENT_ARRAY_BUFFER, 0, TEAPOT_PATCHES_COUNT * sizeof( GLuint ), teapotPatches );
    glBufferSubData( GL_ELEMENT_ARRAY_BUFFER, TEAPOT_PATCHES_COUNT * sizeof( GLuint ), TEACUP_PATCHES_COUNT * sizeof( GLuint ), teacupPatches );
    glBufferSubData( GL_ELEMENT_ARRAY_BUFFER, (TEAPOT_PATCHES_COUNT + TEACUP_PATCHES_COUNT) * sizeof( GLuint ),
                     TEASPOON_PATCHES_COUNT * sizeof( GLuint ), teaspoonPatches );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wielko wejciowego pata wierzchokw
    glPatchParameteri( GL_PATCH_VERTICES, 16 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // dane obsugujcy gwny obiekt programu
    glDeleteProgram( program );
    glDeleteBuffers( 1, &vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( 1, &vertexArray );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
