//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include <GL/glut.h>
#include "text.h"
#include "sphere.h"

//////////////////////////////////////////////////////////////////////
// pooenie kursora myszy
//////////////////////////////////////////////////////////////////////
extern int buttonX, buttonY;

//////////////////////////////////////////////////////////////////////
// wskanik nacinicia lewego przycisku myszy
//////////////////////////////////////////////////////////////////////
extern int buttonState;

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
#ifdef near
#undef near
#endif
#ifdef far
#undef far
#endif
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 6.0f;
GLfloat far = 14.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0
#define NORMAL 1

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora ramki
//////////////////////////////////////////////////////////////////////
GLuint frameBuffer;

//////////////////////////////////////////////////////////////////////
// rozmiary obszarw renderingu do tekstury
//////////////////////////////////////////////////////////////////////
const int FRAME_WIDTH = 4*256;
const int FRAME_HEIGHT = 4*256;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    COLOR0,
    COLOR1,
    DEPTH,
    RENDER_TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury - bufory koloru i gbokoci
//////////////////////////////////////////////////////////////////////
GLuint renderTexture[RENDER_TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// rozmiary obszaru renderingu okna programu
//////////////////////////////////////////////////////////////////////
GLuint windowWidth, windowHeight;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // bufor ramki do zapisu = obiekt bufora ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );

    // wskazanie buforw koloru do zapisu
    GLenum bufs[2] = { GL_COLOR_ATTACHMENT0, GL_COLOR_ATTACHMENT1 };
    glDrawBuffers( 2, bufs );

    // czyszczenie bufora gbokoci
    glClear( GL_DEPTH_BUFFER_BIT );

    // czyszczenie obu buforw koloru
    const GLint black[4] = { 0, 0, 0, 0 };
    const GLfloat white[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    glClearBufferfv( GL_COLOR, COLOR0, white );
    glClearBufferiv( GL_COLOR, COLOR1, black );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program, "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program, "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElementsInstanced( GL_TRIANGLES, SPHERE_MID_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL, 5 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // bufor ramki do odczytu = obiekt bufora ramki
    glBindFramebuffer( GL_READ_FRAMEBUFFER, frameBuffer );

    // odczyt z drugiego bufora koloru
    glReadBuffer( GL_COLOR_ATTACHMENT1 );

    // pobranie informacji o selekcji obiektu
    GLint pixelTest;
    if( buttonState == GLUT_DOWN )
    {
        // obliczenie pooenia kursora myszy we wsprzdnych bufora ramki
        int frameX = static_cast<int>( buttonX * FRAME_WIDTH / static_cast<float>( windowWidth ) );
        int frameY = static_cast<int>( (windowHeight - buttonY) * FRAME_HEIGHT / static_cast<float>( windowHeight ) );

        // pobranie piksela odpowiadajcego pooeniu kursora myszy
        glReadPixels( frameX, frameY, 1, 1, GL_RED_INTEGER, GL_INT, &pixelTest );
    }

    // odczyt z pierwszego bufora koloru
    glReadBuffer( GL_COLOR_ATTACHMENT0 );

    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // kopiowanie buforw - cae okno
    glBlitFramebuffer( 0, 0, FRAME_WIDTH, FRAME_HEIGHT,
                       0, 0, windowWidth, windowHeight,
                       GL_COLOR_BUFFER_BIT, GL_LINEAR );

    // wyczenie testu bufora gbokoci
    glDisable( GL_DEPTH_TEST );

    // wywietlenie informacji o wybranym obiekcie
    switch( pixelTest )
    {
        case 0: DrawText8x16( 3, 3, "to (0)" ); break;
        case 1: DrawText8x16( 3, 3, "czerwona kula (1)" ); break;
        case 2: DrawText8x16( 3, 3, "zielona kula (2)" ); break;
        case 3: DrawText8x16( 3, 3, "niebieska kula (3)" ); break;
        case 4: DrawText8x16( 3, 3, "ta kula (4)" ); break;
        case 5: DrawText8x16( 3, 3, "rowa kula (5)" ); break;
    }

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewportIndexedf( 0, 0.0, 0.0, static_cast<GLfloat>( width ), static_cast<GLfloat>( height ) );

    // zapisanie rozmiarw obszaru okna renderingu
    windowWidth = width;
    windowHeight = height;

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // utworzenie obiektw tekstur - na dane buforw koloru i bufora gbokoci
    glGenTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glBindTexture( GL_TEXTURE_2D, renderTexture[COLOR0] );
    glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, FRAME_WIDTH, FRAME_HEIGHT, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    glBindTexture( GL_TEXTURE_2D, renderTexture[COLOR1] );
    glTexImage2D( GL_TEXTURE_2D, 0, GL_R32I, FRAME_WIDTH, FRAME_HEIGHT, 0, GL_RED_INTEGER, GL_INT, NULL );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
    glBindTexture( GL_TEXTURE_2D, renderTexture[DEPTH] );
    glTexImage2D( GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT, FRAME_WIDTH, FRAME_HEIGHT, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // utworzenie obiektu bufora ramki i doczenie obiektw tekstury
    glGenFramebuffers( 1, &frameBuffer );
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, renderTexture[COLOR0], 0 );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT1, renderTexture[COLOR1], 0 );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, renderTexture[DEPTH], 0 );

    // sprawdzenie poprawnoci i kompletnoci obiektu bufora ramki
    GLenum error = glCheckFramebufferStatus( GL_DRAW_FRAMEBUFFER );
    if( error != GL_FRAMEBUFFER_COMPLETE )
    {
        std::cout << "Niepoprawny obiekt bufora ramki" << std::endl;
        exit( 0 );
    }

    // obszar renderingu - caa powierzchnia bufora ramki
    glViewportIndexedf( 1, 0.0, 0.0, static_cast<GLfloat>( FRAME_WIDTH ), static_cast<GLfloat>( FRAME_HEIGHT ) );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "selekcja_obiektow_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_GEOMETRY_SHADER, "selekcja_obiektow_gs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "selekcja_obiektow_fs.glsl" ) );

    // wskazanie numerw zmiennych wyjciowych shadera fragmentu
    glBindFragDataLocation( program, COLOR0, "outColor" );
    glBindFragDataLocation( program, COLOR1, "outObject" );

    // konsolidacja i walidacja programu
    LinkValidateProgram( program );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
    glBufferData( GL_ARRAY_BUFFER, sizeof( sphereMidPositionNormal ), sphereMidPositionNormal, GL_STATIC_DRAW );
    GLuint positionLoc = glGetAttribLocation( program, "inPosition" );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( sphereMidIndices ), sphereMidIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteBuffers( 1, &vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glDeleteFramebuffers( 1, &frameBuffer );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
