//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <sstream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// liczba warstw renderingu
//////////////////////////////////////////////////////////////////////
const int FRAME_LAYERS = 16;

//////////////////////////////////////////////////////////////////////
// tablica macierzy rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix[FRAME_LAYERS];

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.5f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    DEPTH_OF_FIELD,         // efekt gbi ostroci
    POSTPROCESSING_RECT,    // postprocesing
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION_TEAPOT,
    NORMAL_TEAPOT,
    POSITION_RECT,
    TEXCOORD_RECT,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw:
// wsprzdnymi wierzchokw, wektorw normalnych i tekstury
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL,
    TEX_COORD
};

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    RECTANGLE,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat positionRect [4*2] =
{
    -1.0f, -1.0f,
    1.0f, -1.0f,
    -1.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury w wierzchokach trjktw
// skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat texCoordRect [4*2] =
{
    0.0f, 0.0f,
    1.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora ramki
//////////////////////////////////////////////////////////////////////
GLuint frameBuffer;

//////////////////////////////////////////////////////////////////////
// rozmiary obszarw renderingu do tekstury
//////////////////////////////////////////////////////////////////////
const int FRAME_WIDTH = 4*256;
const int FRAME_HEIGHT = 4*256;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    COLOR0,
    DEPTH,
    RENDER_TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury - bufor koloru i gbokoci
//////////////////////////////////////////////////////////////////////
GLuint renderTexture[RENDER_TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // bufor ramki do zapisu = obiekt bufora ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // wczenie programu
    glUseProgram( program[DEPTH_OF_FIELD] );

    // zaadowanie zmiennej jednorodnej - iloczynw macierzy modelu-widoku i rzutowania
    for( int layer = 0; layer < FRAME_LAYERS; layer++ )
    {
        std::ostringstream txt;
        txt << "modelViewProjectionMatrix[" << layer << "]";
        glm::mat4x4 modelViewProjectionMatrix = projectionMatrix[layer] * modelViewMatrix;
        glUniformMatrix4fv( glGetUniformLocation( program[DEPTH_OF_FIELD], txt.str().c_str() ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );
    }

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[DEPTH_OF_FIELD], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[DEPTH_OF_FIELD], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElementsInstanced( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // czyszczenie bufora koloru
    glClear( GL_COLOR_BUFFER_BIT );

    // wyczenie testu bufora gbokoci
    glDisable( GL_DEPTH_TEST );

    // wybr tablicy tekstur 2D
    glBindTexture( GL_TEXTURE_2D_ARRAY, renderTexture[COLOR0] );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[RECTANGLE] );

    // wczenie programu
    glUseProgram( program[POSTPROCESSING_RECT] );

    // zaadowanie zmiennej jednorodnej - uchwyt tekstury
    glUniform1i( glGetUniformLocation( program[POSTPROCESSING_RECT], "tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie tablicy tekstur 2D
    glBindTexture( GL_TEXTURE_2D_ARRAY, 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewportIndexedf( 0, 0.0, 0.0, static_cast<GLfloat>( width ), static_cast<GLfloat>( height ) );

    // wsprzdne fluktuacji (rdo Red Book)
    const float jitterPoints[16][2] =
    {
        {  0.000000f,  0.000000f },
        {  0.285561f,  0.188437f },
        {  0.360176f, -0.065688f },
        { -0.111751f,  0.275019f },
        { -0.055918f, -0.215197f },
        { -0.080231f, -0.470965f },
        {  0.138721f,  0.409168f },
        {  0.384120f,  0.458500f },
        { -0.454968f,  0.134088f },
        {  0.179271f, -0.331196f },
        { -0.307049f, -0.364927f },
        {  0.105354f, -0.010099f },
        { -0.154180f,  0.021794f },
        { -0.370135f, -0.116425f },
        {  0.451636f, -0.300013f },
        { -0.370610f,  0.387504f }
    };

    // staa regulujca stopie fluktuacji w gbi ostroci
    const GLfloat focus = 20.0;

    // zaadowanie ustawie tablicy z macierzami rzutowania
    for( int layer = 1; layer < FRAME_LAYERS; layer++ )
    {
        // obliczenie wspczynnikw fluktuacji w efekcie gbi ostroci
        GLfloat dx = -0.33f * jitterPoints[layer][0] * near / focus;
        GLfloat dy = -0.33f * jitterPoints[layer][1] * near / focus;

        // parametry bryy obcinania - rzutowanie perspektywiczne
        // wysoko okna wiksza od szerokoci okna
        if( width < height && width > 0 )
             projectionMatrix[layer] = glm::frustum( left + dx, right + dx, (bottom + dy)*height/width, (top + dy)*height/width, near, far );
        else
            // szeroko okna wiksza lub rwna wysokoci okna
            if (width >= height && height > 0)
                projectionMatrix[layer] = glm::frustum( (left + dx)*width/height, (right + dx)*width/height, bottom + dy, top + dy, near, far );
            else
                projectionMatrix[layer] = glm::frustum( left + dx, right + dx, bottom + dy, top + dy, near, far );
    }
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // utworzenie obiektw tekstur - na dane bufora koloru i bufora gbokoci
    glGenTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glBindTexture( GL_TEXTURE_2D_ARRAY, renderTexture[COLOR0] );
    glTexImage3D( GL_TEXTURE_2D_ARRAY, 0, GL_RGBA, FRAME_WIDTH, FRAME_HEIGHT, FRAME_LAYERS, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
    glBindTexture( GL_TEXTURE_2D_ARRAY, renderTexture[DEPTH] );
    glTexImage3D( GL_TEXTURE_2D_ARRAY, 0, GL_DEPTH_COMPONENT, FRAME_WIDTH, FRAME_HEIGHT, FRAME_LAYERS, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL );
    glTexParameteri( GL_TEXTURE_2D_ARRAY, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // utworzenie obiektu bufora ramki i doczenie obiektw tekstury
    glGenFramebuffers( 1, &frameBuffer );
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, renderTexture[COLOR0], 0 );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, renderTexture[DEPTH], 0 );

    // sprawdzenie poprawnoci i kompletnoci obiektu bufora ramki
    GLenum error = glCheckFramebufferStatus( GL_DRAW_FRAMEBUFFER );
    if( error != GL_FRAMEBUFFER_COMPLETE )
    {
        std::cout << "Niepoprawny obiekt bufora renderingu" << std::endl;
        exit( 0 );
    }

    // obszar renderingu - caa powierzchnia bufora ramki
    glViewportIndexedf( 1, 0.0, 0.0, static_cast<GLfloat>( FRAME_WIDTH ), static_cast<GLfloat>( FRAME_HEIGHT ) );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[DEPTH_OF_FIELD] = glCreateProgram();
    glAttachShader( program[DEPTH_OF_FIELD], LoadShader( GL_VERTEX_SHADER, "glebia_ostrosci_vs.glsl" ) );
    glAttachShader( program[DEPTH_OF_FIELD], LoadShader( GL_GEOMETRY_SHADER, "glebia_ostrosci_gs.glsl" ) );
    glAttachShader( program[DEPTH_OF_FIELD], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[DEPTH_OF_FIELD], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[DEPTH_OF_FIELD], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[DEPTH_OF_FIELD], LoadShader( GL_FRAGMENT_SHADER, "glebia_ostrosci_fs.glsl" ) );
    LinkValidateProgram( program[DEPTH_OF_FIELD] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[POSTPROCESSING_RECT] = glCreateProgram();
    glAttachShader( program[POSTPROCESSING_RECT], LoadShader( GL_VERTEX_SHADER, "prostakat_vs.glsl" ) );
    glAttachShader( program[POSTPROCESSING_RECT], LoadShader( GL_FRAGMENT_SHADER, "prostakat_fs.glsl" ) );
    LinkValidateProgram( program[POSTPROCESSING_RECT] );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT] );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[RECTANGLE] );
    glBindVertexArray( vertexArray[RECTANGLE] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionRect ), positionRect, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEXCOORD_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEXCOORD_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoordRect ), texCoordRect, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[DEPTH_OF_FIELD] );
    glDeleteProgram( program[POSTPROCESSING_RECT] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glDeleteFramebuffers( 1, &frameBuffer );
}
