//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// GLSL 1.50
//////////////////////////////////////////////////////////////////////
#version 150

//////////////////////////////////////////////////////////////////////
// struktura opisujca parametry rda wiata
//////////////////////////////////////////////////////////////////////
struct LightSourceParameters
{
    vec4 ambient;               // intensywno wiata otoczenia; oznaczenie La
    vec4 diffuse;               // intensywno wiata rozproszonego; oznaczenie Ld
    vec4 specular;              // intensywno wiata zwierciadlanego; oznaczenie Ls
    vec4 position;              // pooenie rda wiata punktowego (position.w = 1.0)
                                // lub wektor kierunku wiata kierunkowego, skierowany
                                // do rda wiata (position.w = 0.0)
    float constantAttenuation;  // stay wspczynnik tumienia wiata; oznaczenie kc
    float linearAttenuation;    // liniowy wspczynnik tumienia wiata; oznaczenie kl
    float quadraticAttenuation; // kwadratowy wspczynnik tumienia wiata; oznaczenie kq
};

//////////////////////////////////////////////////////////////////////
// prototypy funkcji
//////////////////////////////////////////////////////////////////////
LightSourceParameters LightSource( const int light );
vec4 EyePosition();
vec4 GlobalAmbientLight( const vec4 ambient );
vec4 LocalAmbientLight( const int light, const vec4 ambient );

//////////////////////////////////////////////////////////////////////
// wiato kierunkowe w modelu Blinna-Phonga
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// wektor kierunku wiata lightSource[light].position.xyz musi by
// znormalizowany i wyraony w tym samym ukadzie wsprzdnych co wektor
// normalny i wsprzdne wierzchoka; ponadto wektor ten
// musi by zorientowany w kierunku do rda wiata
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
// specular - wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
// shininess - wykadnik wspczynnika funkcji rozbysku;
//             zakres wartoci <0;128>; oznaczenie Msh
//////////////////////////////////////////////////////////////////////
vec4 BlinnPhongDirectionalLight( const int light, const vec3 normal, const vec3 position,
                                 const vec4 diffuse, const vec4 specular, const float shininess )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, lightSource.position.xyz );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
    {
        // obliczanie wektora powkowego - wektora pomidzy wektorem kierunku
        // wiata i wektorem biegncym od wierzchoka do obserwatora
        vec3 halfVec = normalize( lightSource.position.xyz + normalize( EyePosition().xyz - position ) );

        // obliczenie cosinusa kta pomidzy wektorem normalnym i wektorem powkowym
        float NdotH = max( dot( normal, halfVec ), 0.0 );

        // kocowe obliczenie skadowych owietlenia
        return NdotL * diffuse * lightSource.diffuse +
                specular * lightSource.specular * pow( NdotH, shininess );
    }
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// wiato punktowe w modelu Blinna-Phonga
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// pooenie rda wiata lightSource[light].position.xyz musi by
// wyraone w tym samym ukadzie wsprzdnych co wektor normalny
// i wsprzdne wierzchoka
// ambient - wspczynnik odbicia wiata otoczenia; oznaczenie Ma
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
// specular - wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
// shininess - wykadnik wspczynnika funkcji rozbysku;
//             zakres wartoci <0;128>; oznaczenie Msh
//////////////////////////////////////////////////////////////////////
vec4 BlinnPhongPointLight( const int light, const vec3 normal, const vec3 position,
                           const vec4 ambient, const vec4 diffuse, const vec4 specular,
                           const float shininess )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // wektor z biecego wierzchoka do rda wiata - wektor kierunku rda wiata
    vec3 lightVec = lightSource.position.xyz - position;

    // obliczenie odlegoci od biecego wierzchoka do rda wiata
    float distance = length( lightVec );

    // normalizacja wektora kierunku rda wiata
    lightVec = normalize( lightVec );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, lightVec );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
    {
        // obliczenie wspczynnika tumienia wiata
        float att = 1.0 / ( lightSource.constantAttenuation +
                            lightSource.linearAttenuation * distance +
                            lightSource.quadraticAttenuation * distance * distance );

        // obliczanie wektora powkowego - wektora pomidzy wektorem kierunku
        // wiata i wektorem biegncym od wierzchoka do obserwatora
        vec3 halfVec = normalize( lightVec + normalize( EyePosition().xyz - position ) );

        // obliczenie cosinusa kta pomidzy wektorem normalnym i wektorem powkowym
        float NdotH = max( dot( normal, halfVec ), 0.0 );

        // kocowe obliczenie skadowych owietlenia
        return att * ( LocalAmbientLight( light, ambient ) +
                NdotL * diffuse * lightSource.diffuse +
                specular * lightSource.specular * pow( NdotH, shininess ) );
    }
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// owietlenie Blinna-Phonga
// wymogi dotyczce parametrw jak w opisach funkcji:
// BlinnPhongDirectionalLight i BlinnPhongPointLight
//////////////////////////////////////////////////////////////////////
vec4 BlinnPhongLight( const int light, const vec3 normal, const vec3 position,
                        const vec4 ambient, const vec4 diffuse,
                        const vec4 specular, const float shininess )
{
    // wiato kierunkowe
    if( LightSource( light ).position.w == 0.0 )
        return BlinnPhongDirectionalLight( light, normal, position, diffuse, specular, shininess ) +
                LocalAmbientLight( light, ambient ) + GlobalAmbientLight( ambient );
    else
    // wiato punktowe
        return BlinnPhongPointLight( light, normal, position, ambient, diffuse, specular, shininess ) +
                GlobalAmbientLight( ambient );
}

//////////////////////////////////////////////////////////////////////
// wiato kierunkowe w modelu Blinna-Phonga z przemieszczeniem
// wektora/kierunku wiata
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// wektor kierunku wiata lightSource[light].position.xyz musi by
// znormalizowany i wyraony w tym samym ukadzie wsprzdnych co wektor
// normalny i wsprzdne wierzchoka; ponadto wektor ten
// musi by zorientowany w kierunku do rda wiata
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
// specular - wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
// shininess - wykadnik wspczynnika funkcji rozbysku;
//             zakres wartoci <0;128>; oznaczenie Msh
// TBN - macierz przeksztacenia wektora/kierunku wiata
//////////////////////////////////////////////////////////////////////
vec4 BlinnPhongDirectionalLight( const int light, const vec3 normal, const vec3 position,
                                 const vec4 diffuse, const vec4 specular, const float shininess,
                                 const mat3 TBN )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, TBN * lightSource.position.xyz );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
    {
        // obliczanie wektora powkowego - wektora pomidzy wektorem kierunku
        // wiata i wektorem biegncym od wierzchoka do obserwatora
        vec3 halfVec = TBN * normalize( lightSource.position.xyz + normalize( EyePosition().xyz - position ) );

        // obliczenie cosinusa kta pomidzy wektorem normalnym i wektorem powkowym
        float NdotH = max( dot( normal, halfVec ), 0.0 );

        // kocowe obliczenie skadowych owietlenia
        return NdotL * diffuse * lightSource.diffuse +
                specular * lightSource.specular * pow( NdotH, shininess );
    }
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// wiato punktowe w modelu Blinna-Phonga z przemieszczeniem
// wektora/kierunku wiata
// light - numer rda wiata
// normal - znormalizowany wektor normalny
// position - wsprzdne wierzchoka
// pooenie rda wiata lightSource[light].position.xyz musi by
// wyraone w tym samym ukadzie wsprzdnych co wektor normalny
// i wsprzdne wierzchoka
// ambient - wspczynnik odbicia wiata otoczenia; oznaczenie Ma
// diffuse - wspczynnik odbicia wiata rozproszonego; oznaczenie Md
// specular - wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
// shininess - wykadnik wspczynnika funkcji rozbysku;
//             zakres wartoci <0;128>; oznaczenie Msh
// TBN - macierz przeksztacenia wektora/kierunku wiata
//////////////////////////////////////////////////////////////////////
vec4 BlinnPhongPointLight( const int light, const vec3 normal, const vec3 position,
                           const vec4 ambient, const vec4 diffuse, const vec4 specular,
                           const float shininess, const mat3 TBN )
{
    // pobranie parametrw danego rda wiata
    LightSourceParameters lightSource = LightSource( light );

    // wektor z biecego wierzchoka do rda wiata - wektor kierunku rda wiata
    vec3 lightVec = lightSource.position.xyz - position;

    // obliczenie odlegoci od biecego wierzchoka do rda wiata
    float distance = length( lightVec );

    // normalizacja wektora kierunku rda wiata
    lightVec = TBN * normalize( lightVec );

    // obliczanie cosinusa kta odbicia wiata
    float NdotL = dot( normal, lightVec );

    // obliczanie wiata odbitego tylko dla powierzchni widocznej;
    // warto 0.0 kta odbicia oznacza, e kt odbicia jest w przedziale <0;90)
    if( NdotL > 0.0 )
    {
        // obliczenie wspczynnika tumienia wiata
        float att = 1.0 / ( lightSource.constantAttenuation +
                            lightSource.linearAttenuation * distance +
                            lightSource.quadraticAttenuation * distance * distance );

        // obliczanie wektora powkowego - wektora pomidzy wektorem kierunku
        // wiata i wektorem biegncym od wierzchoka do obserwatora
        vec3 halfVec = TBN * normalize( lightVec + normalize( EyePosition().xyz - position ) );

        // obliczenie cosinusa kta pomidzy wektorem normalnym i wektorem powkowym
        float NdotH = max( dot( normal, halfVec ), 0.0 );

        // kocowe obliczenie skadowych owietlenia
        return att * ( LocalAmbientLight( light, ambient ) +
                NdotL * diffuse * lightSource.diffuse +
                specular * lightSource.specular * pow( NdotH, shininess ) );
    }
    else
        // dla NdotL = 0.0 powierzchnia nie jest widoczna
        return vec4( 0.0 );
}

//////////////////////////////////////////////////////////////////////
// owietlenie Blinna-Phonga z przemieszczeniem wektora/kierunku wiata
// wymogi dotyczce parametrw jak w opisach funkcji:
// BlinnPhongDirectionalLight i BlinnPhongPointLight
// TBN - macierz przeksztacenia wektora/kierunku wiata
//////////////////////////////////////////////////////////////////////
vec4 BlinnPhongLight( const int light, const vec3 normal, const vec3 position,
                        const vec4 ambient, const vec4 diffuse,
                        const vec4 specular, const float shininess,
                        const mat3 TBN )
{
    // wiato kierunkowe
    if( LightSource( light ).position.w == 0.0 )
        return BlinnPhongDirectionalLight( light, normal, position, diffuse, specular, shininess, TBN ) +
                LocalAmbientLight( light, ambient ) + GlobalAmbientLight( ambient );
    else
    // wiato punktowe
        return BlinnPhongPointLight( light, normal, position, ambient, diffuse, specular, shininess, TBN ) +
                GlobalAmbientLight( ambient );
}
