//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "newell_teapot.h"
#include "newell_teacup.h"
#include "newell_teaspoon.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 10.0f;
GLfloat far = 18.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu kierunku wiata
//////////////////////////////////////////////////////////////////////
GLfloat rotateLightX = 0.0f;
GLfloat rotateLightY = 0.0f;

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy wierzchokw ze
// wsprzdnymi wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablic
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// parametr regulujcy jako generowanej powierzchni
//////////////////////////////////////////////////////////////////////
const int DENSITY = 8;

//////////////////////////////////////////////////////////////////////
// liczba danych w tablicy indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
const int indicesCount = 2 * (DENSITY + 1) * DENSITY + DENSITY - 1;

//////////////////////////////////////////////////////////////////////
// numer rysowanego obiektu: 0 - czajnik, 1 - filiaka, 2 - yeczka
//////////////////////////////////////////////////////////////////////
int object = 0;

//////////////////////////////////////////////////////////////////////
// liczba instancji uywanych przy renderingu wybranego obiektu
//////////////////////////////////////////////////////////////////////
const int objectInstancedCount[3] =
{
    TEAPOT_PATCHES_COUNT / 16,
    TEACUP_PATCHES_COUNT / 16,
    TEASPOON_PATCHES_COUNT / 16
};

//////////////////////////////////////////////////////////////////////
// numer indeksu restartu prymitywu
//////////////////////////////////////////////////////////////////////
const int PRIMITIVE_RESTART_INDEX = 0xFFFF;

//////////////////////////////////////////////////////////////////////
// numeracja identyfikatorw obiektw bufora z danymi tekstur
// buforowych i identyfikatorw obiektw tekstur buforowych
//////////////////////////////////////////////////////////////////////
enum
{
    // dane wsprzdnych punktw kontrolnych
    TEAPOT_PATCHES,
    TEACUP_PATCHES,
    TEASPOON_PATCHES,

    // dane indeksw wsprzdnych punktw kontrolnych
    TEAPOT_INDICES,
    TEACUP_INDICES,
    TEASPOON_INDICES,

    // rozmiary
    TEXTURE_BUFFER_SIZE,
    TEXTURE_SIZE = TEXTURE_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tekstur buforowych
//////////////////////////////////////////////////////////////////////
GLuint textureBuffer[TEXTURE_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstur
//////////////////////////////////////////////////////////////////////
GLuint texture[TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION 0

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // obroty wektora kierunku wiata
    glm::mat4x4 rotateLightDir = glm::mat4x4( 1.0 );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightY, glm::vec3( 0.0f, 1.0f, 0.0f ) );
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = rotateLightDir * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition( modelViewMatrixInverse * lightPosition );
    inverseLightPosition = glm::normalize( inverseLightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program, "lightSource[0].position" ), 1, glm::value_ptr( inverseLightPosition ) );
    glUniform4fv( glGetUniformLocation( program, "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program, "material" ), material );

    // zaadowanie numerw jednostek teksturyjcych z teksturami buforowymi
    glUniform1i( glGetUniformLocation( program, "patches" ), object + 0 );
    glUniform1i( glGetUniformLocation( program, "indices" ), object + 3 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElementsInstanced( GL_TRIANGLE_STRIP, indicesCount, GL_UNSIGNED_INT, NULL, objectInstancedCount[object] );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wypisanie nazwy materiau
    std::ostringstream txt;
    DrawText8x16( 3, 3, std::string( "materia: " ) + GetMaterialName( material ) );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "powierzchnia_beziera_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "powierzchnia_beziera_fs.glsl" ) );
    LinkValidateProgram( program );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // generowanie wartoci parametrw uv - danych wsprzdnych wierzchokw
    const int positionCount = 2 * (DENSITY + 1) * (DENSITY + 1);
    GLfloat position[positionCount];
    int idx = 0;
    for( int v = 0; v <= DENSITY; v++ )
        for( int u = 0; u <= DENSITY; u++ )
        {
            position[idx++] = static_cast<GLfloat>( u ) / DENSITY;
            position[idx++] = static_cast<GLfloat>( v ) / DENSITY;
        }

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
    glBufferData( GL_ARRAY_BUFFER, positionCount * sizeof( GLfloat ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // generowanie paskw trjktw tworzcych pat powierzchni
    GLuint indices[indicesCount];
    idx = 0;
    for( int u = 0; u < DENSITY; u++ )
    {
        indices[idx++] = (DENSITY + 1) * (u + 1);
        indices[idx++] = (DENSITY + 1) * (u + 0);
        for( int v = 0; v < DENSITY; v++ )
        {
            indices[idx++] = (DENSITY + 1) * (u + 1) + v + 1;
            indices[idx++] = (DENSITY + 1) * (u + 0) + v + 1;
        }

        // indeks restartu prymitywu (poza ostatnim paskiem trjktw)
        if( u < DENSITY - 1 )
            indices[idx++] = PRIMITIVE_RESTART_INDEX;
    }

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, indicesCount * sizeof( GLuint ), indices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // tekstura buforowa z danymi wsprzdnych punktw kontrolnych
    glGenBuffers( 1, &textureBuffer[TEAPOT_PATCHES] );
    glBindBuffer( GL_TEXTURE_BUFFER, textureBuffer[TEAPOT_PATCHES] );
    glBufferData( GL_TEXTURE_BUFFER, sizeof( teapotVertices ) * sizeof( GLfloat ), teapotVertices, GL_STATIC_DRAW );
    glActiveTexture( GL_TEXTURE0 );
    glGenTextures( 1, &texture[TEAPOT_PATCHES] );
    glBindTexture( GL_TEXTURE_BUFFER, texture[TEAPOT_PATCHES] );
    glTexBuffer( GL_TEXTURE_BUFFER, GL_RGB32F, textureBuffer[TEAPOT_PATCHES] );

    // tekstura buforowa z danymi wsprzdnych punktw kontrolnych
    glGenBuffers( 1, &textureBuffer[TEACUP_PATCHES] );
    glBindBuffer( GL_TEXTURE_BUFFER, textureBuffer[TEACUP_PATCHES] );
    glBufferData( GL_TEXTURE_BUFFER, sizeof( teacupVertices ) * sizeof( GLfloat ), teacupVertices, GL_STATIC_DRAW );
    glActiveTexture( GL_TEXTURE1 );
    glGenTextures( 1, &texture[TEACUP_PATCHES] );
    glBindTexture( GL_TEXTURE_BUFFER, texture[TEACUP_PATCHES] );
    glTexBuffer( GL_TEXTURE_BUFFER, GL_RGB32F, textureBuffer[TEACUP_PATCHES] );

    // tekstura buforowa z danymi wsprzdnych punktw kontrolnych
    glGenBuffers( 1, &textureBuffer[TEASPOON_PATCHES] );
    glBindBuffer( GL_TEXTURE_BUFFER, textureBuffer[TEASPOON_PATCHES] );
    glBufferData( GL_TEXTURE_BUFFER, sizeof( teaspoonVertices ) * sizeof( GLfloat ), teaspoonVertices, GL_STATIC_DRAW );
    glActiveTexture( GL_TEXTURE2 );
    glGenTextures( 1, &texture[TEASPOON_PATCHES] );
    glBindTexture( GL_TEXTURE_BUFFER, texture[TEASPOON_PATCHES] );
    glTexBuffer( GL_TEXTURE_BUFFER, GL_RGB32F, textureBuffer[TEASPOON_PATCHES] );

    // tekstura buforowa z danymi indeksw wsprzdnych punktw kontrolnych
    glGenBuffers( 1, &textureBuffer[TEAPOT_INDICES] );
    glBindBuffer( GL_TEXTURE_BUFFER, textureBuffer[TEAPOT_INDICES] );
    glBufferData( GL_TEXTURE_BUFFER, sizeof( teapotPatches ) * sizeof( GLuint ), teapotPatches, GL_STATIC_DRAW );
    glActiveTexture( GL_TEXTURE3 );
    glGenTextures( 1, &texture[TEAPOT_INDICES] );
    glBindTexture( GL_TEXTURE_BUFFER, texture[TEAPOT_INDICES] );
    glTexBuffer( GL_TEXTURE_BUFFER, GL_R32I, textureBuffer[TEAPOT_INDICES] );

    // tekstura buforowa z danymi indeksw wsprzdnych punktw kontrolnych
    glGenBuffers( 1, &textureBuffer[TEACUP_INDICES] );
    glBindBuffer( GL_TEXTURE_BUFFER, textureBuffer[TEACUP_INDICES] );
    glBufferData( GL_TEXTURE_BUFFER, sizeof( teacupPatches ) * sizeof( GLuint ), teacupPatches, GL_STATIC_DRAW );
    glActiveTexture( GL_TEXTURE4 );
    glGenTextures( 1, &texture[TEACUP_INDICES] );
    glBindTexture( GL_TEXTURE_BUFFER, texture[TEACUP_INDICES] );
    glTexBuffer( GL_TEXTURE_BUFFER, GL_R32I, textureBuffer[TEACUP_INDICES] );

    // tekstura buforowa z danymi indeksw wsprzdnych punktw kontrolnych
    glGenBuffers( 1, &textureBuffer[TEASPOON_INDICES] );
    glBindBuffer( GL_TEXTURE_BUFFER, textureBuffer[TEASPOON_INDICES] );
    glBufferData( GL_TEXTURE_BUFFER, sizeof( teaspoonPatches ) * sizeof( GLuint ), teaspoonPatches, GL_STATIC_DRAW );
    glActiveTexture( GL_TEXTURE5 );
    glGenTextures( 1, &texture[TEASPOON_INDICES] );
    glBindTexture( GL_TEXTURE_BUFFER, texture[TEASPOON_INDICES] );
    glTexBuffer( GL_TEXTURE_BUFFER, GL_R32I, textureBuffer[TEASPOON_INDICES] );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // numer indeksu do restartu prymitywu
    glPrimitiveRestartIndex( PRIMITIVE_RESTART_INDEX );

    // wczenie restartu prymitywu
    glEnable( GL_PRIMITIVE_RESTART );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // dane obsugujcy gwny obiekt programu
    glDeleteProgram( program );
    glDeleteBuffers( 1, &vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteBuffers( TEXTURE_BUFFER_SIZE, textureBuffer );
    glDeleteTextures( TEXTURE_SIZE, texture );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
