//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <cstdlib>
#include <vector>
#include <algorithm>
#include <sstream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "sphere.h"
#include "materials.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.8f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL
};

//////////////////////////////////////////////////////////////////////
// struktura opisujca pojedynczy obiekt sceny 3D
//////////////////////////////////////////////////////////////////////
struct Object
{
    // konstruktor
    Object( glm::vec4 pos, int mat ):
        position( pos ), MVposition( pos ), material( mat ) {};

    // wektor przesunicia obiektu wzgldem rodka sceny
    glm::vec4 position;

    // pooenie rodka obiektu po przeksztaceniu
    // macierz modelu-widoku
    glm::vec4 MVposition;

    // indeks materiau opisujcego waciwoci
    // owietlenia powierzchni obiektu
    int material;
};

//////////////////////////////////////////////////////////////////////
// tablica opisujca obiekty sceny 3D
//////////////////////////////////////////////////////////////////////
std::vector<Object> objects;

//////////////////////////////////////////////////////////////////////
// operator porwnujcy obiekty sceny 3D; w przypadku dwch
// obiektw przezroczystych porwnywane jest ich pooenie
// na osi OZ, a w pozostaych przypadkach porwnaniu podlega
// stopie przezroczystoci zawarty w skadowej A koloru rozproszonego
//////////////////////////////////////////////////////////////////////
bool operator < ( const Object &o1, const Object &o2 )
{
    // sortowanie obiektw przezroczystych po pooeniu na osi OZ
    if( GetMaterial( o1.material, MTL_DIFFUSE )[3] < 1.0 && GetMaterial( o2.material, MTL_DIFFUSE )[3] < 1.0 )
    {
        if( o1.MVposition[2] < o2.MVposition[2] ) return true;
        if( o1.MVposition[2] > o2.MVposition[2] ) return false;
        return false;
    }
    else

    // sortowanie obiektw po stopniu przezroczystoci
    {
        if( GetMaterial( o1.material, MTL_DIFFUSE )[3] > GetMaterial( o2.material, MTL_DIFFUSE )[3] ) return true;
        if( GetMaterial( o1.material, MTL_DIFFUSE )[3] < GetMaterial( o2.material, MTL_DIFFUSE )[3] ) return false;
        return false;
    }
}

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 1.0f, 1.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // obliczenie pooenia rodka poszczeglnych obiektw (kul) po przeksztaceniach
    for( unsigned int i = 0; i < objects.size(); i++ )
        objects[i].MVposition = modelViewMatrix * glm::vec4( objects[i].position[0], objects[i].position[1], objects[i].position[2], 1.0f );

    // sortowanie obiektw
    std::sort( objects.begin(), objects.end() );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program, "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program, "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie zmiennych jednorodnych obiektw nieprzezroczystych
    for( unsigned int i = 0; i < 2; i++ )
    {
        std::ostringstream txt;
        txt << i;
        glUniform4fv( glGetUniformLocation( program, std::string( "positions[" + txt.str() + "]" ).c_str() ), 1, glm::value_ptr( objects[i].position ) );
        glUniform1i( glGetUniformLocation( program, std::string( "materials[" + txt.str() + "]" ).c_str() ), objects[i].material );
    }

    // narysowanie obiektw nieprzezroczystych
    glDrawElementsInstanced( GL_TRIANGLES, SPHERE_MID_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL, 2 );

    // wczenie mieszania kolorw
    glEnable( GL_BLEND );

    // wyczenie zapisu danych bufora gbokoci
    glDepthMask( GL_FALSE );

    // zaadowanie zmiennych jednorodnych obiektw pprzezroczystych
    for( unsigned int i = 2; i < objects.size(); i++ )
    {
        std::ostringstream txt;
        txt << i - 2;
        glUniform4fv( glGetUniformLocation( program, std::string( "positions[" + txt.str() + "]" ).c_str() ), 1, glm::value_ptr( objects[i].position ) );
        glUniform1i( glGetUniformLocation( program, std::string( "materials[" + txt.str() + "]" ).c_str() ), objects[i].material );
    }

    // rysowanie tylko tylnej strony wieloktw
    glCullFace( GL_FRONT );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElementsInstanced( GL_TRIANGLES, SPHERE_MID_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL, 4 );

    // rysowanie tylko przedniej strony wieloktw
    glCullFace( GL_BACK );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElementsInstanced( GL_TRIANGLES, SPHERE_MID_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL, 4 );

    // wyczenie mieszania kolorw
    glDisable( GL_BLEND );

    // wczenie zapisu danych do bufora gbokoci
    glDepthMask( GL_TRUE );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "polprzezroczystosc_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "polprzezroczystosc_fs.glsl" ) );
    LinkValidateProgram( program );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer );
    glBufferData( GL_ARRAY_BUFFER, sizeof( sphereMidPositionNormal ), sphereMidPositionNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( sphereMidIndices ), sphereMidIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wspczynniki mieszania kolorw
    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

    // wczenie renderingu wybranej strony wielokta
    glEnable( GL_CULL_FACE );

    // zaadowanie danych obiektw
    objects.push_back( Object( glm::vec4( 1.5f, 0.0f, 0.0f, 0.0f ), MTL_POLISHED_BRONZE ) );
    objects.push_back( Object( glm::vec4( -1.5f, 0.0f, 0.0f, 0.0f ), MTL_GOLD ) );
    objects.push_back( Object( glm::vec4( 0.0f, 1.5f, 0.0f, 0.0f ), MTL_RUBY ) );
    objects.push_back( Object( glm::vec4( 0.0f, -1.5f, 0.0f, 0.0f ), MTL_TURQUOISE ) );
    objects.push_back( Object( glm::vec4( 0.0f, 0.0f, -1.5f, 0.0f ), MTL_OBSIDIAN ) );
    objects.push_back( Object( glm::vec4( 0.0f, 0.0f, 1.5f, 0.0f ), MTL_EMERALD ) );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteBuffers( 1, &vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
}
