//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// nazwa pliku podana w wierszu polecenia
//////////////////////////////////////////////////////////////////////
char *fileName = NULL;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    TEX_COORD,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat position [4*2] =
{
    -1.0f, -1.0f,
    1.0f, -1.0f,
    -1.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury w wierzchokach trjktw
// skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat texCoord [4*2] =
{
    0.0f, 1.0f,
    1.0f, 1.0f,
    0.0f, 0.0f,
    1.0f, 0.0f
};

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tekstury
//////////////////////////////////////////////////////////////////////
GLuint texture;

//////////////////////////////////////////////////////////////////////
// identyfikatory poszczeglnych filtrw
//////////////////////////////////////////////////////////////////////
#define NEUTRAL_FILTER               0  // filtr neutralny
#define AVERAGE_FILTER               1  // filtr uredniajcy
#define LP1_FILTER                   2  // filtr LP1
#define LP2_FILTER                   3  // filtr LP2
#define LP3_FILTER                   4  // filtr LP3
#define GAUSS_FILTER                 5  // filtr Gaussa
#define MEAN_REMOVAL_FILTER          6  // filtr usuwajcy redni
#define HP1_FILTER                   7  // filtr HP1
#define HP2_FILTER                   8  // filtr HP2
#define HP3_FILTER                   9  // filtr HP3
#define HORIZONTAL_FILTER           10  // filtr poziomy
#define VERTICAL_FILTER             11  // filtr pionowy
#define HORIZONTAL_VERTICAL_FILTER  12  // filtr poziomy/pionowy
#define GRADIENT_EAST_FILTER        13  // filtr gradientowy wschd
#define GRADIENT_SOUTH_EAST_FILTER  14  // filtr gradientowy poudniowy wschd
#define GRADIENT_SOUTH_FILTER       15  // filtr gradientowy poudnie
#define GRADIENT_SOUTH_WEST_FILTER  16  // filtr gradientowy poudniowy zachd
#define GRADIENT_WEST_FILTER        17  // filtr gradientowy zachd
#define GRADIENT_NORTH_WEST_FILTER  18  // filtr gradientowy pnocny zachd
#define GRADIENT_NORTH_FILTER       19  // filtr gradientowy pnoc
#define GRADIENT_NORTH_EAST_FILTER  20  // filtr gradientowy pnocny wschd
#define EMBOSS_EAST_FILTER          21  // filtr uwypuklajcy wschd
#define EMBOSS_SOUTH_EAST_FILTER    22  // filtr uwypuklajcy poudniowy wschd
#define EMBOSS_SOUTH_FILTER         23  // filtr uwypuklajcy poudnie
#define EMBOSS_SOUTH_WEST_FILTER    24  // filtr uwypuklajcy poudniowy zachd
#define EMBOSS_WEST_FILTER          25  // filtr uwypuklajcy zachd
#define EMBOSS_NORTH_WEST_FILTER    26  // filtr uwypuklajcy pnocny zachd
#define EMBOSS_NORTH_FILTER         27  // filtr uwypuklajcy pnoc
#define EMBOSS_NORTH_EAST_FILTER    28  // filtr uwypuklajcy pnocny wschd
#define LAPLACIAN_LAPL1_FILTER      29  // filtr Laplace'a LAPL1
#define LAPLACIAN_LAPL2_FILTER      30  // filtr Laplace'a LAPL2
#define LAPLACIAN_LAPL3_FILTER      31  // filtr Laplace'a LAPL3
#define LAPLACIAN_DIAGONAL_FILTER   32  // filtr Laplace'a skony
#define LAPLACIAN_HORIZONTAL_FILTER 33  // filtr Laplace'a poziomy
#define LAPLACIAN_VERTICAL_FILTER   34  // filtr Laplace'a pionowy
#define SOBEL_HORIZONTAL_FILTER     35  // filtr poziomy Sobela
#define SOBEL_VERTICAL_FILTER       36  // filtr pionowy Sobela
#define PREWITT_HORIZONTAL_FILTER   37  // filtr poziomy Prewitta
#define PREWITT_VERTICAL_FILTER     38  // filtr pionowy Prewitta

//////////////////////////////////////////////////////////////////////
// wybrany filt splotowy
//////////////////////////////////////////////////////////////////////
int filterType = NEUTRAL_FILTER;

//////////////////////////////////////////////////////////////////////
// tablica z nazwami filtrw
//////////////////////////////////////////////////////////////////////
const std::string filterName[] =
{
    "filtr neutralny",
    "filtr uredniajcy",
    "filtr LP1",
    "filtr LP2",
    "filtr LP3",
    "filtr Gaussa",
    "filtr usuwajcy redni",
    "filtr HP1",
    "filtr HP2",
    "filtr HP3",
    "filtr poziomy",
    "filtr pionowy",
    "filtr poziomy/pionowy",
    "filtr gradientowy wschd",
    "filtr gradientowy poudniowy wschd",
    "filtr gradientowy poudnie",
    "filtr gradientowy poudniowy zachd",
    "filtr gradientowy zachd",
    "filtr gradientowy pnocny zachd",
    "filtr gradientowy pnoc",
    "filtr gradientowy pnocny wschd",
    "filtr uwypuklajcy wschd",
    "filtr uwypuklajcy poudniowy wschd",
    "filtr uwypuklajcy poudnie",
    "filtr uwypuklajcy poudniowy zachd",
    "filtr uwypuklajcy zachd",
    "filtr uwypuklajcy pnocny zachd",
    "filtr uwypuklajcy pnoc",
    "filtr uwypuklajcy pnocny wschd",
    "filtr Laplace'a LAPL1",
    "filtr Laplace'a LAPL2",
    "filtr Laplace'a LAPL3",
    "filtr Laplace'a skony",
    "filtr Laplace'a poziomy",
    "filtr Laplace'a pionowy",
    "filtr poziomy Sobela",
    "filtr pionowy Sobela",
    "filtr poziomy Prewitta",
    "filtr pionowy Prewitta"
};

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // wczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, texture );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program ,"tex" ), 0 );

    // zmiana wartoci zmiennej jednorodnej - rodzaju filtra
    glUniform1i( glGetUniformLocation( program ,"filterType" ), filterType );

    // zaadowanie zmiennej jednorodnej macierzy rzutowania
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( projectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // pobranie rozmiarw obrazu tekstury
    GLint width, height;
    glGetTexLevelParameteriv( GL_TEXTURE_2D, 0, GL_TEXTURE_WIDTH, &width );
    glGetTexLevelParameteriv( GL_TEXTURE_2D, 0, GL_TEXTURE_HEIGHT, &height );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wypisanie wymiarw obrazu tekstury
    std::ostringstream txt;
    txt << "wysoko obrazu: " << height;
    DrawText8x16( 3, 3, txt.str(), glm::vec4( 1.0f ) );
    txt.str( "" );
    txt << "szeroko obrazu: " << width;
    DrawText8x16( 3, 21, txt.str(), glm::vec4( 1.0f ) );

    // wypisanie rodzaju filtra
    DrawText8x16( 3, 39, filterName[filterType], glm::vec4( 1.0f ) );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie prostoktne
    projectionMatrix = glm::ortho( -1.0f, 1.0f, -1.0f, 1.0f );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // utworzenie obiektu tekstury
    glGenTextures( 1, &texture );
    glBindTexture( GL_TEXTURE_2D, texture );

    // wczytanie tekstury
    if( !LoadTexture( fileName, GL_TEXTURE_2D ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << fileName << std::endl;
        exit( 0 );
    }

    // filtr pomniejszajcy
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // pobranie wewntrznego formatu danych tekstury
    GLint internalFormat;
    glGetTexLevelParameteriv( GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &internalFormat );
    if( internalFormat == GL_RED )
    {
        // podstawianie skadowych tekseli: RGBA -> RRRA
        GLint swizzle[4] = { GL_RED, GL_RED, GL_RED, GL_ALPHA };
        glTexParameteriv( GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_RGBA, swizzle );
    }

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "filtracja_obrazow_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "../../common/convolution_filters_3x3.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "filtracja_obrazow_fs.glsl" ) );
    LinkValidateProgram( program );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoord ), texCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // usunicie obiektu tekstury
    glDeleteTextures( 1, &texture );

    // usunicie obiektu programu
    glDeleteProgram( program );

    // usunicie obiektw bufora wierzchokw
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );

    // usunicie obiektu tablic wierzchokw
    glDeleteVertexArrays( 1, &vertexArray );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
