//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <cstring>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "teapot_low.h"
#include "teapot_high.h"
#include "sphere.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 10.0f;
GLfloat far = 18.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// pooenie rda wiata punktowego w ukadzie wsprzdnych obiektu
//////////////////////////////////////////////////////////////////////
glm::vec4 lightPosition( 0.0f, 0.0f, 6.0f, 1.0f );

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    LIGHT_PER_VERTEX,       // owietlenie obliczane na wierzchoek
    LIGHT_PER_FRAGMENT,     // owietlenie obliczane na fragment
    LIGHT_POINT,            // pooenie rda wiata
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// rodzaj owietlenia
//////////////////////////////////////////////////////////////////////
int lightType = LIGHT_PER_VERTEX;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL,
    POSITION_TEAPOT = POSITION,
    POSITION_TEAPOT_LOW,
    POSITION_TEAPOT_HIGH,
    NORMAL_TEAPOT,
    NORMAL_TEAPOT_LOW,
    NORMAL_TEAPOT_HIGH,
    POSITION_LIGHT,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw:
// wsprzdnymi wierzchokw i wektorw normalnych
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora indeksw wierzchokw, obiektw
// tablic wierzchokw oraz rodzajw czajnikw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    TEAPOT_LOW,
    TEAPOT_HIGH,
    LIGHT,
    VERTEX_ARRAY_SIZE,
    INDICES_BUFFER_SIZE = VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer[INDICES_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// numery nazwanych blokw jednorodnych
//////////////////////////////////////////////////////////////////////
enum
{
    MATERIALS,
    LIGHT_MODEL,
    UBO_SIZE
};

//////////////////////////////////////////////////////////////////////
// rozmiary blokw zmiennych jednorodnych
//////////////////////////////////////////////////////////////////////
GLint size[UBO_SIZE];

//////////////////////////////////////////////////////////////////////
// przesunicia pooenia blokw zmiennych jednorodnych
//////////////////////////////////////////////////////////////////////
GLint offset[UBO_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja modyfikowanych zmiennych z bloku jednorodnego
//////////////////////////////////////////////////////////////////////
enum
{
    LIGHT_SOURCE_0_AMBIENT,
    LIGHT_SOURCE_0_DIFFUSE,
    LIGHT_SOURCE_0_SPECULAR,
    LIGHT_SOURCE_0_POSITION,
    LIGHT_SOURCE_0_CONSTANT_ATTENUATION,
    LIGHT_SOURCE_0_LINEAR_ATTENUATION,
    LIGHT_SOURCE_0_QUADRATIC_ATTENUATION,
    LIGHT_MODEL_AMBIENT,
    EYE_POSITION,
    UNIFORMS_SIZE
};

//////////////////////////////////////////////////////////////////////
// nazwy modyfikowanych zmiennych z bloku jednorodnego
//////////////////////////////////////////////////////////////////////
const GLchar *uniformsName[UNIFORMS_SIZE] =
{
    "lightSource[0].ambient",
    "lightSource[0].diffuse",
    "lightSource[0].specular",
    "lightSource[0].position",
    "lightSource[0].constantAttenuation",
    "lightSource[0].linearAttenuation",
    "lightSource[0].quadraticAttenuation",
    "lightModel.ambient",
    "eyePosition"
};

//////////////////////////////////////////////////////////////////////
// wartoci przesunicia (pooenia) zmiennych w bloku jednorodnym
//////////////////////////////////////////////////////////////////////
GLint uniformsOffset[UNIFORMS_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora bloku jednorodnego
//////////////////////////////////////////////////////////////////////
GLuint uniformBuffer;

//////////////////////////////////////////////////////////////////////
// liczba danych w tablicach indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
const GLuint indicesCount[3] =
{
    TEAPOT_INDICES_COUNT * 3,
    TEAPOT_LOW_INDICES_COUNT * 3,
    TEAPOT_HIGH_INDICES_COUNT * 3
};

//////////////////////////////////////////////////////////////////////
// numer rysowanego czajnika
//////////////////////////////////////////////////////////////////////
int teapot = TEAPOT;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[teapot] );

    // wczenie programu
    glUseProgram( program[lightType] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[lightType], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - macierzy modelu-widoku
    glUniformMatrix4fv( glGetUniformLocation( program[lightType], "modelViewMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewMatrix ) );

    // zaadowanie zmiennej jednorodnej - macierzy przeksztacenia wektora normalnego
    // przeksztacenia macierzy wektora normalnego, tj. transponowanie i odwracanie,
    // nie s potrzebne z uwagi na to, e jest to macierz ortogonalna
    glm::mat3x3 normalMatrix( modelViewMatrix );
    glUniformMatrix3fv( glGetUniformLocation( program[lightType], "normalMatrix" ), 1, GL_FALSE, glm::value_ptr( normalMatrix ) );

    // zaadowanie zmiennej jednorodnej - rodzaj materiau
    glUniform1i( glGetUniformLocation( program[lightType], "material" ), material );

    // wczenie UBO
    glBindBufferRange( GL_UNIFORM_BUFFER, LIGHT_MODEL, uniformBuffer, offset[LIGHT_MODEL], size[LIGHT_MODEL] );

    // zaadowanie zmiennej jednorodnej z pooeniem rda wiata punktowego,
    // wsprzdne musz by wyraone w ukadzie wsprzdnych obserwatora
    lightPosition[2] -= (near+far)/2.0f;
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_POSITION], 16, glm::value_ptr( lightPosition ) );
    lightPosition[2] += (near+far)/2.0f;

    // wyczenie UBO
    glBindBuffer( GL_UNIFORM_BUFFER, 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, indicesCount[teapot], GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // transformacje pooenia rda wiata punktowego
    modelViewMatrix = glm::mat4x4( 1.0f );
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( lightPosition ) );
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( 0.2f, 0.2f, 0.2f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT] );

    // wczenie programu
    glUseProgram( program[LIGHT_POINT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[LIGHT_POINT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, SPHERE_LOW_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wypisanie wektora kierunku rda wiata
    std::ostringstream txt;
    txt << std::setprecision( 1 ) << std::fixed
        << "Pooenie rda wiata: (" << lightPosition[0] << ";" << lightPosition[1] << ";" << lightPosition[2] << ")";
    DrawText8x16( 3, 3, txt.str() );

    // wypisanie nazwy materiau
    DrawText8x16( 3, 21, std::string( "Materia: " ) + GetMaterialName( material ) );

    // wypisanie rodzaju owietlenia
    DrawText8x16( 3, 39, lightType == LIGHT_PER_VERTEX ? "Owietlenie na wierzchoek" : "Owietlenie na fragment" );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_PER_VERTEX] = glCreateProgram();
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "../../common/light_model.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "../../common/materials.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "swiatlo_wierzcholek_vs.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_FRAGMENT_SHADER, "swiatlo_wierzcholek_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_PER_VERTEX] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_PER_FRAGMENT] = glCreateProgram();
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_VERTEX_SHADER, "swiatlo_fragment_vs.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "swiatlo_fragment_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_PER_FRAGMENT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_POINT] = glCreateProgram();
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_VERTEX_SHADER, "punkt_swiatla_vs.glsl" ) );
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_FRAGMENT_SHADER, "punkt_swiatla_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_POINT] );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO)
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie drugiego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_LOW] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT_LOW] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT_LOW] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotLowPosition ), teapotLowPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT_LOW] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT_LOW] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotLowNormal ), teapotLowNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_LOW] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_LOW] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotLowIndices ), teapotLowIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie trzeciego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_HIGH] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT_HIGH] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT_HIGH] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighPosition ), teapotHighPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT_HIGH] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT_HIGH] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighNormal ), teapotHighNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_HIGH] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_HIGH] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotHighIndices ), teapotHighIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie czwartego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_LIGHT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_LIGHT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( sphereLowPositionNormal ), sphereLowPositionNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[LIGHT] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[LIGHT] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( sphereLowIndices ), sphereLowIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // pobranie indeksw blokw jednorodnych
    GLuint index[UBO_SIZE];
    index[MATERIALS] = glGetUniformBlockIndex( program[LIGHT_PER_VERTEX], "Materials" );
    index[LIGHT_MODEL] = glGetUniformBlockIndex( program[LIGHT_PER_VERTEX], "LightModel" );

    // pobranie rozmiarw blokw jednorodnych
    glGetActiveUniformBlockiv( program[LIGHT_PER_VERTEX], index[MATERIALS], GL_UNIFORM_BLOCK_DATA_SIZE, &size[MATERIALS] );
    glGetActiveUniformBlockiv( program[LIGHT_PER_VERTEX], index[LIGHT_MODEL], GL_UNIFORM_BLOCK_DATA_SIZE, &size[LIGHT_MODEL] );

    // okrelenie rozmiaru blokw z uwzgldnieniem wyrwnywania
    GLint alignment;
    glGetIntegerv( GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT, &alignment );
    size[MATERIALS] += alignment - size[MATERIALS] % alignment;
    size[LIGHT_MODEL] += alignment - size[LIGHT_MODEL] % alignment;

    // okrelenie przesuni blokw zmiennych jednorodnych,
    // ktre zale od numeru indeksu bloku
    offset[MATERIALS] = index[MATERIALS] > index[LIGHT_MODEL] ? size[LIGHT_MODEL] : 0;
    offset[LIGHT_MODEL] = index[LIGHT_MODEL] > index[MATERIALS] ? size[MATERIALS] : 0;

    // utworzenie obiektu bufora bloku jednorodnego (UBO) bez definiowania danych
    glGenBuffers( 1, &uniformBuffer );
    glBindBuffer( GL_UNIFORM_BUFFER, uniformBuffer );
    glBufferData( GL_UNIFORM_BUFFER, size[MATERIALS] + size[LIGHT_MODEL], NULL, GL_DYNAMIC_DRAW );

    // powizanie bloku jednorodnego do punktw wizania
    glUniformBlockBinding( program[LIGHT_PER_VERTEX], index[MATERIALS], MATERIALS );
    glUniformBlockBinding( program[LIGHT_PER_FRAGMENT], index[MATERIALS], MATERIALS );
    glBindBufferRange( GL_UNIFORM_BUFFER, MATERIALS, uniformBuffer, offset[MATERIALS], size[MATERIALS] );

    // zaadowanie danych bloku jednorodnego
    GLintptr offsetBuf = offset[MATERIALS];
    const GLsizeiptr sizeMat = 4 * sizeof( GLfloat );
    for( unsigned int material = MTL_DEFAULT; material <= MTL_YELLOW_RUBBER; material++ )
    {
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 0 * sizeMat, sizeMat, GetMaterial( material, MTL_AMBIENT ) );
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 1 * sizeMat, sizeMat, GetMaterial( material, MTL_DIFFUSE ) );
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 2 * sizeMat, sizeMat, GetMaterial( material, MTL_SPECULAR ) );
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 3 * sizeMat, sizeMat/4, GetMaterial( material, MTL_SHININESS ) );
        offsetBuf += 4 * sizeMat;
    }

    // powizanie bloku jednorodnego do punktw wizania
    glUniformBlockBinding( program[LIGHT_PER_VERTEX], index[LIGHT_MODEL], LIGHT_MODEL );
    glUniformBlockBinding( program[LIGHT_PER_FRAGMENT], index[LIGHT_MODEL], LIGHT_MODEL );
    glBindBufferRange( GL_UNIFORM_BUFFER, LIGHT_MODEL, uniformBuffer, offset[LIGHT_MODEL], size[LIGHT_MODEL] );

    // pobranie indeksw i wartoci przesuni wybranych zmiennych z bloku jednorodnego
    GLuint uniformsIndex[UNIFORMS_SIZE];
    glGetUniformIndices( program[LIGHT_PER_VERTEX], UNIFORMS_SIZE, uniformsName, uniformsIndex );
    glGetActiveUniformsiv( program[LIGHT_PER_VERTEX], UNIFORMS_SIZE, uniformsIndex, GL_UNIFORM_OFFSET, uniformsOffset );

    // modyfikacja przesuni
    for( int i = 0; i < UNIFORMS_SIZE; i++ )
        uniformsOffset[i] += offset[LIGHT_MODEL];

    // domylne parametry rda wiata nr 0
    const GLfloat lightSource0ambient[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    const GLfloat lightSource0diffuse[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    const GLfloat lightSource0specular[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    const GLfloat lightSource0position[4] = { 0.0f, 0.0f, 1.0f, 0.0f };
    const GLfloat lightSource0constantAttenuation = 1.0f;
    const GLfloat lightSource0linearAttenuation = 0.0f;
    const GLfloat lightSource0quadraticAttenuation = 0.0f;

    // zaadowanie domylnych parametrw rda wiata nr 0 do LigthSource[0]
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_AMBIENT], 16, lightSource0ambient );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_DIFFUSE], 16, lightSource0diffuse );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_SPECULAR], 16, lightSource0specular );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_POSITION], 16, lightSource0position );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_CONSTANT_ATTENUATION], 4, &lightSource0constantAttenuation );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_LINEAR_ATTENUATION], 4, &lightSource0linearAttenuation );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_QUADRATIC_ATTENUATION], 4, &lightSource0quadraticAttenuation );

    // zaadowanie globalnego wiata otoczenia do lightModel.ambient
    const GLfloat lightModelAmbient[4] = { 0.2f, 0.2f, 0.2f, 1.0f };
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_MODEL_AMBIENT], 16, lightModelAmbient );

    // zaadowanie wsprzdnych pooenia obserwatora
    const GLfloat eyePosition[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[EYE_POSITION], 16, eyePosition );

    // wyczenie powizania
    glBindBuffer( GL_UNIFORM_BUFFER, 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    for( int i = 0; i < PROGRAM_SIZE; i++ )
    {
        glDeleteProgram( program[i] );
    }
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( INDICES_BUFFER_SIZE, indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteBuffers( 1, &uniformBuffer );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
