//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// GLSL 1.50
//////////////////////////////////////////////////////////////////////
#version 150

//////////////////////////////////////////////////////////////////////
// struktura opisujca parametry materiau
//////////////////////////////////////////////////////////////////////
struct MaterialParameters
{
    vec4 ambient;       // wspczynnik odbicia wiata otoczenia; oznaczenie Ma
    vec4 diffuse;       // wspczynnik odbicia wiata rozproszonego; oznaczenie Md
    vec4 specular;      // wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
    float shininess;    // wykadnik wspczynnika funkcji rozbysku;
                        // zakres wartoci <0;128>; oznaczenie Msh
};

//////////////////////////////////////////////////////////////////////
// prototypy funkcji
//////////////////////////////////////////////////////////////////////
MaterialParameters GetMaterial( const int material );
vec4 BlinnPhongLight( const int light, const vec3 normal, const vec3 position,
                        const vec4 ambient, const vec4 diffuse,
                        const vec4 specular, const float shininess );
vec4 LambertLight( const int light, const vec3 normal, const vec3 position,
                    const vec4 ambient, const vec4 diffuse );

//////////////////////////////////////////////////////////////////////
// rozmiar tablicy z mask filtra
//////////////////////////////////////////////////////////////////////
#define FILTER_5X5_SIZE 25

//////////////////////////////////////////////////////////////////////
// rozkad Poissona 5x5
// File:    Poisson.fxh
// Author:  Louis Bavoil
// Email:   sdkfeedback@nvidia.com
// Copyright (c) NVIDIA Corporation. All rights reserved.
//////////////////////////////////////////////////////////////////////
const vec2 Poisson25[FILTER_5X5_SIZE] = vec2[FILTER_5X5_SIZE]
(
    vec2( -0.978698, -0.0884121 ),
    vec2( -0.841121, 0.521165 ),
    vec2( -0.71746, -0.50322 ),
    vec2( -0.702933, 0.903134 ),
    vec2( -0.663198, 0.15482 ),
    vec2( -0.495102, -0.232887 ),
    vec2( -0.364238, -0.961791 ),
    vec2( -0.345866, -0.564379 ),
    vec2( -0.325663, 0.64037 ),
    vec2( -0.182714, 0.321329 ),
    vec2( -0.142613, -0.0227363 ),
    vec2( -0.0564287, -0.36729 ),
    vec2( -0.0185858, 0.918882 ),
    vec2( 0.0381787, -0.728996 ),
    vec2( 0.16599, 0.093112 ),
    vec2( 0.253639, 0.719535 ),
    vec2( 0.369549, -0.655019 ),
    vec2( 0.423627, 0.429975 ),
    vec2( 0.530747, -0.364971 ),
    vec2( 0.566027, -0.940489 ),
    vec2( 0.639332, 0.0284127 ),
    vec2( 0.652089, 0.669668 ),
    vec2( 0.773797, 0.345012 ),
    vec2( 0.968871, 0.840449 ),
    vec2( 0.991882, -0.657338 )
);

//////////////////////////////////////////////////////////////////////
// numer materiau
//////////////////////////////////////////////////////////////////////
uniform int material;

//////////////////////////////////////////////////////////////////////
// tekstura gbokoci z map cienia
//////////////////////////////////////////////////////////////////////
uniform sampler2DShadow shadowTex;

//////////////////////////////////////////////////////////////////////
// interpolowany wektor normalny
//////////////////////////////////////////////////////////////////////
in vec3 inoutNormal;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne wierzchoka
//////////////////////////////////////////////////////////////////////
in vec3 inoutPosition;

//////////////////////////////////////////////////////////////////////
// interpolowane wsprzdne tekstury
//////////////////////////////////////////////////////////////////////
in vec4 inoutTexCoord;

//////////////////////////////////////////////////////////////////////
// wyjciowy kolor fragmentu
//////////////////////////////////////////////////////////////////////
out vec4 outColor;

void main()
{
    // normalizacja wektora normalnego zalena od strony wielokta
    vec3 normal = gl_FrontFacing ? normalize( inoutNormal ) : -normalize( inoutNormal );

    // pobranie wybranego materiau
    MaterialParameters mat = GetMaterial( material );

    // kolor owietlonego fragmentu
    vec4 lightColor = BlinnPhongLight( 0, normal, inoutPosition, mat.ambient, mat.diffuse, mat.specular, mat.shininess );

    // kolor fragmentu w cieniu
    vec4 shadowColor = LambertLight( 0, normal, inoutPosition, mat.ambient, mat.diffuse );

    // rozmiar tekstury z map cienia
    vec2 shadowTexSize = textureSize( shadowTex, 0 );

    // prbkowanie tekstury cienia
    float shadow = 0.0;
    for( int i = 0; i < FILTER_5X5_SIZE; i++ )
    {
        vec2 offset = (5.0f * Poisson25[i] - 2.5f ) / shadowTexSize;
        shadow += textureProj( shadowTex, inoutTexCoord + vec4( offset, 0, 0 ) );
    }
    shadow /= FILTER_5X5_SIZE;

    // kolor fragmentu z cieniem pomnoonym przez kolor cienia
    outColor = mix( 1.5 * shadowColor * vec4( 0.184314f, 0.309804f, 0.309804f, 1.000000f ), lightColor, shadow );

}
