//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    TEX_COORD,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat position [4*2] =
{
    -1.0f, -1.0f,
    1.0f, -1.0f,
    -1.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury w wierzchokach trjktw
// skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat texCoord [4*2] =
{
    0.0f, 1.0f,
    1.0f, 1.0f,
    0.0f, 0.0f,
    1.0f, 0.0f
};

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    TEXTURE_MIPMAP_LOAD,
    TEXTURE_MIPMAP_GEN,
    TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury
//////////////////////////////////////////////////////////////////////
GLuint texture[TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// numer wybranego obiektu tekstury
//////////////////////////////////////////////////////////////////////
int texNumber = TEXTURE_MIPMAP_LOAD;

//////////////////////////////////////////////////////////////////////
// bazowy poziom obrazu tekstury
//////////////////////////////////////////////////////////////////////
GLuint baseLevel = 0;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu prbkowania tekstury
//////////////////////////////////////////////////////////////////////
GLuint sampler;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // wybr obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, texture[texNumber] );

    // wybr bazowego poziomu obrazu tekstury
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_BASE_LEVEL, baseLevel );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program ,"tex" ), 0 );

    // zaadowanie zmiennej jednorodnej macierzy rzutowania
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( projectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // pobranie rozmiarw danego poziomu obrazu tekstury
    GLint width, height;
    glGetTexLevelParameteriv( GL_TEXTURE_2D, baseLevel, GL_TEXTURE_WIDTH, &width );
    glGetTexLevelParameteriv( GL_TEXTURE_2D, baseLevel, GL_TEXTURE_HEIGHT, &height );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wypisanie wymiarw obrazu tekstury
    std::ostringstream txt;
    txt << "wysoko obrazu: " << height;
    DrawText8x16( 3, 3, txt.str(), glm::vec4( 1.0f ) );
    txt.str( "" );
    txt << "szeroko obrazu: " << width;
    DrawText8x16( 3, 21, txt.str(), glm::vec4( 1.0f ) );

    // wypisanie informacji o rodzaju generowanych mipmap
    if( texNumber == TEXTURE_MIPMAP_LOAD )
        DrawText8x16( 3, 39, "mipmapy zaadowane z pliku", glm::vec4( 1.0f ) );
    else
        DrawText8x16( 3, 39, "mipmapy generowane automatycznie", glm::vec4( 1.0f ) );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie prostoktne
    projectionMatrix = glm::ortho( -1.0f, 1.0f, -1.0f, 1.0f );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // generowanie identyfikatorw obiektw tekstury
    glGenTextures( TEXTURE_SIZE, texture );

    // utworzenie pierwszego obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, texture[TEXTURE_MIPMAP_LOAD] );

    // nazwy plikw z obrazami kolejnych poziomw szczegowoci tekstury
    const std::string fileNames[10] =
    {
        "../../media/lena/lena.png",
        "../../media/lena/lena_256x256.png",
        "../../media/lena/lena_128x128.png",
        "../../media/lena/lena_64x64.png",
        "../../media/lena/lena_32x32.png",
        "../../media/lena/lena_16x16.png",
        "../../media/lena/lena_8x8.png",
        "../../media/lena/lena_4x4.png",
        "../../media/lena/lena_2x2.png",
        "../../media/lena/lena_1x1.png"
    };

    // odczyt kolejnych plikw i ewentualny komunikat o bdzie
    for( int i = 0; i < 10; i++ )
        if( !LoadTexture( fileNames[i].c_str(), GL_TEXTURE_2D, i ) )
        {
            std::cout << "Niepoprawny odczyt pliku " << fileNames[i] << std::endl;
            exit( 0 );
        }

    // utworzenie drugiego obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, texture[TEXTURE_MIPMAP_GEN] );

    // odczyt pliku i ewentualny komunikat o bdzie
    if( !LoadTexture( fileNames[0].c_str(), GL_TEXTURE_2D ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << fileNames[0] << std::endl;
        exit( 0 );
    }

    // generownanie mipmap
    glGenerateMipmap( GL_TEXTURE_2D );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, 0 );

    // generowanie identyfikatora obiektu prbkowania tekstury
    glGenSamplers( 1, &sampler );

    // powizanie obiektu prbkowania tekstury z obiektami tekstury powizanymi
    // z jednostk teksturujc GL_TEXTURE0
    glBindSampler( 0, sampler );

    // filtr pomniejszajcy
    glSamplerParameteri( sampler, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );

    // filtr powikszajcy
    glSamplerParameteri( sampler, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "mipmapy2d_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "mipmapy2d_fs.glsl" ) );
    LinkValidateProgram( program );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoord ), texCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // usunicie obiektw tekstury
    glDeleteTextures( TEXTURE_SIZE, texture );

    // usunicie obiektu probkowania tekstury
    glDeleteSamplers( 1, &sampler );

    // usunicie obiektu programu
    glDeleteProgram( program );

    // usunicie obiektw bufora wierzchokw
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );

    // usunicie obiektu tablic wierzchokw
    glDeleteVertexArrays( 1, &vertexArray );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
