//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <cstring>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "teapot_low.h"
#include "teapot_high.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 10.0f;
GLfloat far = 18.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu kierunku wiata
//////////////////////////////////////////////////////////////////////
GLfloat rotateLightX = 0.0f;
GLfloat rotateLightY = 0.0f;

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    LIGHT_PER_VERTEX,       // owietlenie obliczane na wierzchoek
    LIGHT_PER_FRAGMENT,     // owietlenie obliczane na fragment
    LIGHT_VECTOR,           // kierunek wiata
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// rodzaj owietlenia
//////////////////////////////////////////////////////////////////////
int lightType = LIGHT_PER_VERTEX;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL,
    POSITION_TEAPOT = POSITION,
    POSITION_TEAPOT_LOW,
    POSITION_TEAPOT_HIGH,
    NORMAL_TEAPOT,
    NORMAL_TEAPOT_LOW,
    NORMAL_TEAPOT_HIGH,
    POSITION_LIGHT,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw:
// wsprzdnymi wierzchokw i wektorw normalnych
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora indeksw wierzchokw, obiektw
// tablic wierzchokw oraz rodzajw czajnikw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    TEAPOT_LOW,
    TEAPOT_HIGH,
    INDICES_BUFFER_SIZE,
    LIGHT = INDICES_BUFFER_SIZE,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer[INDICES_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// liczba danych w tablicach indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
const GLuint indicesCount[3] =
{
    TEAPOT_INDICES_COUNT * 3,
    TEAPOT_LOW_INDICES_COUNT * 3,
    TEAPOT_HIGH_INDICES_COUNT * 3
};

//////////////////////////////////////////////////////////////////////
// numer rysowanego czajnika
//////////////////////////////////////////////////////////////////////
int teapot = TEAPOT;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // obroty wektora kierunku wiata
    glm::mat4x4 rotateLightDir = glm::mat4x4( 1.0 );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightY, glm::vec3( 0.0f, 1.0f, 0.0f ) );
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = rotateLightDir * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[teapot] );

    // wczenie programu
    glUseProgram( program[lightType] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[lightType], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - macierzy modelu-widoku
    glUniformMatrix4fv( glGetUniformLocation( program[lightType], "modelViewMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewMatrix ) );

    // zaadowanie zmiennej jednorodnej - macierzy przeksztacenia wektora normalnego
    // przeksztacenia macierzy wektora normalnego, tj. transponowanie i odwracanie,
    // nie s potrzebne z uwagi na to, e jest to macierz ortogonalna
    glm::mat3x3 normalMatrix( modelViewMatrix );
    glUniformMatrix3fv( glGetUniformLocation( program[lightType], "normalMatrix" ), 1, GL_FALSE, glm::value_ptr( normalMatrix ) );

    // zaadowanie zmiennej jednorodnej - rodzaj materiau
    glUniform1i( glGetUniformLocation( program[lightType], "material" ), material );

    // zaadowanie zmiennej jednorodnej z kierunkiem wiata
    glUniform4fv( glGetUniformLocation( program[lightType], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, indicesCount[teapot], GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // transformacje pooenia wektora kierunku wiata
    modelViewMatrix = glm::mat4x4( 1.0 );
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );
    modelViewMatrix *= rotateLightDir;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT] );

    // wczenie programu
    glUseProgram( program[LIGHT_VECTOR] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[LIGHT_VECTOR], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_LINES, 0, 2 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wypisanie wektora kierunku rda wiata
    std::ostringstream txt;
    txt << std::setprecision( 4 ) << std::fixed
        << "Wektor kierunku wiata: (" << lightPosition[0] << ";" << lightPosition[1] << ";" << lightPosition[2] << ")";
    DrawText8x16( 3, 3, txt.str() );

    // wypisanie nazwy materiau
    DrawText8x16( 3, 21, std::string( "Materia: " ) + GetMaterialName( material ) );

    // wypisanie rodzaju owietlenia
    DrawText8x16( 3, 39, lightType == LIGHT_PER_VERTEX ? "Owietlenie na wierzchoek" : "Owietlenie na fragment" );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_PER_VERTEX] = glCreateProgram();
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_VERTEX_SHADER, "swiatlo_wierzcholek_vs.glsl" ) );
    glAttachShader( program[LIGHT_PER_VERTEX], LoadShader( GL_FRAGMENT_SHADER, "swiatlo_wierzcholek_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_PER_VERTEX] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_PER_FRAGMENT] = glCreateProgram();
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_VERTEX_SHADER, "swiatlo_fragment_vs.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[LIGHT_PER_FRAGMENT], LoadShader( GL_FRAGMENT_SHADER, "swiatlo_fragment_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_PER_FRAGMENT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_VECTOR] = glCreateProgram();
    glAttachShader( program[LIGHT_VECTOR], LoadShader( GL_VERTEX_SHADER, "wektor_swiatla_vs.glsl" ) );
    glAttachShader( program[LIGHT_VECTOR], LoadShader( GL_FRAGMENT_SHADER, "wektor_swiatla_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_VECTOR] );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO)
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT] );

    // pocztkowe wartoci wektorw normalnych
    GLfloat normals[TEAPOT_VERTEX_COUNT * 3];
    memset( normals, 0, sizeof( normals ) );

    // obliczenie wektora normalnego cian i nastpnie dodanie tego
    // wektora do wektorw normalnych poszczeglnych wierzchokw
    for( unsigned int i = 0; i < TEAPOT_INDICES_COUNT; i++ )
    {
        // wektory zbudowane na podstawie wierzchokw trjkta, ktry zawiera bieca ciana
        const glm::vec3 v10( glm::vec3( teapotPosition[3*teapotIndices[3*i + 1] + 0],
                                        teapotPosition[3*teapotIndices[3*i + 1] + 1],
                                        teapotPosition[3*teapotIndices[3*i + 1] + 2] ) -
                             glm::vec3( teapotPosition[3*teapotIndices[3*i + 0] + 0],
                                        teapotPosition[3*teapotIndices[3*i + 0] + 1],
                                        teapotPosition[3*teapotIndices[3*i + 0] + 2] ) );
        const glm::vec3 v21( glm::vec3( teapotPosition[3*teapotIndices[3*i + 2] + 0],
                                        teapotPosition[3*teapotIndices[3*i + 2] + 1],
                                        teapotPosition[3*teapotIndices[3*i + 2] + 2] ) -
                             glm::vec3( teapotPosition[3*teapotIndices[3*i + 1] + 0],
                                        teapotPosition[3*teapotIndices[3*i + 1] + 1],
                                        teapotPosition[3*teapotIndices[3*i + 1] + 2] ) );

        // obliczenie wektora normalnego za pomoc iloczynu wektorowego
        const glm::vec3 normal( glm::normalize( glm::cross( v10, v21 ) ) );

        // dodanie wektora normalnego ciany do urednionego wektora normalnego wierzchoka
        normals[3*teapotIndices[3*i + 0] + 0] += normal[0];
        normals[3*teapotIndices[3*i + 0] + 1] += normal[1];
        normals[3*teapotIndices[3*i + 0] + 2] += normal[2];

        normals[3*teapotIndices[3*i + 1] + 0] += normal[0];
        normals[3*teapotIndices[3*i + 1] + 1] += normal[1];
        normals[3*teapotIndices[3*i + 1] + 2] += normal[2];

        normals[3*teapotIndices[3*i + 2] + 0] += normal[0];
        normals[3*teapotIndices[3*i + 2] + 1] += normal[1];
        normals[3*teapotIndices[3*i + 2] + 2] += normal[2];
    }

    // normalizacja wektorw normalnych dla kolejnych wierzchokw
    for( int i = 0; i < TEAPOT_VERTEX_COUNT; i++ )
    {
        const glm::vec3 normal( glm::normalize( glm::vec3( normals[3*i + 0], normals[3*i + 1], normals[3*i + 2] ) ) );
        normals[3*i + 0] = normal[0];
        normals[3*i + 1] = normal[1];
        normals[3*i + 2] = normal[2];
    }

    // zaadowanie danych obiektu bufora wierzchokw (VBO)
    glBufferData( GL_ARRAY_BUFFER, sizeof( normals ), normals, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie drugiego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_LOW] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT_LOW] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT_LOW] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotLowPosition ), teapotLowPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT_LOW] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT_LOW] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotLowNormal ), teapotLowNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_LOW] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_LOW] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotLowIndices ), teapotLowIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie trzeciego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_HIGH] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT_HIGH] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT_HIGH] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighPosition ), teapotHighPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT_HIGH] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT_HIGH] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighNormal ), teapotHighNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_HIGH] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_HIGH] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotHighIndices ), teapotHighIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie czwartego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    GLfloat lightPosition[3*2] =
    {
        0.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 5.0f,
    };
    glGenBuffers( 1, &vertexBuffer[POSITION_LIGHT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_LIGHT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( lightPosition ), lightPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
    // szeroko linii
    glLineWidth( 3.0f );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    for( int i = 0; i < PROGRAM_SIZE; i++ )
    {
        glDeleteProgram( program[i] );
    }
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( INDICES_BUFFER_SIZE, indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
