//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// nazwa pliku podana w wierszu polecenia
//////////////////////////////////////////////////////////////////////
char *fileName = NULL;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    TEX_COORD,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat position [4*2] =
{
    -1.0f, -1.0f,
    1.0f, -1.0f,
    -1.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury w wierzchokach trjktw
// skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat texCoord [4*2] =
{
    0.0f, 1.0f,
    1.0f, 1.0f,
    0.0f, 0.0f,
    1.0f, 0.0f
};

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tekstury
//////////////////////////////////////////////////////////////////////
GLuint texture;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // wczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, texture );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program ,"tex" ), 0 );

    // zaadowanie zmiennej jednorodnej macierzy rzutowania
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( projectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // pobranie rozmiarw obrazu tekstury
    GLint width, height;
    glGetTexLevelParameteriv( GL_TEXTURE_2D, 0, GL_TEXTURE_WIDTH, &width );
    glGetTexLevelParameteriv( GL_TEXTURE_2D, 0, GL_TEXTURE_HEIGHT, &height );

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wypisanie wymiarw obrazu tekstury
    std::ostringstream txt;
    txt << "wysoko obrazu: " << height;
    DrawText8x16( 3, 3, txt.str(), glm::vec4( 1.0f ) );
    txt.str( "" );
    txt << "szeroko obrazu: " << width;
    DrawText8x16( 3, 21, txt.str(), glm::vec4( 1.0f ) );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie prostoktne
    projectionMatrix = glm::ortho( -1.0f, 1.0f, -1.0f, 1.0f );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // utworzenie obiektu tekstury
    glGenTextures( 1, &texture );
    glBindTexture( GL_TEXTURE_2D, texture );

    // wczytanie tekstury
    if( !LoadTexture( fileName, GL_TEXTURE_2D ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << fileName << std::endl;
        exit( 0 );
    }

    // filtr pomniejszajcy
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // pobranie wewntrznego formatu danych tekstury
    GLint internalFormat;
    glGetTexLevelParameteriv( GL_TEXTURE_2D, 0, GL_TEXTURE_INTERNAL_FORMAT, &internalFormat );
    if( internalFormat == GL_RED )
    {
        // podstawianie skadowych tekseli: RGBA -> RRRA
        GLint swizzle[4] = { GL_RED, GL_RED, GL_RED, GL_ALPHA };
        glTexParameteriv( GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_RGBA, swizzle );
    }

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "tekstura2d_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "tekstura2d_fs.glsl" ) );
    LinkValidateProgram( program );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoord ), texCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // usunicie obiektu tekstury
    glDeleteTextures( 1, &texture );

    // usunicie obiektu programu
    glDeleteProgram( program );

    // usunicie obiektw bufora wierzchokw
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );

    // usunicie obiektu tablic wierzchokw
    glDeleteVertexArrays( 1, &vertexArray );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
