// Listing 17.2. Przykad uycia przepywu optycznego Farnebacka

#include <vector>
#include <iostream>
#include <cstdlib>
#include <fstream>
#include <opencv2/opencv.hpp>
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>

using std::cout;
using std::cerr;
using std::string;

// Wartoci argumentw dla calcOpticalFlowFarneback
//
const double pyr_scale = 0.85;  // skala midzy poziomami piramidy (< 1.0)
const int levels = 7;           // liczba poziomw piramidy
const int winsize = 13;         // rozmiar okienka dla przebiegu wstpnego
const int iterations = 10;      // liczba iteracji dla kadego poziomu piramidy
const int poly_n = 5;           // obszar dopasowywania wielomianu
const double poly_sigma = 1.1;  // szeroko wielokta

// funkcja zwraca obiekt cv::Mat z wizualizacj przepywu optycznego
//
cv::Mat get_optflow_image(cv::Mat& optflow, cv::Mat& img) {
    cv::Scalar arrow_color(0, 0, 255);
    cv::Mat res = img.clone();
    res /= 2;  // przyciemnianie obrazu
    int rows = res.rows;
    int cols = res.cols;
    const int step = 12;
    for (int x = (step >> 1); x < rows; x += step)
        for (int y = (step >> 1); y < cols; y += step) {
            float vx = optflow.at<cv::Vec2f>(x, y)[0];
            float vy = optflow.at<cv::Vec2f>(x, y)[1];
            cv::Point pt1(y, x);
            cv::Point pt2(y + vx, x + vy);
            cv::arrowedLine(res, pt1, pt2, arrow_color, 1);
        }
    return res;
}

int main(int argc, char** argv) {
    // program oczekuje przynajmniej cieki do pliku wideo
    //
    if (argc < 2) {
        cerr 	<< "\nPrzykad 17.2: Przykad uycia przepywu optycznego Farnbacka\n"
				<< "Use:\n" << argv[0] << " <path/video_file>\n"
				<< "Przykad:\n" << argv[0] << " ../test.avi\n"
				<< std::endl;
        exit(1);
    }

    string file_name = string(argv[1]);
    cv::VideoCapture capture(file_name);

    if (!capture.isOpened()) {
        cerr << "Nie udao si otworzy pliku \"" << file_name << "\"\n";
        exit(-1);
    }

    cv::Mat optflow;  // wynik przepywu optycznego
    cv::Mat optflow_image;  // wizualizacja przepywu optycznego
    cv::Mat prev_frame;  // monochromatyczny obraz poprzedniej klatki
    cv::Mat frame;  // monochromatyczny obraz aktualnej klatki
    cv::Mat colored_frame;  // obraz RGB aktualnej klatki

    cv::namedWindow("video");

    // uytkownik moe zamkn program, naciskajc klawisz ESC
    //
    while ((cv::waitKey(10) & 255) != 27) {
        capture >> colored_frame;
        if (!colored_frame.rows || !colored_frame.cols) {
            break;
        }
        if (colored_frame.type() == CV_8UC3) {
            cvtColor(colored_frame, frame, CV_BGR2GRAY);
        }
        if (prev_frame.rows) {
            calcOpticalFlowFarneback(prev_frame, frame, optflow, pyr_scale, levels, winsize,
                iterations, poly_n, poly_sigma, cv::OPTFLOW_FARNEBACK_GAUSSIAN);
            optflow_image = get_optflow_image(optflow, colored_frame);
            cv::imshow("video", optflow_image);
        }
        prev_frame = frame.clone();
    }
    cv::destroyAllWindows();

    return 0;
}
