package concurrency;

import org.junit.Test;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.concurrent.*;

import static org.junit.Assert.*;

public class CompletableFutureTests {

    @Test(expected = CancellationException.class)
    public void joinAfterCancel() throws Exception {
        CompletableFuture<String> future = new CompletableFuture<>();
        future.cancel(false);
        future.complete("Skoczyem");
        System.out.println(future.join());
    }

    @Test
    public void completeWithGet() throws Exception {
        CompletableFuture<String> future = new CompletableFuture<>();
        boolean finished = future.complete("Skoczyem");
        assertTrue(finished);
        try {
            assertEquals("Skoczyem", future.get());
        } catch (InterruptedException | ExecutionException e) {
            e.printStackTrace();
        }
    }

    @Test
    public void completeWithJoin() throws Exception {
        CompletableFuture<String> future = new CompletableFuture<>();
        boolean finished = future.complete("Skoczyem");
        assertTrue(finished);
        assertEquals("Skoczyem", future.join());
    }

    @Test(expected = ExecutionException.class)
    public void completeExceptionally() throws Exception {
        parseNumber("abc").get();
    }

    @Test
    public void completeExceptionallyWithCause() throws Exception {
        try {
            parseNumber("abc").get();
            fail("Nie powinienem tu dotrze");
        } catch (Exception e) {
            assertEquals(ExecutionException.class, e.getClass());
            assertEquals(NumberFormatException.class, e.getCause().getClass());
        }
    }

    @Test
    public void completeLong() throws Exception {
        assertTrue(parseNumber("42").get() == 42);
    }

    private CompletableFuture<Long> parseNumber(String arg) {
        CompletableFuture<Long> future = new CompletableFuture<>();
        try {
            future.complete(Long.parseLong(arg));
        } catch (Exception e) {
            future.completeExceptionally(e);
        }
        return future;
    }

    @Test
    public void supplyThenAccept() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        System.setOut(new PrintStream(baos));

        CompletableFuture.supplyAsync(() -> "42")
                .thenApply(Integer::parseInt)
                .thenApply(x -> 2 * x)
                .thenAccept(System.out::println);
        System.out.println("Dziaa...");

        String result = baos.toString();
        System.out.println(result);
        assertTrue(result.contains("84"));
        assertTrue(result.contains("Dziaa..."));
    }

    @Test
    public void supplyThenAcceptAsyncWithExecutor() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        System.setOut(new PrintStream(baos));

        ExecutorService service = Executors.newFixedThreadPool(4);
        CompletableFuture.supplyAsync(() -> "42", service)
                .thenApply(Integer::parseInt)
                .thenApply(x -> 2 * x)
                .thenAccept(System.out::println);
        System.out.println("Dziaa...");

        String result = baos.toString();
        System.out.println(result);
        assertTrue(result.contains("84"));
        assertTrue(result.contains("Dziaa..."));

    }

    @Test
    public void compose() throws Exception {
        int x = 2;
        int y = 3;
        CompletableFuture<Integer> completableFuture =
                CompletableFuture.supplyAsync(() -> x)
                        .thenCompose(n -> CompletableFuture.supplyAsync(() -> n + y));

        assertTrue(5 == completableFuture.get());
    }

    @Test
    public void combine() throws Exception {
        int x = 2;
        int y = 3;
        CompletableFuture<Integer> completableFuture =
                CompletableFuture.supplyAsync(() -> x)
                        .thenCombine(CompletableFuture.supplyAsync(() -> y),
                                (n1, n2) -> n1 + n2);

        assertTrue(5 == completableFuture.get());
    }

    private CompletableFuture<Integer> getIntegerCompletableFuture(String num) {
        return CompletableFuture.supplyAsync(() -> Integer.parseInt(num))
                .handle((val, exc) -> val != null ? val : 0);
    }

    @Test
    public void handleWithException() throws Exception {
        String num = "abc";
        CompletableFuture<Integer> value = getIntegerCompletableFuture(num);
        assertTrue(value.get() == 0);
    }

    @Test
    public void handleWithoutException() throws Exception {
        String num = "42";
        CompletableFuture<Integer> value = getIntegerCompletableFuture(num);
        assertTrue(value.get() == 42);
    }

}
