"use strict";
/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
Object.defineProperty(exports, "__esModule", { value: true });
const architect_1 = require("@angular-devkit/architect");
const build_webpack_1 = require("@angular-devkit/build-webpack");
const core_1 = require("@angular-devkit/core");
const node_1 = require("@angular-devkit/core/node");
const fs_1 = require("fs");
const path = require("path");
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const ts = require("typescript");
const url = require("url");
const webpack = require("webpack");
const index_html_webpack_plugin_1 = require("../angular-cli-files/plugins/index-html-webpack-plugin");
const check_port_1 = require("../angular-cli-files/utilities/check-port");
const package_chunk_sort_1 = require("../angular-cli-files/utilities/package-chunk-sort");
const read_tsconfig_1 = require("../angular-cli-files/utilities/read-tsconfig");
const browser_1 = require("../browser");
const utils_1 = require("../utils");
const version_1 = require("../utils/version");
const open = require('open');
exports.devServerBuildOverriddenKeys = [
    'watch',
    'optimization',
    'aot',
    'sourceMap',
    'vendorSourceMap',
    'evalSourceMap',
    'vendorChunk',
    'commonChunk',
    'baseHref',
    'progress',
    'poll',
    'verbose',
    'deployUrl',
];
/**
 * Reusable implementation of the build angular webpack dev server builder.
 * @param options Dev Server options.
 * @param context The build context.
 * @param transforms A map of transforms that can be used to hook into some logic (such as
 *     transforming webpack configuration before passing it to webpack).
 */
function serveWebpackBrowser(options, context, transforms = {}) {
    // Check Angular version.
    version_1.assertCompatibleAngularVersion(context.workspaceRoot, context.logger);
    const browserTarget = architect_1.targetFromTargetString(options.browserTarget);
    const root = context.workspaceRoot;
    let first = true;
    const host = new node_1.NodeJsSyncHost();
    const loggingFn = transforms.logging || browser_1.createBrowserLoggingCallback(!!options.verbose, context.logger);
    async function setup() {
        // Get the browser configuration from the target name.
        const rawBrowserOptions = await context.getTargetOptions(browserTarget);
        // Override options we need to override, if defined.
        const overrides = Object.keys(options)
            .filter(key => options[key] !== undefined && exports.devServerBuildOverriddenKeys.includes(key))
            .reduce((previous, key) => ({
            ...previous,
            [key]: options[key],
        }), {});
        // In dev server we should not have budgets because of extra libs such as socks-js
        overrides.budgets = undefined;
        const browserName = await context.getBuilderNameForTarget(browserTarget);
        const browserOptions = await context.validateOptions({ ...rawBrowserOptions, ...overrides }, browserName);
        const webpackConfigResult = await browser_1.buildBrowserWebpackConfigFromContext(browserOptions, context, host);
        // No differential loading for dev-server, hence there is just one config
        let webpackConfig = webpackConfigResult.config[0];
        const port = await check_port_1.checkPort(options.port || 0, options.host || 'localhost', 4200);
        const webpackDevServerConfig = (webpackConfig.devServer = buildServerConfig(root, options, browserOptions, context.logger));
        if (transforms.webpackConfiguration) {
            webpackConfig = await transforms.webpackConfiguration(webpackConfig);
        }
        return {
            browserOptions,
            webpackConfig,
            webpackDevServerConfig,
            port,
            workspace: webpackConfigResult.workspace,
        };
    }
    return rxjs_1.from(setup()).pipe(operators_1.switchMap(({ browserOptions, webpackConfig, webpackDevServerConfig, port, workspace }) => {
        options.port = port;
        // Resolve public host and client address.
        let clientAddress = url.parse(`${options.ssl ? 'https' : 'http'}://0.0.0.0:0`);
        if (options.publicHost) {
            let publicHost = options.publicHost;
            if (!/^\w+:\/\//.test(publicHost)) {
                publicHost = `${options.ssl ? 'https' : 'http'}://${publicHost}`;
            }
            clientAddress = url.parse(publicHost);
            options.publicHost = clientAddress.host;
        }
        // Add live reload config.
        if (options.liveReload) {
            _addLiveReload(options, browserOptions, webpackConfig, clientAddress, context.logger);
        }
        else if (options.hmr) {
            context.logger.warn('Live reload is disabled. HMR option ignored.');
        }
        webpackConfig.plugins = [...(webpackConfig.plugins || [])];
        if (!options.watch) {
            // There's no option to turn off file watching in webpack-dev-server, but
            // we can override the file watcher instead.
            webpackConfig.plugins.push({
                // tslint:disable-next-line:no-any
                apply: (compiler) => {
                    compiler.hooks.afterEnvironment.tap('angular-cli', () => {
                        compiler.watchFileSystem = { watch: () => { } };
                    });
                },
            });
        }
        if (browserOptions.index) {
            const { scripts = [], styles = [], index, baseHref, tsConfig } = browserOptions;
            const projectName = context.target
                ? context.target.project
                : workspace.getDefaultProjectName();
            if (!projectName) {
                throw new Error('Must either have a target from the context or a default project.');
            }
            const projectRoot = core_1.resolve(workspace.root, core_1.normalize(workspace.getProject(projectName).root));
            const { options: compilerOptions } = read_tsconfig_1.readTsconfig(tsConfig, context.workspaceRoot);
            const target = compilerOptions.target || ts.ScriptTarget.ES5;
            const buildBrowserFeatures = new utils_1.BuildBrowserFeatures(core_1.getSystemPath(projectRoot), target);
            const entrypoints = package_chunk_sort_1.generateEntryPoints({ scripts, styles });
            const moduleEntrypoints = buildBrowserFeatures.isDifferentialLoadingNeeded()
                ? package_chunk_sort_1.generateEntryPoints({ scripts: [], styles })
                : [];
            webpackConfig.plugins.push(new index_html_webpack_plugin_1.IndexHtmlWebpackPlugin({
                input: path.resolve(root, index),
                output: path.basename(index),
                baseHref,
                moduleEntrypoints,
                entrypoints,
                deployUrl: browserOptions.deployUrl,
                sri: browserOptions.subresourceIntegrity,
                noModuleEntrypoints: ['polyfills-es5'],
                postTransform: transforms.indexHtml,
                crossOrigin: browserOptions.crossOrigin,
            }));
        }
        const normalizedOptimization = utils_1.normalizeOptimization(browserOptions.optimization);
        if (normalizedOptimization.scripts || normalizedOptimization.styles) {
            context.logger.error(core_1.tags.stripIndents `
          ****************************************************************************************
          This is a simple server for use in testing or debugging Angular applications locally.
          It hasn't been reviewed for security issues.

          DON'T USE IT FOR PRODUCTION!
          ****************************************************************************************
        `);
        }
        return build_webpack_1.runWebpackDevServer(webpackConfig, context, { logging: loggingFn }).pipe(operators_1.map(buildEvent => {
            // Resolve serve address.
            const serverAddress = url.format({
                protocol: options.ssl ? 'https' : 'http',
                hostname: options.host === '0.0.0.0' ? 'localhost' : options.host,
                pathname: webpackDevServerConfig.publicPath,
                port: buildEvent.port,
            });
            if (first) {
                first = false;
                context.logger.info(core_1.tags.oneLine `
              **
              Angular Live Development Server is listening on ${options.host}:${buildEvent.port},
              open your browser on ${serverAddress}
              **
            `);
                if (options.open) {
                    open(serverAddress);
                }
            }
            return { ...buildEvent, baseUrl: serverAddress };
        }));
    }));
}
exports.serveWebpackBrowser = serveWebpackBrowser;
/**
 * Create a webpack configuration for the dev server.
 * @param workspaceRoot The root of the workspace. This comes from the context.
 * @param serverOptions DevServer options, based on the dev server input schema.
 * @param browserOptions Browser builder options. See the browser builder from this package.
 * @param logger A generic logger to use for showing warnings.
 * @returns A webpack dev-server configuration.
 */
function buildServerConfig(workspaceRoot, serverOptions, browserOptions, logger) {
    if (serverOptions.host) {
        // Check that the host is either localhost or prints out a message.
        if (!/^127\.\d+\.\d+\.\d+/g.test(serverOptions.host) && serverOptions.host !== 'localhost') {
            logger.warn(core_1.tags.stripIndent `
          WARNING: This is a simple server for use in testing or debugging Angular applications
          locally. It hasn't been reviewed for security issues.

          Binding this server to an open connection can result in compromising your application or
          computer. Using a different host than the one passed to the "--host" flag might result in
          websocket connection issues. You might need to use "--disableHostCheck" if that's the
          case.
        `);
        }
    }
    if (serverOptions.disableHostCheck) {
        logger.warn(core_1.tags.oneLine `
        WARNING: Running a server with --disable-host-check is a security risk.
        See https://medium.com/webpack/webpack-dev-server-middleware-security-issues-1489d950874a
        for more information.
      `);
    }
    const servePath = buildServePath(serverOptions, browserOptions, logger);
    const { styles, scripts } = utils_1.normalizeOptimization(browserOptions.optimization);
    const config = {
        host: serverOptions.host,
        port: serverOptions.port,
        headers: { 'Access-Control-Allow-Origin': '*' },
        historyApiFallback: !!browserOptions.index &&
            {
                index: `${servePath}/${path.basename(browserOptions.index)}`,
                disableDotRule: true,
                htmlAcceptHeaders: ['text/html', 'application/xhtml+xml'],
                rewrites: [
                    {
                        from: new RegExp(`^(?!${servePath})/.*`),
                        to: context => url.format(context.parsedUrl),
                    },
                ],
            },
        stats: false,
        compress: styles || scripts,
        watchOptions: {
            poll: browserOptions.poll,
        },
        https: serverOptions.ssl,
        overlay: {
            errors: !(styles || scripts),
            warnings: false,
        },
        // inline is always false, because we add live reloading scripts in _addLiveReload when needed
        inline: false,
        public: serverOptions.publicHost,
        disableHostCheck: serverOptions.disableHostCheck,
        publicPath: servePath,
        hot: serverOptions.hmr,
        contentBase: false,
    };
    if (serverOptions.ssl) {
        _addSslConfig(workspaceRoot, serverOptions, config);
    }
    if (serverOptions.proxyConfig) {
        _addProxyConfig(workspaceRoot, serverOptions, config);
    }
    return config;
}
exports.buildServerConfig = buildServerConfig;
/**
 * Resolve and build a URL _path_ that will be the root of the server. This resolved base href and
 * deploy URL from the browser options and returns a path from the root.
 * @param serverOptions The server options that were passed to the server builder.
 * @param browserOptions The browser options that were passed to the browser builder.
 * @param logger A generic logger to use for showing warnings.
 */
function buildServePath(serverOptions, browserOptions, logger) {
    let servePath = serverOptions.servePath;
    if (!servePath && servePath !== '') {
        const defaultPath = _findDefaultServePath(browserOptions.baseHref, browserOptions.deployUrl);
        const showWarning = serverOptions.servePathDefaultWarning;
        if (defaultPath == null && showWarning) {
            logger.warn(core_1.tags.oneLine `
        WARNING: --deploy-url and/or --base-href contain unsupported values for ng serve. Default
        serve path of '/' used. Use --serve-path to override.
      `);
        }
        servePath = defaultPath || '';
    }
    if (servePath.endsWith('/')) {
        servePath = servePath.substr(0, servePath.length - 1);
    }
    if (!servePath.startsWith('/')) {
        servePath = `/${servePath}`;
    }
    return servePath;
}
exports.buildServePath = buildServePath;
/**
 * Private method to enhance a webpack config with live reload configuration.
 * @private
 */
function _addLiveReload(options, browserOptions, webpackConfig, clientAddress, logger) {
    if (webpackConfig.plugins === undefined) {
        webpackConfig.plugins = [];
    }
    // This allows for live reload of page when changes are made to repo.
    // https://webpack.js.org/configuration/dev-server/#devserver-inline
    let webpackDevServerPath;
    try {
        webpackDevServerPath = require.resolve('webpack-dev-server/client');
    }
    catch (_a) {
        throw new Error('The "webpack-dev-server" package could not be found.');
    }
    // If a custom path is provided the webpack dev server client drops the sockjs-node segment.
    // This adds it back so that behavior is consistent when using a custom URL path
    let sockjsPath = '';
    if (clientAddress.pathname) {
        clientAddress.pathname = path.posix.join(clientAddress.pathname, 'sockjs-node');
        sockjsPath = '&sockPath=' + clientAddress.pathname;
    }
    const entryPoints = [`${webpackDevServerPath}?${url.format(clientAddress)}${sockjsPath}`];
    if (options.hmr) {
        const webpackHmrLink = 'https://webpack.js.org/guides/hot-module-replacement';
        logger.warn(core_1.tags.oneLine `NOTICE: Hot Module Replacement (HMR) is enabled for the dev server.`);
        const showWarning = options.hmrWarning;
        if (showWarning) {
            logger.info(core_1.tags.stripIndents `
          The project will still live reload when HMR is enabled,
          but to take advantage of HMR additional application code is required'
          (not included in an Angular CLI project by default).'
          See ${webpackHmrLink}
          for information on working with HMR for Webpack.`);
            logger.warn(core_1.tags.oneLine `To disable this warning use "hmrWarning: false" under "serve"
           options in "angular.json".`);
        }
        entryPoints.push('webpack/hot/dev-server');
        webpackConfig.plugins.push(new webpack.HotModuleReplacementPlugin());
        if (browserOptions.extractCss) {
            logger.warn(core_1.tags.oneLine `NOTICE: (HMR) does not allow for CSS hot reload
                when used together with '--extract-css'.`);
        }
    }
    if (typeof webpackConfig.entry !== 'object' || Array.isArray(webpackConfig.entry)) {
        webpackConfig.entry = {};
    }
    if (!Array.isArray(webpackConfig.entry.main)) {
        webpackConfig.entry.main = [];
    }
    webpackConfig.entry.main.unshift(...entryPoints);
}
/**
 * Private method to enhance a webpack config with SSL configuration.
 * @private
 */
function _addSslConfig(root, options, config) {
    let sslKey = undefined;
    let sslCert = undefined;
    if (options.sslKey) {
        const keyPath = path.resolve(root, options.sslKey);
        if (fs_1.existsSync(keyPath)) {
            sslKey = fs_1.readFileSync(keyPath, 'utf-8');
        }
    }
    if (options.sslCert) {
        const certPath = path.resolve(root, options.sslCert);
        if (fs_1.existsSync(certPath)) {
            sslCert = fs_1.readFileSync(certPath, 'utf-8');
        }
    }
    config.https = true;
    if (sslKey != null && sslCert != null) {
        config.https = {
            key: sslKey,
            cert: sslCert,
        };
    }
}
/**
 * Private method to enhance a webpack config with Proxy configuration.
 * @private
 */
function _addProxyConfig(root, options, config) {
    let proxyConfig = {};
    const proxyPath = path.resolve(root, options.proxyConfig);
    if (fs_1.existsSync(proxyPath)) {
        proxyConfig = require(proxyPath);
    }
    else {
        const message = 'Proxy config file ' + proxyPath + ' does not exist.';
        throw new Error(message);
    }
    config.proxy = proxyConfig;
}
/**
 * Find the default server path. We don't want to expose baseHref and deployUrl as arguments, only
 * the browser options where needed. This method should stay private (people who want to resolve
 * baseHref and deployUrl should use the buildServePath exported function.
 * @private
 */
function _findDefaultServePath(baseHref, deployUrl) {
    if (!baseHref && !deployUrl) {
        return '';
    }
    if (/^(\w+:)?\/\//.test(baseHref || '') || /^(\w+:)?\/\//.test(deployUrl || '')) {
        // If baseHref or deployUrl is absolute, unsupported by ng serve
        return null;
    }
    // normalize baseHref
    // for ng serve the starting base is always `/` so a relative
    // and root relative value are identical
    const baseHrefParts = (baseHref || '').split('/').filter(part => part !== '');
    if (baseHref && !baseHref.endsWith('/')) {
        baseHrefParts.pop();
    }
    const normalizedBaseHref = baseHrefParts.length === 0 ? '/' : `/${baseHrefParts.join('/')}/`;
    if (deployUrl && deployUrl[0] === '/') {
        if (baseHref && baseHref[0] === '/' && normalizedBaseHref !== deployUrl) {
            // If baseHref and deployUrl are root relative and not equivalent, unsupported by ng serve
            return null;
        }
        return deployUrl;
    }
    // Join together baseHref and deployUrl
    return `${normalizedBaseHref}${deployUrl || ''}`;
}
exports.default = architect_1.createBuilder(serveWebpackBrowser);
