package r10.game;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

/**
 * The playing field for our game. Now we can setup some constants for
 * other game classes to use and create member variables for our player and some trees.
 */
public class Field extends JComponent implements ActionListener {
    public static final float GRAVITY = 9.8f;  // feet per second per second
    public static final int STEP = 40;   // duration of an animation frame in milliseconds
    public static final int APPLE_SIZE_IN_PIXELS = 30;
    public static final int TREE_WIDTH_IN_PIXELS = 60;
    public static final int TREE_HEIGHT_IN_PIXELS = 2 * TREE_WIDTH_IN_PIXELS;
    public static final int PHYSICIST_SIZE_IN_PIXELS = 75;

    public static final int MAX_PHYSICISTS = 5;
    public static final int MAX_APPLES_PER_PHYSICIST = 3;
    public static final int MAX_TREES = 12;

    Color fieldColor = Color.GRAY;

	// Keep track of our own score and also make room for multiple players
    int myScore = 0;
    String[] scores = new String[3];
    JLabel[] scoreLabels = new JLabel[3];
	
    List<Physicist> physicists = Collections.synchronizedList(new ArrayList<>());
    List<Apple> apples = Collections.synchronizedList(new ArrayList<>());
    List<Tree> trees = Collections.synchronizedList(new ArrayList<>());

    boolean animating = false;
    Thread animationThread;
    Timer animationTimer;

    protected void paintComponent(Graphics g) {
        g.setColor(fieldColor);
        g.fillRect(0,0, getWidth(), getHeight());
        for (Physicist p : physicists) {
            p.draw(g);
        }
        for (Tree t : trees) {
            t.draw(g);
        }
        for (Apple a : apples) {
            a.draw(g);
        }
    }

    public void actionPerformed(ActionEvent event) {
        if (animating && event.getActionCommand().equals("odwieanie")) {
            System.out.println("Liczba przesuwanych jabek " + apples.size() + ".");
            for (Apple a : apples) {
                a.step();
                detectCollisions(a);
            }
            repaint();
            cullFallenApples();
        }
    }

    /**
     * Toss an apple from the given physicist using that physicist's aim and force.
     * Make sure the field is in the animating state.
     *
     * @param physicist the player whose apple should be tossed
     */
    public void startTossFromPlayer(Physicist physicist) {
        if (!animating) {
            System.out.println("Rozpoczynanie animacji!");
            animating = true;
            startAnimation();
        }
        if (animating) {
            // Check to make sure we have an apple to toss
            if (physicist.aimingApple != null) {
                Apple apple = physicist.takeApple();
                apple.toss(physicist.aimingAngle, physicist.aimingForce);
                apples.add(apple);
                Timer appleLoader = new Timer(800, physicist);
                appleLoader.setActionCommand("Nowe jabko");
                appleLoader.setRepeats(false);
                appleLoader.start();
            }
        }
    }

    void cullFallenApples() {
        Iterator<Apple> iterator = apples.iterator();
        while (iterator.hasNext()) {
            Apple a = iterator.next();
            if (a.getCollidedPiece() != null) {
                GamePiece otherPiece = a.getCollidedPiece();
                if (otherPiece instanceof Physicist) {
                    hitPhysicist((Physicist) otherPiece);
                } else if (otherPiece instanceof Tree) {
                    hitTree((Tree) otherPiece);
                }
                // Remove ourselves. If the other piece we hit was an apple, leave it alone.
                // It will be removed when the iterator comes to it.
                iterator.remove();
            } else if (a.getPositionY() > 600) {
                System.out.println("Rzucanie jabkiem");
                iterator.remove();
            }
        }
        if (apples.size() <= 0) {
            animating = false;
            if (animationTimer != null && animationTimer.isRunning()) {
                animationTimer.stop();
            }
        }
    }

    void detectCollisions(Apple apple) {
        // Check for other apples
        for (Apple a : apples) {
            if (apple.isTouching(a)) {
                System.out.println("Dotyka innego jabka!");
                return;
            }
        }
        // Check for physicists
        for (Physicist p : physicists) {
            if (apple.isTouching(p)) {
                System.out.println("Dotyka fizyka");
                return;
            }
        }
        // Check for trees
        for (Tree t : trees) {
            if (apple.isTouching(t)) {
                System.out.println("Dotyka drzewa!");
                return;
            }
        }
    }

    void hitPhysicist(Physicist physicist) {
        // do any scoring or notifications here
        physicists.remove(physicist);
    }

    void hitTree(Tree tree) {
        // do any scoring or notifications here
        myScore += 10;
        trees.remove(tree);
        setScore(1, String.valueOf(myScore));
    }

    public void resetScore(int playerNumber) {
		myScore = 0;
		setScore(playerNumber, "0");
    }

    public String getScore(int playerNumber) {
        return scores[playerNumber];
    }

    public void setScore(int playerNumber, String score) {
        scores[playerNumber] = score;
        SwingUtilities.invokeLater(new Runnable() {
            public void run() {
                String newScore = " Gracz " + playerNumber + ": " + score;
                scoreLabels[playerNumber].setText(newScore);
            }
        });
    }

    void startAnimation() {
        // Animator myAnimator = new Animator();
        // animationThread = new Thread(myAnimator);
        // animationThread.start();
        if (animationTimer == null) {
            animationTimer = new Timer(STEP, this);
            animationTimer.setActionCommand("odwieanie");
            animationTimer.setRepeats(true);
            animationTimer.start();
        } else if (!animationTimer.isRunning()) {
            animationTimer.restart();
        }
    }

    class Animator implements Runnable {
        public void run() {
            while (animating) {
                System.out.println("Liczba przesuwanych jabek " + apples.size() + ".");
                for (Apple a : apples) {
                    a.step();
                    detectCollisions(a);
                }
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        Field.this.repaint();
                    }
                });
                cullFallenApples();
                try {
                    Thread.sleep((int)(STEP * 1000));
                } catch (InterruptedException ie) {
                    System.err.println("Animowanie przerwane");
                    animating = false;
                }
            }
        }
    }
}
