package game;

import javax.swing.*;
import java.io.*;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;

public class Multiplayer {
    private Field gameField;
    private int gamePort;
    private volatile boolean keepListening;
    private volatile boolean keepPlaying;
    private volatile boolean startNewGame;
    private volatile boolean disconnecting;
    private Server server;
    private Thread serverThread;
    private Thread clientThread;

    public Multiplayer(Field field) {
        this(field,8677);
    }

    public Multiplayer(Field field, int port) {
        gameField = field;
        gamePort = port;
    }

    public void startServer() {
        keepListening = true;
        keepPlaying = false;
        startNewGame = true;
        disconnecting = false;
        server = new Server();
        serverThread = new Thread(server);
        serverThread.start();
        gameField.setScore(2, "waiting...");
    }

    public void joinGame(String otherServer) {
        clientThread = new Thread(new Client(otherServer));
        clientThread.start();
    }

    public void startGame() {
        startNewGame = true;
    }

    public void disconnect() {
        disconnecting = true;
        keepListening = false;
        // Are we in the middle of a game and regularly checking these flags?
        // If not, just close the server socket to interrupt the blocking accept() method.
        if (server != null && keepPlaying == false) {
            server.stopListening();
        }
        keepPlaying = false;
    }

    class Server implements Runnable {
        ServerSocket listener;

        public void run() {
            Socket socket = null;
            try {
                listener = new ServerSocket(gamePort);
                while (keepListening) {
                    socket = listener.accept();  // wait for connection

                    InputStream in = socket.getInputStream();
                    BufferedReader reader = new BufferedReader( new InputStreamReader(in) );
                    OutputStream out = socket.getOutputStream();
                    PrintWriter writer = new PrintWriter(out, true);

                    while (startNewGame) {
                        // Create a new game, but assume this will be the last
                        writer.println("NOWA_GRA");
                        startNewGame = false;

                        // If the client agrees, Send over the location of the trees
                        String response = reader.readLine();
                        if (response != null && response.equals("OK")) {
                            gameField.setupNewGame();
                            for (Tree tree : gameField.trees) {
                                writer.println("DRZEWO " + tree.getPositionX() + " " + tree.getPositionY());
                            }
                        } else {
                            System.err.println("Nieoczekiwana reakcja na start: " + response);
                            System.err.println("Pomijanie gry i ponowne oczekiwanie.");
                            keepPlaying = false;
                            break;
                        }

                        // Start the action!
                        writer.println("START");
                        response = reader.readLine();
                        keepPlaying = response.equals("OK");

                        while (keepPlaying) {
                            try {
                                if (gameField.trees.size() > 0) {
                                    writer.print("WYNIK ");
                                } else {
                                    writer.print("KONIEC ");
                                    keepPlaying = false;
                                }
                                writer.println(gameField.getScore(1));
                                response = reader.readLine();
                                if (response == null) {
                                    keepPlaying = false;
                                    disconnecting = true;
                                } else {
                                    String parts[] = response.split(" ");
                                    switch (parts[0]) {
                                        case "KONIEC":
                                            keepPlaying = false;
                                        case "WYNIK":
                                            gameField.setScore(2, parts[1]);
                                            break;
                                        case "ROZCZ":
                                            disconnecting = true;
                                            keepPlaying = false;
                                            break;
                                        default:
                                            System.err.println("Ostrzeenie. Nieoczekiwane polecenie: " + parts[0] + ". Ignorowanie.");
                                    }
                                }
                                Thread.sleep(500);
                            } catch(InterruptedException e) {
                                System.err.println("Przerwano podczas odpytywania. Ignorowanie.");
                            }
                        }

                        // If we're not disconnecting, ask about playing again with the same player
                        if (!disconnecting) {
                            String message = gameField.getWinner() + " Czy chcesz poprosi go o ponown rozgrywk?";
                            int myPlayAgain = JOptionPane.showConfirmDialog(gameField, message, "Gramy ponownie?", JOptionPane.YES_NO_OPTION);

                            if (myPlayAgain == JOptionPane.YES_OPTION) {
                                // If they haven't disconnected, ask if they want to play again
                                writer.println("GRAMY_PONOWNIE");
                                String playAgain = reader.readLine();
                                if (playAgain != null) {
                                    switch (playAgain) {
                                        case "TAK":
                                            startNewGame = true;
                                            break;
                                        case "ROZCZ":
                                            keepPlaying = false;
                                            startNewGame = false;
                                            disconnecting = true;
                                            break;
                                        default:
                                            System.err.println("Ostrzeenie. Nieoczekiwana odpowied: " + playAgain + ". Nie gramy ponownie.");
                                    }
                                }
                            }
                        }
                    }

                    if (socket.isConnected()) {
                        // say goodbye
                        writer.println("ROZCZ");
                        socket.close();
                    }
                }
            } catch (SocketException se) {
                System.err.println("Rozczanie. Zamykanie serwera.");
                keepListening = false;
            } catch (IOException ioe) {
                System.err.println("Bd sieciowy. Zamykanie serwera.");
                ioe.printStackTrace();
                keepListening = false;
            } catch (Exception e) {
                System.err.println("Wystpi inny bd. Zamykanie serwera.");
                e.printStackTrace();
                keepListening = false;
            } finally {
                try {
                    if (socket != null && !socket.isClosed()) {
                        socket.close();
                    }
                } catch (IOException closingException) {
                    System.err.println("Bd zamykanie gniazda klienta: " + closingException.getMessage());
                }
            }
        }

        public void stopListening() {
            if (listener != null && !listener.isClosed()) {
                try {
                    listener.close();
                } catch (IOException ioe) {
                    System.err.println("Bd zamykanie nasuchiwacza: " + ioe.getMessage());
                }
            }
        }
    }

    class Client implements Runnable {
        String gameHost;
        boolean startNewGame;

        public Client(String host) {
            gameHost = host;
            keepPlaying = false;
            startNewGame = false;
        }

        public void run() {
            try (Socket socket = new Socket(gameHost, gamePort)) {

                InputStream in = socket.getInputStream();
                BufferedReader reader = new BufferedReader( new InputStreamReader( in ) );
                OutputStream out = socket.getOutputStream();
                PrintWriter writer = new PrintWriter( out, true );

                // Assume the first game will start...
                startNewGame = true;
                while (startNewGame) {
                    // ... but only the first
                    startNewGame = false;

                    // We expect to see the NEW_GAME command first
                    String response = reader.readLine();

                    // If we don't see that command, bail
                    if (response == null || !response.equals("NOWA_GRA")) {
                        System.err.println("Nieoczekiwane polecenie pocztkowe: " + response);
                        System.err.println("Rozczanie");
                        writer.println("ROZCZ");
                        return;
                    }
                    // Yay! We're going to play a game. Acknowledge this command
                    writer.println("OK");
                    // And now gather the trees and setup our field
                    gameField.trees.clear();
                    response = reader.readLine();
                    while (response.startsWith("DRZEWO")) {
                        String[] parts = response.split(" ");
                        int x = Integer.parseInt(parts[1]);
                        int y = Integer.parseInt(parts[2]);
                        Tree tree = new Tree();
                        tree.setPosition(x, y);
                        gameField.trees.add(tree);
                        response = reader.readLine();
                    }
                    if (!response.equals("START")) {
                        // Hmm, we should have ended the list of trees with a START, but didn't. Bail out.
                        System.err.println("Nieoczekiwane uruchomienie gry: " + response);
                        System.err.println("Rozczenie");
                        writer.println("ROZCZ");
                        return;
                    } else {
                        // Yay again! We're starting a game. Acknowledge this command
                        writer.println("OK");
                        keepPlaying = true;
                        gameField.repaint();
                    }
                    while (keepPlaying) {
                        response = reader.readLine();
                        System.out.println("DEBUGOWANIE: --" + response + "--");
                        String[] parts = response.split(" ");
                        switch (parts[0]) {
                            case "KONIEC":
                                keepPlaying = false;
                            case "WYNIK":
                                gameField.setScore(2, parts[1]);
                                break;
                            case "ROZCZ":
                                disconnecting = true;
                                keepPlaying = false;
                                break;
                            default:
                                System.err.println("Nieoczekiwane polecenie gry: " + response + ". Ignorowanie.");
                        }
                        if (disconnecting) {
                            // We're disconnecting or they are. Acknowledge and quit.
                            writer.println("ROZCZ");
                            return;
                        } else {
                            // If we're not disconnecting, reply with our current score
                            if (gameField.trees.size() > 0) {
                                writer.print("WYNIK ");
                            } else {
                                keepPlaying = false;
                                writer.print("KONIEC ");
                            }
                            writer.println(gameField.getScore(1));
                        }
                    }
                    if (!disconnecting) {
                        // Check to see if they want to play again
                        response = reader.readLine();
                        if (response != null && response.equals("GRAMY_PONOWNIE")) {
                            // Do we want to play again?
                            String message = gameField.getWinner() + " Czy chesz zagra ponownie?";
                            int myPlayAgain = JOptionPane.showConfirmDialog(gameField, message, "Gramy ponownie?", JOptionPane.YES_NO_OPTION);
                            if (myPlayAgain == JOptionPane.YES_OPTION) {
                                writer.println("TAK");
                                startNewGame = true;
                            } else {
                                // Not playing again so disconnect.
                                disconnecting = true;
                                writer.println("ROZCZ");
                            }
                        }
                    }
                }
            }
            catch (IOException e ) {
                System.err.println("Bd sieciowy. zamykanie klienta.");
                e.printStackTrace();
            }
        }
    }
}
