<?php
#@ _OUTLINE_PART_1_
# score_entry.php - Skrypt pozwalający na wstawienie wyników w projekcie ocen uczniów.

require_once "sampdb_pdo.php";

# Zdefiniowanie stałych określających podejmowane działania.
define ("SHOW_INITIAL_PAGE", 0);
define ("SOLICIT_EVENT", 1);
define ("ADD_EVENT", 2);
define ("DISPLAY_SCORES", 3);
define ("ENTER_SCORES", 4);
#@ _OUTLINE_PART_1_

#@ _DISPLAY_CELL_
# Wyświetlenie komórki tabeli HTML. Zmienna $tag to nazwa znacznika ("th" lub "td"
# dla komórki nagłówka lub danych), $value to wartość do wyświetlenia, natomiast zmienna
# $encode powinna mieć wartość true lub false, wskazując tym samym, czy wartość HTML
# ma zostać zakodowana przed jej wyświetleniem. Zmienna $encode jest opcjonalna,
# domyślnie ma przypisaną wartość true.

function display_cell ($tag, $value, $encode = TRUE)
{
  if (strlen ($value) == 0) # Czy wartość jest pusta lub nieustawiona?
    $value = "&nbsp;";
  else if ($encode) # Przeprowadzenie kodowania HTML, jeśli wartością zmiennej jest true.
    $value = htmlspecialchars ($value);
  print ("<$tag>$value</$tag>\n");
}
#@ _DISPLAY_CELL_

# Wyświetlenie listy istniejących zdarzeń. Użytkownik może wybrać dowolne,
# a następnie dodać lub edytować wyniki tego zdarzenia. Wyświetlane jest także łącze
# "Utwórz nowe zdarzenie" pozwalające na utworzenie nowego zdarzenia.

#@ _DISPLAY_EVENTS_
function display_events ($dbh)
{
  print ("Wybierz zdarzenie klikając jego numer. Ewentualnie utwórz\n");
  print ("nowe klikając łącze Utwórz nowe zdarzenie:<br /><br />\n");
  print ("<table border=\"1\">\n");

  # Wyświetlenie wiersza tabeli zawierającego nagłówki.

  print ("<tr>\n");
  display_cell ("th", "Identyfikator zdarzenia");
  display_cell ("th", "Data");
  display_cell ("th", "Kategoria");
  print ("</tr>\n");

  # Wyświetlenie listy zdarzeń. Powiązanie każdego identyfikatora zdarzenia
  # z łączem wyświetlającym wyniki w danym zdarzeniu.

  $stmt = "SELECT event_id, date, category
           FROM grade_event ORDER BY event_id";
  $sth = $dbh->query ($stmt);

  while ($row = $sth->fetch ())
  {
    print ("<tr>\n");
    $url = sprintf ("%s?action=%d&event_id=%d",
                    script_name (),
                    DISPLAY_SCORES,
                    $row["event_id"]);
    display_cell ("td",
                  "<a href=\"$url\">"
                    . $row["event_id"]
                    . "</a>",
                  FALSE);
    display_cell ("td", $row["date"]);
    display_cell ("td", $row["category"]);
    print ("</tr>\n");
  }

  # Dodanie jeszcze jednego łącza, jego kliknięcie powoduje utworzenie nowego zdarzenia.

  print ("<tr align=\"center\">\n");
  $url = sprintf ("%s?action=%d",
                  script_name (),
                  SOLICIT_EVENT);
  display_cell ("td colspan=\"3\"",
                "<a href=\"$url\">Utwórz nowe zdarzenie</a>",
                FALSE);
  print ("</tr>\n");

  print ("</table>\n");
}
#@ _DISPLAY_EVENTS_

# Wyświetlenie informacji dla nowego zdarzenia.

#@ _SOLICIT_EVENT_INFO_
function solicit_event_info ()
{
  printf ("<form method=\"post\" action=\"%s?action=%d\">\n",
          script_name (),
          ADD_EVENT);
  print ("Podaj informacje o nowym zdarzeniu:<br /><br />\n");
  print ("Data: ");
  print ("<input type=\"text\" name=\"date\" value=\"\" size=\"10\" />\n");
  print ("<br />\n");
  print ("Kategoria: ");
  print ("<input type=\"radio\" name=\"category\" value=\"T\"");
  print (" checked=\"checked\" />Test\n");
  print ("<input type=\"radio\" name=\"category\" value=\"Q\" />Sprawdzian\n");
  print ("<br /><br />\n");
  print ("<input type=\"submit\" name=\"submit\" value=\"Wyślij\" />\n");
  print ("</form>\n");
}
#@ _SOLICIT_EVENT_INFO_

# Dodanie nowego zdarzenia do tabeli.

#@ _ADD_NEW_EVENT_
function add_new_event ($dbh)
{
  $date = script_param ("date");          # Pobranie wprowadzonych przez użytkownika
  $category = script_param ("category");  # daty i kategorii zdarzenia.

  if (empty ($date))  # Wprowadzona data powinna być w formacie ISO 8601.
    die ("Nie podano daty\n");
  if (!preg_match ('/^\d{4}\D\d{1,2}\D\d{1,2}$/', $date))
    die ("Proszę podać datę w formacie ISO 8601 (RRRR-MM-DD).\n");
  if ($category != "T" && $category != "Q")
    die ("Nieprawidłowa kategoria zdarzenia.\n");

  $stmt = "INSERT INTO grade_event (date,category) VALUES(?,?)";
  $sth = $dbh->prepare ($stmt);
  $sth->execute (array ($date, $category));
}
#@ _ADD_NEW_EVENT_

# Wyświetlenie wszystkich wyników dla danego zdarzenia, posortowanych według imion uczniów.
# Imiona są wyświetlane jako statyczny tekst, wyniki jako edytowalne pola.

#@ _DISPLAY_SCORES_
function display_scores ($dbh)
{
  # Pobranie identyfikatora zdarzenia, który musi być liczbą całkowitą.
  $event_id = script_param ("event_id");
  if (!ctype_digit ($event_id))
    die ("Nieprawidłowy identyfikator zdarzenia.\n");

  # Pobranie wyników dla danego zdarzenia.
  $stmt = "
    SELECT
      student.student_id, student.name, grade_event.date,
      score.score AS score, grade_event.category
    FROM student
      INNER JOIN grade_event
      LEFT JOIN score ON student.student_id = score.student_id
                      AND grade_event.event_id = score.event_id
    WHERE grade_event.event_id = ?
    ORDER BY student.name";
  $sth = $dbh->prepare ($stmt);
  $sth->execute (array ($event_id));

  # Pobranie rekordów i umieszczenie w tablicy, aby można było je policzyć.
  $rows = $sth->fetchAll ();
  if (count ($rows) == 0)
    die ("Nie znaleziono danych dla wskazanego zdarzenia.\n");

  printf ("<form method=\"post\" action=\"%s?action=%d&event_id=%d\">\n",
          script_name (),
          ENTER_SCORES,
          $event_id);

  # Wyświetlenie wyników w postaci tabeli HTML.

  for ($row_num = 0; $row_num < count ($rows); $row_num++)
  {
    $row = $rows[$row_num];
    # Wyświetlenie informacji o zdarzeniu i nagłówka tabeli przed pierwszym rekordem.
    if ($row_num == 0)
    {
      printf ("Identyfikator zdarzenia: %d, Data zdarzenia: %s, Kategoria zdarzenia: %s\n",
              $event_id,
              $row["date"],
              $row["category"]);
      print ("<br /><br />\n");
      print ("<table border=\"1\">\n");
      print ("<tr>\n");
      display_cell ("th", "Imię i nazwisko");
      display_cell ("th", "Wynik");
      print "</tr>\n";
    }
    print ("<tr>\n");
    display_cell ("td", $row["name"]);
    $col_val = sprintf ("<input type=\"text\" name=\"score[%d]\"",
                        $row["student_id"]);
    $col_val .= sprintf (" value=\"%d\" size=\"5\" /><br />\n",
                         $row["score"]);
    display_cell ("td", $col_val, FALSE);
    print ("</tr>\n");
  }

  print ("</table>\n");
  print ("<br />\n");
  print ("<input type=\"submit\" name=\"submit\" value=\"Wyślij\" />\n");
  print "</form>\n";
}
#@ _DISPLAY_SCORES_

#@ _ENTER_SCORES_
function enter_scores ($dbh)
{
  # Pobranie identyfikatora zdarzenia i tablicy wyników dla danego zdarzenia.

  $event_id = script_param ("event_id");
#@ _GET_SCORE_PARAM_
  $score = script_param ("score");
#@ _GET_SCORE_PARAM_

  if (!ctype_digit ($event_id)) # Identyfikator zdarzenia musi być liczbą całkowitą.
    die ("Nieprawidłowy identyfikator zdarzenia\n");

  # Przygotowanie zapytań, które później będą wielokrotnie wykonywane.
  $sth_del = $dbh->prepare ("DELETE FROM score
                             WHERE event_id = ? AND student_id = ?");
  $sth_repl = $dbh->prepare ("REPLACE INTO score
                              (event_id,student_id,score)
                              VALUES(?,?,?)");

  # Wyniki są wprowadzane w ramach transakcji.
  try
  {
    $dbh->beginTransaction ();

    $blank_count = 0;
    $nonblank_count = 0;
    foreach ($score as $student_id => $new_score)
    {
      $new_score = trim ($new_score);
      if (empty ($new_score))
      {
        # Jeżeli w formularzu nie znajduje się wynik dla danego ucznia, należy usunąć
        # ewentualny wynik tego ucznia, który mógł się dotychczas znajdować w bazie danych.
        ++$blank_count;
        $sth = $sth_del;
        $params = array ($event_id, $student_id);
      }
      else if (ctype_digit ($new_score)) # Identyfikator zdarzenia musi być liczbą całkowitą.
      {
        # Jeżeli wynik został podany, należy nim zastąpić wynik,
        # który mógł dotychczas znajdować się w bazie danych.
        ++$nonblank_count;
        $sth = $sth_repl;
        $params = array ($event_id, $student_id, $new_score);
      }
      else
      {
        throw new PDOException ("Nieprawidłowy wynik: $new_score");
      }
      $sth->execute ($params);
    }
    # Transakcja zakończyła się powodzeniem, trzeba ją zatwierdzić.
    $dbh->commit ();
    printf ("Liczba wprowadzonych wyników: %d<br />\n", $nonblank_count);
    printf ("Liczba brakujących wyników: %d<br />\n", $blank_count);
  }
  catch (PDOException $e)
  {
    printf ("Score entry failed: %s<br />\n",
            htmlspecialchars ($e->getMessage ()));
   # Wycofanie transakcji, używamy pustej procedury obsługi wyjątków,
   # aby przechwycić niepowodzenie w trakcie wycofywania.
   try
   {
     $dbh->rollback ();
   }
   catch (PDOException $e) { }
  }
  print ("<br />\n");
}
#@ _ENTER_SCORES_

#@ _OUTLINE_PART_2_
$title = "Projekt ocen uczniów -- wstawianie wyników";
html_begin ($title, $title);

$dbh = sampdb_connect ();

# Ustalenie akcji, która powinna zostać podjęta (domyślnie to
# wyświetlenie strony początkowej, jeśli nie zostanie podana żadna akcja).

#@ _GET_ACTION_PARAM_
$action = script_param ("action");
#@ _GET_ACTION_PARAM_
if (is_null ($action))
  $action = SHOW_INITIAL_PAGE;

switch ($action)
{
case SHOW_INITIAL_PAGE:   # Wyświetlenie strony początkowej.
  display_events ($dbh);
  break;
case SOLICIT_EVENT:       # Pobranie informacji o nowym zdarzeniu.
  solicit_event_info ();
  break;
case ADD_EVENT:           # Wstawienie nowego zdarzenia do bazy danych.
  add_new_event ($dbh);
  display_events ($dbh);
  break;
case DISPLAY_SCORES:      # Wyświetlenie wyników wskazanego zdarzenia.
  display_scores ($dbh);
  break;
case ENTER_SCORES:        # Wstawienie nowych lub edycja istniejących wyników.
  enter_scores ($dbh);
  display_events ($dbh);
  break;
default:
  die ("Nieznany kod ($action)\n");
}

$dbh = NULL;  # Zamknięcie połączenia.

html_end ();
#@ _OUTLINE_PART_2_
?>
