import java.awt.Color;

/**
 * Wielokąt foremny, tj. taki, którego wszystkie kąty mają taką samą miarę
 * i wszystkie boki mają taką samą długość. Figura ta ma również kolor.
 */
public class RegularPolygon extends DrawablePolygon {

    public static final String[] NAMES = {null, null, null,
            "Trójkąt", "Kwadrat", "Pięciokąt", "Sześciokąt",
            "Siedmiokąt", "Ośmiokąt", "Dziewięciokąt", "Dziesięciokąt"};

    /**
     * Konstruuje wielokąt foremny o określonej liczbie boków.
     * 
     * @param nsides liczba boków
     */
    public RegularPolygon(int nsides) {
        this(nsides, 50);
    }

    /**
     * Konstruuje wielokąt foremny o określonej liczbie boków i promieniu.
     * 
     * @param nsides liczba boków
     * @param radius odległość wierzchołka od środka
     */
    public RegularPolygon(int nsides, int radius) {
        this(nsides, radius, Color.BLACK);
    }

    /**
     * Konstruuje wielokąt foremny o określonej liczbie boków, promieniu
     * i kolorze wypełnienia.
     * 
     * @param nsides liczba boków
     * @param radius odległość wierzchołka od środka
     * @param color początkowy kolor wypełnienia
     */
    public RegularPolygon(int nsides, int radius, Color color) {

        // zwaliduj argumenty
        if (nsides < 3) {
            throw new IllegalArgumentException("niepoprawna wartość nsides");
        }
        if (radius <= 0) {
            throw new IllegalArgumentException("niepoprawna wartość radius");
        }
        if (color == null) {
            throw new NullPointerException("niepoprawna wartość color");
        }

        // zainicjalizuj atrybuty obiektu klasy Polygon
        this.npoints = nsides;
        this.xpoints = new int[nsides];
        this.ypoints = new int[nsides];
        this.color = color;

        // wyznacz kąt obrotu dla każdego z wierzchołków (w radianach)
        double angle = 2.0 * Math.PI / nsides;

        // określ rotację sprawiającą, że wielokąt rysowany jest od prawej strony w górę
        double rotate = Math.PI / nsides + Math.PI / 2.0;

        // oblicz współrzędne x i y, uwzględniając punkt początkowy jako środek
        for (int i = 0; i < nsides; i++) {
            double x = radius * Math.cos(i * angle + rotate);
            xpoints[i] = (int) Math.round(x);
            double y = radius * Math.sin(i * angle + rotate);
            ypoints[i] = (int) Math.round(y);
        }
    }

    @Override
    public String toString() {
        StringBuilder str = new StringBuilder();
        if (npoints < NAMES.length) {
            str.append(NAMES[npoints]);
        } else {
            str.append("Wielokąt");
        }
        // append the list of vertexes
        str.append('[');
        for (int i = 0; i < npoints; i++) {
            if (i > 0) {
                str.append(", ");
            }
            // dodaj następny punkt (x, y)
            str.append('(');
            str.append(xpoints[i]);
            str.append(", ");
            str.append(ypoints[i]);
            str.append(')');
        }
        str.append(']');
        return str.toString();
    }

}
