<?php
/**
 * Backports of useful constants and functions in later PHP versions.
 * (We assume we are running on at least PHP 4.1.2, since that is the
 * required version for Mambo 4.5.1.)
 *
 * Significant portions of this file are based on PEAR's PHP_Compat
 * package, version 1.3.1 (2004-11-23), by Aidan Lister and Stephen
 * Schmidt, and thus subject to the following copyright notice:
 * 
 * Copyright (c) 1997-2004 The PHP Group
 * This source file is subject to version 3.0 of the PHP license,
 * that is bundled with this package in the file LICENSE, and is
 * available at through the world-wide-web at
 * http://www.php.net/license/3_0.txt.
 * If you did not receive a copy of the PHP license and are unable to
 * obtain it through the world-wide-web, please send a note to
 * license@php.net so we can mail you a copy immediately.
 *
 * @author  Aidan Lister <aidan@php.net>
 * @author  Stephan Schmidt <schst@php.net>
 * @author  Thiemo Mttig (http://maettig.com/)
 * @author  David Irvine <dave@codexweb.co.za>
 * @author  Arto Bendiken <arto@xaneon.com>
 */

/** Ensure this file is being included by a parent file */
defined( '_VALID_MOS' ) or die( 'Direct access to this location is not allowed.' );

/**
 * Replace floatval()
 * @link      http://php.net/function.floatval
 * @since     PHP 4.2.0
 * @require   PHP 4.0.0 (type casting)
 */
if (!function_exists( 'floatval' )) {
  function floatval( $var ) {
    return (float)$var;
  }
}

/**
 * Replace function is_a()
 * @link      http://php.net/function.is_a
 * @since     PHP 4.2.0
 * @require   PHP 4.0.0 (is_subclass_of)
 */
if (!function_exists( 'is_a' )) {
  function is_a( $object, $class ) {
    if (!is_object( $object ))
      return false;
    if (get_class( $object ) == strtolower( $class )) {
      return true;
    }
    else {
      return is_subclass_of( $object, $class );
    }
  }
}

/**
 * Replace array_chunk()
 * @link      http://php.net/function.array_chunk
 * @since     PHP 4.2.0
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'array_chunk' )) {
  function array_chunk( $input, $size, $preserve_keys = false ) {
    if (!is_array( $input )) {
      trigger_error( 'array_chunk() expects parameter 1 to be array, ' .
        gettype( $input ) . ' given', E_USER_WARNING );
      return;
    }

    if (!is_numeric( $size )) {
      trigger_error( 'array_chunk() expects parameter 2 to be long, ' .
        gettype( $size ) . ' given', E_USER_WARNING );
      return;
    }

    $size = (int)$size;
    if ($size <= 0) {
      trigger_error( 'array_chunk() Size parameter expected to be greater than 0', E_USER_WARNING );
      return;
    }

    $chunks = array();
    $i = 0;

    if ($preserve_keys !== false) {
      foreach ($input as $key => $value) {
        $chunks[(int)($i++ / $size)][$key] = $value;
      }
    }
    else {
      foreach ($input as $value) {
        $chunks[(int)($i++ / $size)][] = $value;
      }
    }

    return $chunks;
  }
}

/**
 * Replace ob_clean()
 * @link      http://php.net/function.ob_clean
 * @since     PHP 4.2.0
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'ob_clean' )) {
  function ob_clean() {
    if (@ob_end_clean())
      return ob_start();
    trigger_error( 'ob_clean() failed to delete buffer. No buffer to delete.', E_USER_NOTICE );
    return false;
  }
}

/**
 * Replace ob_flush()
 * @link      http://php.net/function.ob_flush
 * @since     PHP 4.2.0
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'ob_flush' )) {
  function ob_flush() {
    if (@ob_end_flush())
      return ob_start();
    trigger_error( 'ob_flush() Failed to flush buffer. No buffer to flush.', E_USER_NOTICE );
    return false;
  }
}

/**
 * Replace ob_get_clean()
 * @link      http://php.net/function.ob_get_clean
 * @since     PHP 4.3.0
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'ob_get_clean' )) {
  function ob_get_clean() {
    $contents = ob_get_contents();
    if ($contents !== false)
      ob_end_clean();
    return $contents;
  }
}

/**
 * Replace ob_get_flush()
 * @link      http://php.net/function.ob_get_flush
 * @since     PHP 4.3.0
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'ob_get_flush' )) {
  function ob_get_flush() {
    $contents = ob_get_contents();
    if ($contents !== false)
      ob_end_flush();
    return $contents;
  }
}

/**
 * Replace get_include_path()
 * @link      http://php.net/function.get_include_path
 * @since     PHP 4.3.0
 */
if (!function_exists( 'get_include_path' )) {
  function get_include_path() {
    return ini_get( 'include_path' );
  }
}

/**
 * Replace set_include_path()
 * @link      http://php.net/function.set_include_path
 * @since     PHP 4.3.0
 */
if (!function_exists( 'set_include_path' )) {
  function set_include_path( $new_include_path ) {
    return ini_set( 'include_path', $new_include_path );
  }
}

/**
 * Replace restore_include_path()
 * @link      http://php.net/function.restore_include_path
 * @since     PHP 4.3.0
 */
if (!function_exists( 'restore_include_path' )) {
  function restore_include_path() {
    return ini_restore( 'include_path' );
  }
}

/**
 * Replace constant PATH_SEPARATOR.
 * @link      http://php.net/ref.dir
 * @since     PHP 4.3.0
 */
if (!defined( 'PATH_SEPARATOR' )) {
  define( 'PATH_SEPARATOR',
    (strtoupper( substr( PHP_OS, 0, 3 ) ) == 'WIN' ? ';' : ':') );
}

/**
 * Replace html_entity_decode()
 * @link      http://php.net/function.html_entity_decode
 * @since     PHP 4.3.0
 * @internal  Setting the charset will not do anything
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!defined( 'ENT_NOQUOTES' )) {
  define( 'ENT_NOQUOTES', 0 );
}
if (!defined( 'ENT_COMPAT' )) {
  define( 'ENT_COMPAT', 2 );
}
if (!defined( 'ENT_QUOTES' )) {
  define( 'ENT_QUOTES', 3 );
}
if (!function_exists( 'html_entity_decode' )) {
  function html_entity_decode( $string, $quote_style = ENT_COMPAT, $charset = null ) {
    if (!is_int( $quote_style )) {
      trigger_error( 'html_entity_decode() expects parameter 2 to be long, ' .
        gettype( $quote_style ) . ' given', E_USER_WARNING );
      return;
    }

    $trans_tbl = get_html_translation_table( HTML_ENTITIES );
    $trans_tbl = array_flip( $trans_tbl );

    // Add single quote to translation table;
    $trans_tbl['&#039;'] = '\'';

    // Not translating double quotes
    if ($quote_style & ENT_NOQUOTES) {
      // Remove double quote from translation table
      unset( $trans_tbl['&quot;'] );
    }

    return strtr( $string, $trans_tbl );
  }
}

/**
 * Replace file_get_contents()
 * @link      http://php.net/function.file_get_contents
 * @internal  resource_context is not supported
 * @since     PHP 4.3.0, PHP 5
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'file_get_contents' )) {
  function file_get_contents( $filename, $incpath = false, $resource_context = null ) {
    if (false === ($fh = fopen( $filename, 'rb', $incpath ))) {
      trigger_error( 'file_get_contents() failed to open stream: No such file or directory', E_USER_WARNING );
      return false;
    }

    clearstatcache();
    if ($fsize = @filesize( $filename )) {
      $data = fread( $fh, $fsize );
    }
    else {
      $data = '';
      while (!feof( $fh ))
        $data .= fread( $fh, 8192 );
    }

    fclose( $fh );
    return $data;
  }
}

/**
 * Replace PHP_EOL constant.
 * @link      http://php.net/reserved.constants.core
 * @since     PHP 4.3.10, PHP 5.0.2
 */
if (!defined( 'PHP_EOL' )) {
  switch (strtoupper( substr( PHP_OS, 0, 3 ) )) {
    case 'WIN': // Windows
      define( 'PHP_EOL', "\r\n" );
      break;
    case 'DAR': // Mac
    case 'MAC':
      define( 'PHP_EOL', "\r" );
      break;
    default:    // Unix
      define( 'PHP_EOL', "\n" );
  }
}

/**
 * Replace constant E_STRICT.
 * @link      http://php.net/ref.errorfunc
 * @since     PHP 5
 */
if (!defined( 'E_STRICT' )) {
  define( 'E_STRICT', 2048 );
}

/**
 * Replace array_combine()
 * @link      http://php.net/function.array_combine
 * @since     PHP 5
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'array_combine' )) {
  function array_combine( $keys, $values ) {
    if (!is_array( $keys )) {
      trigger_error( 'array_combine() expects parameter 1 to be array, ' .
        gettype( $keys ) . ' given', E_USER_WARNING );
      return;
    }

    if (!is_array( $values )) {
      trigger_error( 'array_combine() expects parameter 2 to be array, ' .
        gettype( $values ) . ' given', E_USER_WARNING );
      return;
    }

    if (count( $keys ) !== count( $values )) {
      trigger_error( 'array_combine() Both parameters should have equal number of elements', E_USER_WARNING );
      return false;
    }

    if (count( $keys ) === 0 || count( $values ) === 0) {
      trigger_error( 'array_combine() Both parameters should have number of elements at least 0', E_USER_WARNING );
      return false;
    }

    $keys = array_values( $keys );
    $values = array_values( $values );
    $combined = array();
    for ($i = 0, $count = count($values); $i < $count; $i++)
      $combined[$keys[$i]] = $values[$i];
    return $combined;
  }
}

/**
 * Replace file_put_contents()
 * @link      http://php.net/function.file_put_contents
 * @internal  resource_context is not supported
 * @since     PHP 5
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!defined( 'FILE_USE_INCLUDE_PATH' )) {
  define( 'FILE_USE_INCLUDE_PATH', 1 );
}
if (!defined( 'FILE_APPEND' )) {
  define( 'FILE_APPEND', 8 );
}
if (!function_exists( 'file_put_contents' )) {
  function file_put_contents( $filename, $content, $flags = null, $resource_context = null ) {

    // If $content is an array, convert it to a string
    if (is_array( $content ))
      $content = implode( '', $content );

    // If we don't have a string, throw an error
    if (!is_scalar( $content )) {
      $errormsg = 'file_put_contents() The 2nd parameter should be either a string or an array';
      trigger_error( $errormsg, E_USER_WARNING );
      return false;
    }

    // Get the length of date to write
    $length = strlen( $content );

    // Check what mode we are using and whether we're using the include path
    $mode = ($flags & FILE_APPEND) ? 'a' : 'w';
    $use_inc_path = ($flags & FILE_USE_INCLUDE_PATH) ? true : false;

    // Open the file for writing
    if (($fh = @fopen( $filename, $mode, $use_inc_path )) === false) {
      $errormsg = 'file_put_contents() failed to open stream: Permission denied';
      trigger_error( $errormsg, E_USER_WARNING );
      return false;
    }

    // Write to the file
    $bytes = 0;
    if (($bytes = @fwrite( $fh, $content )) === false) {
      $errormsg = sprintf( 'file_put_contents() Failed to write %d bytes to %s', $length, $filename );
      trigger_error( $errormsg, E_USER_WARNING );
      return false;
    }

    // Close the handle
    @fclose( $fh );

    // Check all the data was written
    if ($bytes != $length) {
      $errormsg = sprintf( 'file_put_contents() Only %d of %d bytes written, possibly out of free disk space.', $bytes, $length );
      trigger_error( $errormsg, E_USER_WARNING );
      return false;
    }

    // Return length written
    return $bytes;
  }
}

/**
 * Replace fprintf()
 * @link      http://php.net/function.fprintf
 * @since     PHP 5
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'fprintf' )) {
  function fprintf() {
    $args = func_get_args();
    if (count( $args ) < 2) {
      trigger_error( 'Wrong parameter count for fprintf()', E_USER_WARNING );
      return;
    }
    $resource_handle = array_shift( $args );
    $format = array_shift( $args );
    if (!is_resource( $resource_handle )) {
      trigger_error( 'fprintf(): supplied argument is not a valid stream resource', E_USER_WARNING );
      return false;
    }
    return fwrite( $resource_handle, vsprintf($format, $args ) );
  }
}

/**
 * Replace str_split()
 * @link      http://php.net/function.str_split
 * @since     PHP 5
 * @require   PHP 4.0.1 (trigger_error)
 */
if (!function_exists( 'str_split' )) {
  function str_split( $string, $split_length = 1 ) {
    if (!is_scalar( $split_length )) {
      trigger_error( 'str_split() expects parameter 2 to be long, ' .
        gettype( $split_length ) . ' given', E_USER_WARNING );
      return false;
    }
    $split_length = (int)$split_length;
    if ($split_length < 1) {
      trigger_error( 'str_split() The length of each segment must be greater than zero', E_USER_WARNING );
      return false;
    }
    preg_match_all( '/.{1,' . $split_length . '}/s', $string, $matches );
    return $matches[0];
  }
}

?>