/*
 * Copyright 2020 Wuyi Chen.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.kafkadefinitiveguide.consumer.assignpartitions;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;

/**
 * Przykład przypisywania consumenta do partycji zamiast subskrybowania tematu
 *
 * @author  Wuyi Chen
 * @date    06/05/2020
 * @version 1.0
 * @since   1.0
 */
public class AssignParitionsExample {
	public static void main(String[] args) {
		Properties props = new Properties();
		props.put("bootstrap.servers",  "localhost:9092");
		props.put("group.id",           "CountryCounter");                                                   // Określamy grupę consumentów, do których należy instacja klasy KafkaConsumer
		props.put("key.deserializer",   "org.apache.kafka.common.serialization.StringDeserializer");
		props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
		
		KafkaConsumer<String, String> consumer = new KafkaConsumer<>(props);
		
		List<PartitionInfo> partitionInfos = null;
		partitionInfos = consumer.partitionsFor("customerCountries");                                        // Pobieramy wszystkie dostępne partycje dla tego tematu

		List<TopicPartition> partitions = new ArrayList<>();
		if (partitionInfos != null) {   
		    for (PartitionInfo partition : partitionInfos) {
		        partitions.add(new TopicPartition(partition.topic(), partition.partition()));
		    }
		    consumer.assign(partitions);                                                                     // Przypisujemy konsumenta do określonych partycju zamiast subskrybować temat           

		    while (true) {
		        ConsumerRecords<String, String> records = consumer.poll(Duration.ofMillis(1000));

		        for (ConsumerRecord<String, String> record: records) {
		            System.out.printf("temat = %s, partycja = %s, przesunięcie = %d, klient = %s, kraj = %s%n",
		                record.topic(), record.partition(), record.offset(), record.key(), record.value());
		        }
		        consumer.commitSync();
		    }
		}
	}
}
